//
//  BarWidths.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/15.
//

#ifndef BarWidths_h
#define BarWidths_h

#import <Foundation/Foundation.h>

#import <UniPRT/ABarWidths.h>
#import <UniPRT/Ruler.h>
#import <UniPRT/PrintPlane.h>
#import <UniPRT/BarcodeTypeEnum_1D.h>

/**
 * \~English
 * @class BarWidths
 * @brief A class representing the widths of bars in barcodes for TSPL.
 *
 * This class inherits from ABarWidths and provides functionality for initializing bar widths and calculating barcode magnifications in TSPL.
 *
 * @ingroup TSPL
 *
 * \~Chinese
 * @class BarWidths
 * @brief 表示用于 TSPL 条码中条宽的类。
 *
 * 该类继承自 ABarWidths，并提供了在 TSPL 中初始化条宽和计算条码放大的功能。
 *
 * @ingroup TSPL
 *
 * \~Chinese-Traditional
 * @class BarWidths
 * @brief 表示用於 TSPL 條碼中條寬的類。
 *
 * 該類繼承自 ABarWidths，並提供了在 TSPL 中初始化條寬和計算條碼放大的功能。
 *
 * @ingroup TSPL
 */
@interface BarWidths : ABarWidths

/**
 * \~English
 * @brief Initializes a new instance of the BarWidths class.
 *
 * @return A new instance of BarWidths.
 *
 * \~Chinese
 * @brief 初始化 BarWidths 类的新实例。
 *
 * @return BarWidths 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 初始化 BarWidths 類的新實例。
 *
 * @return BarWidths 的新實例。
 */
- (instancetype)init;

/**
 * \~English
 * @brief Initializes a new instance of the BarWidths class with specified narrow and wide bar widths.
 *
 * @param narrowBar The width of the narrow bar.
 * @param wideBar The width of the wide bar.
 * @return A new instance of BarWidths.
 *
 * \~Chinese
 * @brief 使用指定的窄条和宽条宽度初始化 BarWidths 类的新实例。
 *
 * @param narrowBar 窄条的宽度。
 * @param wideBar 宽条的宽度。
 * @return BarWidths 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的窄條和寬條寬度初始化 BarWidths 類的新實例。
 *
 * @param narrowBar 窄條的寬度。
 * @param wideBar 寬條的寬度。
 * @return BarWidths 的新實例。
 */
- (instancetype)initWithNarrowBar:(float)narrowBar wideBar:(float)wideBar;

/**
 * \~English
 * @brief Calculates the magnification of a 1D barcode with the specified type, print plane, and bar widths.
 *
 * @param bcdType The type of the 1D barcode.
 * @param pp The print plane to be used.
 * @param barWidths The bar widths to be used for magnification calculation.
 * @return A string representing the magnification of the 1D barcode.
 *
 * \~Chinese
 * @brief 计算具有指定类型、打印平面和条宽的 1D 条码的放大倍数。
 *
 * @param bcdType 1D 条码的类型。
 * @param pp 要使用的打印平面。
 * @param barWidths 用于放大倍数计算的条宽。
 * @return 表示 1D 条码放大倍数的字符串。
 *
 * \~Chinese-Traditional
 * @brief 計算具有指定類型、打印平面和條寬的 1D 條碼的放大倍數。
 *
 * @param bcdType 1D 條碼的類型。
 * @param pp 要使用的打印平面。
 * @param barWidths 用於放大倍數計算的條寬。
 * @return 表示 1D 條碼放大倍數的字符串。
 */
- (NSString *)bcdMagnification1DWithType:(BarcodeTypeEnum1D)bcdType printPlane:(PrintPlane *)pp barWidths:(BarWidths *)barWidths;

@end

#endif /* BarWidths_h */
