//
//  BtComm.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/3/26.
//

#import <Foundation/Foundation.h>
#import <IOBluetooth/IOBluetooth.h>
#import <AutoIdSDK/AComm.h>

/**
 * @class BtComm
 * @brief A class for Bluetooth communication, inheriting from AComm.
 * @ingroup Comm
 */
@interface BtComm : AComm

/**
 * @brief The data received from the device.
 */
@property (nonatomic, strong) NSMutableData *receivedData;

/**
 * @brief Initializes a new instance with the given device address.
 *
 * @param deviceAddress The Bluetooth address of the device.
 * @return An initialized instance of BtComm, or nil if the object could not be created.
 */
- (instancetype)initWithDeviceAddress:(NSString *)deviceAddress;

/**
 * @brief Opens the connection to the Bluetooth device.
 */
- (void)open;

/**
 * @brief Closes the connection to the Bluetooth device.
 */
- (void)close;

/**
 * @brief Writes data to the Bluetooth device.
 *
 * @param text The data to write to the device.
 */
- (void)write:(NSData *)text;

/**
 * @brief Reads data from the Bluetooth device.
 *
 * @return The data read from the device.
 */
- (NSData *)read;

/**
 * @brief Writes data to the Bluetooth device and waits for a response.
 *
 * @param text The data to write to the device.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the device.
 */
- (NSData *)writeAndWaitForResponse:(NSData *)text
                 responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                   responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                      completionToken:(NSString *)completionToken;

@end

