//
//  UsbComm.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/3/12.
//

#import <Foundation/Foundation.h>
#import <IOKit/usb/IOUSBLib.h>
#import <AutoIdSDK/UsbInterfaceInfo.h>
#import <AutoIdSDK/AComm.h>



/**
 * @class UsbComm
 * @brief Class that implements the USB expecific communication and extends the AComm class.
 * @ingroup Comm
*/

@interface UsbComm: AComm

@property (nonatomic, strong, readonly) NSMutableArray<UsbInterfaceInfo *> *discoveredInterfaces;
@property (nonatomic, readonly) BOOL shouldStopReadLoop;
@property (readonly, nonatomic) NSMutableDictionary *deviceList;
@property (nonatomic, readonly) UsbInterfaceInfo *matchedInterfaces;

/**
 * @brief Initializes a new instance with the given vendor ID and product ID.
 *
 * @param vendorId The vendor ID for the instance.
 * @param productId The product ID for the instance.
 * @return An initialized instance of the class, or nil if the object could not be created.
 */
- (instancetype)initWithVendorId:(NSString *)vendorId productId:(NSString *)productId;
/**
 * @brief Opens the connection to the device.
 */
- (void)open;

/**
 * @brief Closes the connection to the device.
 */
- (void)close;

/**
 * @brief Writes data to the device.
 *
 * @param text The data to write to the device.
 */
- (void)write:(NSData *)text;

/**
 * @brief Reads data from the device.
 *
 * @return The data read from the device.
 */
- (NSData*)read;

/**
 * @brief Writes data to the device and waits for a response.
 *
 * @param text The data to write to the device.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the device.
 */
- (NSData*)writeAndWaitForResponse:(NSData *)text
              responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                   completionToken:(NSString *)completionToken;

/**
 * @brief Asynchronously listens for and reads data from the device.
 */
- (void)asyncListenRead;
@end

