/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *
 * @format
 */
"use strict";

function ownKeys(object, enumerableOnly) {
  var keys = Object.keys(object);

  if (Object.getOwnPropertySymbols) {
    var symbols = Object.getOwnPropertySymbols(object);
    if (enumerableOnly)
      symbols = symbols.filter(function(sym) {
        return Object.getOwnPropertyDescriptor(object, sym).enumerable;
      });
    keys.push.apply(keys, symbols);
  }

  return keys;
}

function _objectSpread(target) {
  for (var i = 1; i < arguments.length; i++) {
    var source = arguments[i] != null ? arguments[i] : {};

    if (i % 2) {
      ownKeys(Object(source), true).forEach(function(key) {
        _defineProperty(target, key, source[key]);
      });
    } else if (Object.getOwnPropertyDescriptors) {
      Object.defineProperties(target, Object.getOwnPropertyDescriptors(source));
    } else {
      ownKeys(Object(source)).forEach(function(key) {
        Object.defineProperty(
          target,
          key,
          Object.getOwnPropertyDescriptor(source, key)
        );
      });
    }
  }

  return target;
}

function _defineProperty(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
}

function _slicedToArray(arr, i) {
  return (
    _arrayWithHoles(arr) ||
    _iterableToArrayLimit(arr, i) ||
    _unsupportedIterableToArray(arr, i) ||
    _nonIterableRest()
  );
}

function _nonIterableRest() {
  throw new TypeError(
    "Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."
  );
}

function _unsupportedIterableToArray(o, minLen) {
  if (!o) return;
  if (typeof o === "string") return _arrayLikeToArray(o, minLen);
  var n = Object.prototype.toString.call(o).slice(8, -1);
  if (n === "Object" && o.constructor) n = o.constructor.name;
  if (n === "Map" || n === "Set") return Array.from(o);
  if (n === "Arguments" || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(n))
    return _arrayLikeToArray(o, minLen);
}

function _arrayLikeToArray(arr, len) {
  if (len == null || len > arr.length) len = arr.length;

  for (var i = 0, arr2 = new Array(len); i < len; i++) arr2[i] = arr[i];

  return arr2;
}

function _iterableToArrayLimit(arr, i) {
  if (typeof Symbol === "undefined" || !(Symbol.iterator in Object(arr)))
    return;
  var _arr = [];
  var _n = true;
  var _d = false;
  var _e = undefined;

  try {
    for (
      var _i = arr[Symbol.iterator](), _s;
      !(_n = (_s = _i.next()).done);
      _n = true
    ) {
      _arr.push(_s.value);

      if (i && _arr.length === i) break;
    }
  } catch (err) {
    _d = true;
    _e = err;
  } finally {
    try {
      if (!_n && _i["return"] != null) _i["return"]();
    } finally {
      if (_d) throw _e;
    }
  }

  return _arr;
}

function _arrayWithHoles(arr) {
  if (Array.isArray(arr)) return arr;
}

const AbstractConsumer = require("./AbstractConsumer");

const createConsumer = require("./createConsumer");

const _require = require("./constants"),
  FIRST_COLUMN = _require.FIRST_COLUMN,
  FIRST_LINE = _require.FIRST_LINE,
  EMPTY_POSITION = _require.EMPTY_POSITION;

const _require2 = require("./positionMath"),
  subtractOffsetFromPosition = _require2.subtractOffsetFromPosition;

const _require3 = require("./search"),
  greatestLowerBound = _require3.greatestLowerBound;

const _require4 = require("ob1"),
  add = _require4.add,
  get0 = _require4.get0,
  get1 = _require4.get1,
  add0 = _require4.add0,
  sub1 = _require4.sub1,
  sub = _require4.sub;
/**
 * A source map consumer that supports "indexed" source maps (that have a
 * `sections` field and no top-level mappings).
 */

class SectionsConsumer extends AbstractConsumer {
  constructor(sourceMap) {
    super(sourceMap);
    this._consumers = sourceMap.sections.map((section, index) => {
      const generatedOffset = {
        lines: add0(section.offset.line),
        columns: add0(section.offset.column)
      };
      const consumer = createConsumer(section.map);
      return [generatedOffset, consumer];
    });
  }

  originalPositionFor(generatedPosition) {
    const _ref = this._consumerForPosition(generatedPosition) || [],
      _ref2 = _slicedToArray(_ref, 2),
      generatedOffset = _ref2[0],
      consumer = _ref2[1];

    if (!consumer) {
      return EMPTY_POSITION;
    }

    return consumer.originalPositionFor(
      subtractOffsetFromPosition(generatedPosition, generatedOffset)
    );
  }

  *generatedMappings() {
    for (const _ref3 of this._consumers) {
      var _ref4 = _slicedToArray(_ref3, 2);

      const generatedOffset = _ref4[0];
      const consumer = _ref4[1];
      let first = true;

      for (const mapping of consumer.generatedMappings()) {
        if (
          first &&
          (get1(mapping.generatedLine) > 1 || get0(mapping.generatedColumn) > 0)
        ) {
          yield {
            generatedLine: FIRST_LINE,
            generatedColumn: FIRST_COLUMN,
            source: null,
            name: null,
            originalLine: null,
            originalColumn: null
          };
        }

        first = false;
        yield _objectSpread(
          _objectSpread({}, mapping),
          {},
          {
            generatedLine: add(mapping.generatedLine, generatedOffset.lines),
            generatedColumn: add(
              mapping.generatedColumn,
              generatedOffset.columns
            )
          }
        );
      }
    }
  }

  _consumerForPosition(generatedPosition) {
    const line = generatedPosition.line,
      column = generatedPosition.column;

    if (line == null || column == null) {
      return null;
    }

    const index = greatestLowerBound(
      this._consumers,
      generatedPosition,
      (position, _ref5) => {
        let _ref6 = _slicedToArray(_ref5, 1),
          offset = _ref6[0];

        const line0 = sub1(line);
        const column0 = column;

        if (line0 === offset.lines) {
          return get0(sub(column0, offset.columns));
        }

        return get0(sub(line0, offset.lines));
      }
    );
    return index != null ? this._consumers[index] : null;
  }

  sourceContentFor(source, nullOnMissing) {
    for (const _ref7 of this._consumers) {
      var _ref8 = _slicedToArray(_ref7, 2);

      const _ = _ref8[0];
      const consumer = _ref8[1];
      const content = consumer.sourceContentFor(source, nullOnMissing);

      if (content != null) {
        return content;
      }
    }

    return null;
  }
}

module.exports = SectionsConsumer;
