/*
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

#pragma once

#include <fbjni/fbjni.h>
#include <react/jni/JMessageQueueThread.h>
#include <react/jni/JRuntimeExecutor.h>
#include <react/jni/ReadableNativeMap.h>
#include <react/renderer/animations/LayoutAnimationDriver.h>
#include <react/renderer/scheduler/Scheduler.h>
#include <react/renderer/scheduler/SchedulerDelegate.h>
#include <react/renderer/uimanager/LayoutAnimationStatusDelegate.h>
#include <memory>
#include <mutex>
#include "ComponentFactory.h"
#include "EventBeatManager.h"
#include "JBackgroundExecutor.h"

namespace facebook {
namespace react {

class Instance;

struct CppMountItem final {
#pragma mark - Designated Initializers
  static CppMountItem CreateMountItem(ShadowView shadowView);
  static CppMountItem DeleteMountItem(ShadowView shadowView);
  static CppMountItem
  InsertMountItem(ShadowView parentView, ShadowView shadowView, int index);
  static CppMountItem
  RemoveMountItem(ShadowView parentView, ShadowView shadowView, int index);
  static CppMountItem UpdatePropsMountItem(ShadowView shadowView);
  static CppMountItem UpdateStateMountItem(ShadowView shadowView);
  static CppMountItem UpdateLayoutMountItem(ShadowView shadowView);
  static CppMountItem UpdateEventEmitterMountItem(ShadowView shadowView);
  static CppMountItem UpdatePaddingMountItem(ShadowView shadowView);

#pragma mark - Type

  enum Type {
    Undefined = -1,
    Multiple = 1,
    Create = 2,
    Delete = 4,
    Insert = 8,
    Remove = 16,
    UpdateProps = 32,
    UpdateState = 64,
    UpdateLayout = 128,
    UpdateEventEmitter = 256,
    UpdatePadding = 512
  };

#pragma mark - Fields

  Type type = {Create};
  ShadowView parentShadowView = {};
  ShadowView oldChildShadowView = {};
  ShadowView newChildShadowView = {};
  int index = {};
};

class Binding : public jni::HybridClass<Binding>,
                public SchedulerDelegate,
                public LayoutAnimationStatusDelegate {
 public:
  constexpr static const char *const kJavaDescriptor =
      "Lcom/facebook/react/fabric/Binding;";

  constexpr static auto UIManagerJavaDescriptor =
      "com/facebook/react/fabric/FabricUIManager";

  static void registerNatives();

 private:
  jni::global_ref<jobject> getJavaUIManager();
  std::shared_ptr<Scheduler> getScheduler();

  void setConstraints(
      jint surfaceId,
      jfloat minWidth,
      jfloat maxWidth,
      jfloat minHeight,
      jfloat maxHeight,
      jfloat offsetX,
      jfloat offsetY,
      jboolean isRTL,
      jboolean doLeftAndRightSwapInRTL);

  static jni::local_ref<jhybriddata> initHybrid(jni::alias_ref<jclass>);

  void installFabricUIManager(
      jni::alias_ref<JRuntimeExecutor::javaobject> runtimeExecutorHolder,
      jni::alias_ref<jobject> javaUIManager,
      EventBeatManager *eventBeatManager,
      jni::alias_ref<JavaMessageQueueThread::javaobject> jsMessageQueueThread,
      ComponentFactory *componentsRegistry,
      jni::alias_ref<jobject> reactNativeConfig);

  void startSurface(
      jint surfaceId,
      jni::alias_ref<jstring> moduleName,
      NativeMap *initialProps);

  void startSurfaceWithConstraints(
      jint surfaceId,
      jni::alias_ref<jstring> moduleName,
      NativeMap *initialProps,
      jfloat minWidth,
      jfloat maxWidth,
      jfloat minHeight,
      jfloat maxHeight,
      jfloat offsetX,
      jfloat offsetY,
      jboolean isRTL,
      jboolean doLeftAndRightSwapInRTL);

  void renderTemplateToSurface(jint surfaceId, jstring uiTemplate);

  void stopSurface(jint surfaceId);

  void schedulerDidFinishTransaction(
      MountingCoordinator::Shared const &mountingCoordinator) override;

  void schedulerDidRequestPreliminaryViewAllocation(
      const SurfaceId surfaceId,
      const ShadowView &shadowView) override;

  void schedulerDidDispatchCommand(
      const ShadowView &shadowView,
      std::string const &commandName,
      folly::dynamic const args) override;

  void schedulerDidSetJSResponder(
      SurfaceId surfaceId,
      const ShadowView &shadowView,
      const ShadowView &initialShadowView,
      bool blockNativeResponder) override;

  void schedulerDidClearJSResponder() override;

  void setPixelDensity(float pointScaleFactor);

  void driveCxxAnimations();

  void uninstallFabricUIManager();

  // Private member variables
  jni::global_ref<jobject> javaUIManager_;
  std::mutex javaUIManagerMutex_;

  // LayoutAnimations
  virtual void onAnimationStarted() override;
  virtual void onAllAnimationsComplete() override;
  LayoutAnimationDriver *getAnimationDriver();
  std::shared_ptr<LayoutAnimationDriver> animationDriver_;
  std::unique_ptr<JBackgroundExecutor> backgroundExecutor_;

  std::shared_ptr<Scheduler> scheduler_;
  std::mutex schedulerMutex_;

  std::recursive_mutex commitMutex_;

  float pointScaleFactor_ = 1;

  std::shared_ptr<const ReactNativeConfig> reactNativeConfig_{nullptr};
  bool useIntBufferBatchMountItem_{false};
  bool collapseDeleteCreateMountingInstructions_{false};
  bool disablePreallocateViews_{false};
  bool disableVirtualNodePreallocation_{false};
  bool enableFabricLogs_{false};

 private:
  void schedulerDidFinishTransactionIntBuffer(
      MountingCoordinator::Shared const &mountingCoordinator);
};

} // namespace react
} // namespace facebook
