/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ScopeManager = void 0;

var _scope = require("./scope");

var _ClassFieldInitializerScope = require("./scope/ClassFieldInitializerScope");

var _ClassStaticBlockScope = require("./scope/ClassStaticBlockScope");

var _variable = require("./variable");

var _ScopeType = require("./scope/ScopeType");

class ScopeManager {
  /**
   * The root scope
   * @public
   */

  /**
   * All scopes
   * @public
   */
  // $FlowExpectedError[unsafe-getters-setters]
  get variables() {
    const variables = new Set();

    function recurse(scope) {
      scope.variables.forEach(v => variables.add(v));
      scope.childScopes.forEach(recurse);
    }

    this.scopes.forEach(recurse);
    return Array.from(variables).sort((a, b) => a.$id - b.$id);
  }

  constructor(options) {
    this.currentScope = null;
    this.declaredVariables = new WeakMap();
    this.globalScope = void 0;
    this.nodeToScope = new WeakMap();
    this._options = void 0;
    this.scopes = [];
    this._options = options;
  }

  isGlobalReturn() {
    return this._options.globalReturn === true;
  }

  isModule() {
    return this._options.sourceType === 'module';
  }

  isImpliedStrict() {
    return this.isModule();
  }

  isStrictModeSupported() {
    return true;
  }

  isES6() {
    return true;
  }
  /**
   * Get the variables that a given AST node defines. The gotten variables' `def[].node`/`def[].parent` property is the node.
   * If the node does not define any variable, this returns an empty array.
   * @param node An AST node to get their variables.
   * @public
   */


  getDeclaredVariables(node) {
    var _this$declaredVariabl;

    return (_this$declaredVariabl = this.declaredVariables.get(node)) != null ? _this$declaredVariabl : [];
  }
  /**
   * Get the scope of a given AST node. The gotten scope's `block` property is the node.
   * This method never returns `function-expression-name` scope. If the node does not have their scope, this returns `null`.
   *
   * @param node An AST node to get their scope.
   * @param inner If the node has multiple scopes, this returns the outermost scope normally.
   *                If `inner` is `true` then this returns the innermost scope.
   * @public
   */


  acquire(node, inner = false) {
    function predicate(testScope) {
      if (testScope.type === 'function' && testScope.functionExpressionScope) {
        return false;
      }

      return true;
    }

    const scopes = this.nodeToScope.get(node);

    if (!scopes || scopes.length === 0) {
      return null;
    } // Heuristic selection from all scopes.
    // If you would like to get all scopes, please use ScopeManager#acquireAll.


    if (scopes.length === 1) {
      return scopes[0];
    }

    if (inner) {
      for (let i = scopes.length - 1; i >= 0; --i) {
        const scope = scopes[i];

        if (predicate(scope)) {
          return scope;
        }
      }
    } else {
      for (let i = 0; i < scopes.length; ++i) {
        const scope = scopes[i];

        if (predicate(scope)) {
          return scope;
        }
      }
    }

    return null;
  }

  _assertCurrentScope() {
    if (this.currentScope == null) {
      throw new Error('currentScope was unexpectedly null.');
    } // $FlowExpectedError[incompatible-return]


    return this.currentScope;
  }

  _nestScope(scope) {
    if (scope instanceof _scope.GlobalScope) {
      this.globalScope = scope;
    }

    this.currentScope = scope;
    return scope;
  }

  nestBlockScope(node) {
    return this._nestScope(new _scope.BlockScope(this, this._assertCurrentScope(), node));
  }

  nestCatchScope(node) {
    return this._nestScope(new _scope.CatchScope(this, this._assertCurrentScope(), node));
  }

  nestClassScope(node) {
    return this._nestScope(new _scope.ClassScope(this, this._assertCurrentScope(), node));
  }

  nestClassFieldInitializerScope(node) {
    const currentScope = this._assertCurrentScope();

    if (currentScope.type !== _ScopeType.ScopeType.Class) {
      throw new Error('Expected current scope to be a class scope.');
    }

    return this._nestScope(new _ClassFieldInitializerScope.ClassFieldInitializerScope(this, currentScope, node));
  }

  nestClassStaticBlockScope(node) {
    const currentScope = this._assertCurrentScope();

    if (currentScope.type !== _ScopeType.ScopeType.Class) {
      throw new Error('Expected current scope to be a class scope.');
    }

    return this._nestScope(new _ClassStaticBlockScope.ClassStaticBlockScope(this, currentScope, node));
  }

  nestDeclareModuleScope(node) {
    return this._nestScope(new _scope.DeclareModuleScope(this, this._assertCurrentScope(), node));
  }

  nestDeclareNamespaceScope(node) {
    return this._nestScope(new _scope.DeclareNamespaceScope(this, this._assertCurrentScope(), node));
  }

  nestForScope(node) {
    return this._nestScope(new _scope.ForScope(this, this._assertCurrentScope(), node));
  }

  nestFunctionExpressionNameScope(node) {
    return this._nestScope(new _scope.FunctionExpressionNameScope(this, this._assertCurrentScope(), node));
  }

  nestComponentScope(node) {
    return this._nestScope(new _scope.ComponentScope(this, this._assertCurrentScope(), node));
  }

  nestHookScope(node) {
    return this._nestScope(new _scope.HookScope(this, this._assertCurrentScope(), node));
  }

  nestFunctionScope(node, isMethodDefinition) {
    return this._nestScope(new _scope.FunctionScope(this, this._assertCurrentScope(), node, isMethodDefinition));
  }

  nestGlobalScope(node) {
    return this._nestScope(new _scope.GlobalScope(this, node));
  }

  nestModuleScope(node) {
    return this._nestScope(new _scope.ModuleScope(this, this._assertCurrentScope(), node));
  }

  nestSwitchScope(node) {
    return this._nestScope(new _scope.SwitchScope(this, this._assertCurrentScope(), node));
  }

  nestTypeScope(node) {
    return this._nestScope(new _scope.TypeScope(this, this._assertCurrentScope(), node));
  }

  nestWithScope(node) {
    return this._nestScope(new _scope.WithScope(this, this._assertCurrentScope(), node));
  }

}

exports.ScopeManager = ScopeManager;