/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.analyze = analyze;

var _hermesParser = require("hermes-parser");

var _referencer = require("./referencer");

var _ScopeManager = require("./ScopeManager");

const DEFAULT_OPTIONS = {
  globalReturn: false,
  jsxPragma: 'React',
  jsxFragmentName: null,
  sourceType: 'module',
  fbt: true
};
const JS_IDENTIFIER_REGEX = /^[_$a-zA-Z][_$a-zA-Z0-9]*$/;

function extractIdentifier(directive) {
  // handle `@\jsx Foo.bar` -> we want to extract `Foo`, not `Foo.bar`
  const foundPragma = directive == null ? void 0 : directive[0].split('.')[0];

  if (foundPragma != null && JS_IDENTIFIER_REGEX.test(foundPragma)) {
    return foundPragma;
  }

  return null;
}

function getJsxPragma(ast, providedOptions) {
  var _providedOptions$jsxP;

  // search for a pragma comment in the docblock only
  // we do this for speed and simplicity
  if (ast.docblock) {
    const foundPragma = extractIdentifier(ast.docblock.directives.jsx);

    if (foundPragma != null) {
      return foundPragma;
    }
  }

  if (providedOptions && // intentionally differentiate between null and undefined
  // -- null      = don't reference JSX pragma
  // -- undefined = not set, use default
  // eslint-disable-next-line eqeqeq
  providedOptions.jsxPragma === null) {
    return null;
  }

  return (_providedOptions$jsxP = providedOptions == null ? void 0 : providedOptions.jsxPragma) != null ? _providedOptions$jsxP : DEFAULT_OPTIONS.jsxPragma;
}

function getJsxFragmentPragma(ast, providedOptions) {
  var _providedOptions$jsxF;

  // search for a pragma comment in the docblock only
  // we do this for speed and simplicity
  if (ast.docblock) {
    const foundPragma = extractIdentifier(ast.docblock.directives.jsxFrag);

    if (foundPragma != null) {
      return foundPragma;
    }
  }

  return (_providedOptions$jsxF = providedOptions == null ? void 0 : providedOptions.jsxFragmentName) != null ? _providedOptions$jsxF : DEFAULT_OPTIONS.jsxFragmentName;
}
/**
 * Takes an AST and returns the analyzed scopes.
 */


function analyze(ast, providedOptions) {
  var _providedOptions$glob, _providedOptions$sour, _providedOptions$fbt;

  const scopeManager = new _ScopeManager.ScopeManager({
    globalReturn: (_providedOptions$glob = providedOptions == null ? void 0 : providedOptions.globalReturn) != null ? _providedOptions$glob : DEFAULT_OPTIONS.globalReturn,
    sourceType: (_providedOptions$sour = providedOptions == null ? void 0 : providedOptions.sourceType) != null ? _providedOptions$sour : DEFAULT_OPTIONS.sourceType
  });
  const referencer = new _referencer.Referencer({
    childVisitorKeys: _hermesParser.FlowVisitorKeys,
    fbtSupport: (_providedOptions$fbt = providedOptions == null ? void 0 : providedOptions.fbt) != null ? _providedOptions$fbt : DEFAULT_OPTIONS.fbt,
    jsxPragma: getJsxPragma(ast, providedOptions),
    jsxFragmentName: getJsxFragmentPragma(ast, providedOptions)
  }, scopeManager);
  referencer.visit(ast);
  return scopeManager;
}