/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ClassVisitor = void 0;

var _definition = require("../definition");

var _TypeVisitor = require("./TypeVisitor");

var _Visitor = require("./Visitor");

class ClassVisitor extends _Visitor.Visitor {
  constructor(referencer, _node) {
    super(referencer);
    this._classNode = void 0;
    this._referencer = void 0;

    this.visitType = node => {
      if (!node) {
        return;
      }

      _TypeVisitor.TypeVisitor.visit(this._referencer, node);
    };

    this._referencer = referencer;
    this._classNode = _node;
  }

  static visit(referencer, node) {
    const classVisitor = new ClassVisitor(referencer, node);
    classVisitor.visitClass(node);
  }

  visit(node) {
    // make sure we only handle the nodes we are designed to handle
    if (node && node.type in this) {
      super.visit(node);
    } else {
      this._referencer.visit(node);
    }
  } ///////////////////
  // Visit helpers //
  ///////////////////


  visitClass(node) {
    var _node$decorators, _node$implements;

    if (node.type === 'ClassDeclaration' && node.id) {
      const id = node.id;

      this._referencer.currentScope().defineIdentifier(id, new _definition.ClassNameDefinition(id, node));
    }

    (_node$decorators = node.decorators) == null ? void 0 : _node$decorators.forEach(d => this._referencer.visit(d));

    this._referencer.scopeManager.nestClassScope(node);

    if (node.id) {
      const id = node.id; // define the class name again inside the new scope
      // references to the class should not resolve directly to the parent class

      this._referencer.currentScope().defineIdentifier(id, new _definition.ClassNameDefinition(id, node));
    }

    this._referencer.visit(node.superClass); // visit the type param declarations


    this.visitType(node.typeParameters); // then the usages

    this.visitType(node.superTypeParameters);
    (_node$implements = node.implements) == null ? void 0 : _node$implements.forEach(imp => this.visitType(imp));
    this.visit(node.body);

    this._referencer.close(node);
  }

  visitPropertyDefinition(node) {
    this.visitProperty(node);
    this.visitType(node.typeAnnotation);
  }

  visitProperty(node) {
    if (node.computed) {
      this._referencer.visit(node.key);
    }

    if (node.value) {
      const value = node.value;

      if (node.type === 'PropertyDefinition') {
        this._referencer.scopeManager.nestClassFieldInitializerScope(value);
      }

      this._referencer.visit(value);

      if (node.type === 'PropertyDefinition') {
        this._referencer.close(value);
      }
    }
  }

  visitMethod(node) {
    if (node.computed) {
      this._referencer.visit(node.key);
    }

    this._referencer.visitFunction(node.value);
  }

  /////////////////////
  // Visit selectors //
  /////////////////////
  ClassBody(node) {
    // this is here on purpose so that this visitor explicitly declares visitors
    // for all nodes it cares about (see the instance visit method above)
    this.visitChildren(node);
  }

  PropertyDefinition(node) {
    this.visitPropertyDefinition(node);
  }

  MethodDefinition(node) {
    this.visitMethod(node);
  }

  Identifier(node) {
    this._referencer.visit(node);
  }

  PrivateIdentifier() {// intentionally skip
  }

}

exports.ClassVisitor = ClassVisitor;