/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Referencer = void 0;

var _ClassVisitor = require("./ClassVisitor");

var _ExportVisitor = require("./ExportVisitor");

var _ImportVisitor = require("./ImportVisitor");

var _PatternVisitor = require("./PatternVisitor");

var _Reference = require("./Reference");

var _ScopeManager = require("../ScopeManager");

var _TypeVisitor = require("./TypeVisitor");

var _Visitor = require("./Visitor");

var _definition = require("../definition");

function getJsxName(name) {
  switch (name.type) {
    case 'JSXIdentifier':
      return name.name;

    case 'JSXNamespacedName':
      return getJsxName(name.namespace);

    case 'JSXMemberExpression':
      return getJsxName(name.object);

    default:
      throw new Error(`Unexpected JSX tag name ${name.type}`);
  }
}

const FBT_NAMES = new Set(['fbt', 'fbs']);

// Referencing variables and creating bindings.
class Referencer extends _Visitor.Visitor {
  constructor({
    childVisitorKeys,
    fbtSupport,
    jsxFragmentName,
    jsxPragma
  }, scopeManager) {
    super({
      childVisitorKeys
    });
    this._jsxPragma = void 0;
    this._jsxFragmentName = void 0;
    this._fbtSupport = void 0;
    this.scopeManager = void 0;

    this.currentScope = dontThrowOnNull => {
      if (dontThrowOnNull !== true) {
        if (this.scopeManager.currentScope == null) {
          throw new Error('Expected there to be a current scope.');
        }
      } // $FlowExpectedError[incompatible-type]


      return this.scopeManager.currentScope;
    };

    this.visitType = node => {
      if (!node) {
        return;
      }

      _TypeVisitor.TypeVisitor.visit(this, node);
    };

    this.scopeManager = scopeManager;
    this._jsxPragma = jsxPragma;
    this._jsxFragmentName = jsxFragmentName;
    this._fbtSupport = fbtSupport;
  }

  close(node) {
    while (this.currentScope(true) && node === this.currentScope().block) {
      this.scopeManager.currentScope = this.currentScope().close(this.scopeManager);
    }
  }

  referencingDefaultValue(pattern, assignments, maybeImplicitGlobal, init) {
    assignments.forEach(assignment => {
      this.currentScope().referenceValue(pattern, _Reference.ReferenceFlag.Write, assignment.right, maybeImplicitGlobal, init);
    });
  }

  _referenceJsxPragma() {
    const jsxPragma = this._jsxPragma;

    if (jsxPragma == null) {
      return;
    }

    this.currentScope().indirectlyReferenceValue(jsxPragma);
  }

  _referenceJsxFragment() {
    const jsxFragmentName = this._jsxFragmentName;

    if (jsxFragmentName == null) {
      return;
    }

    this.currentScope().indirectlyReferenceValue(jsxFragmentName);
  } ///////////////////
  // Visit helpers //
  ///////////////////


  visitClass(node) {
    _ClassVisitor.ClassVisitor.visit(this, node);
  }

  visitForIn(node) {
    if (node.left.type === 'VariableDeclaration' && node.left.kind !== 'var') {
      this.scopeManager.nestForScope(node);
    }

    if (node.left.type === 'VariableDeclaration') {
      const left = node.left;
      this.visit(left);
      this.visitPattern(left.declarations[0].id, pattern => {
        this.currentScope().referenceValue(pattern, _Reference.ReferenceFlag.Write, node.right, null, true);
      }, typeAnnotation => {
        this.visitType(typeAnnotation);
      });
    } else {
      this.visitPattern(node.left, (pattern, info) => {
        const maybeImplicitGlobal = !this.currentScope().isStrict ? {
          pattern,
          node
        } : null;
        this.referencingDefaultValue(pattern, info.assignments, maybeImplicitGlobal, false);
        this.currentScope().referenceValue(pattern, _Reference.ReferenceFlag.Write, node.right, maybeImplicitGlobal, false);
      }, typeAnnotation => {
        this.visitType(typeAnnotation);
      }, {
        processRightHandNodes: true
      });
    }

    this.visit(node.right);
    this.visit(node.body);
    this.close(node);
  }

  visitFunction(node) {
    // FunctionDeclaration name is defined in upper scope
    // NOTE: Not referring variableScope. It is intended.
    // Since
    //  in ES5, FunctionDeclaration should be in FunctionBody.
    //  in ES6, FunctionDeclaration should be block scoped.
    switch (node.type) {
      case 'FunctionExpression':
        {
          if (node.id) {
            // FunctionExpression with name creates its special scope;
            // FunctionExpressionNameScope.
            this.scopeManager.nestFunctionExpressionNameScope(node);
          }

          break;
        }

      case 'FunctionDeclaration':
        {
          if (node.id != null) {
            const id = node.id; // id is defined in upper scope

            this.currentScope().defineIdentifier(id, new _definition.FunctionNameDefinition(id, node));
          }

          break;
        }

      case 'ArrowFunctionExpression':
        {
          break;
        }
    }

    this.scopeManager.nestFunctionScope(node, false); // function type parameters can be referenced by function params, so have to be declared first

    this.visitType(node.typeParameters); // Return type may reference type parameters but not function parameters, so visit it before the parameters

    this.visitType(node.returnType); // Process parameter declarations.

    for (const param of node.params) {
      if (param.type === 'Identifier' && param.name === 'this') {
        // `this` parameters don't declare variables, nor can they have default values
        // but will have an annotation
        this.visitType(param.typeAnnotation);
        continue;
      }

      this.visitPattern(param, (pattern, info) => {
        this.currentScope().defineIdentifier(pattern, new _definition.ParameterDefinition(pattern, node, info.rest));
        this.referencingDefaultValue(pattern, info.assignments, null, true);
      }, typeAnnotation => {
        this.visitType(typeAnnotation);
      }, {
        processRightHandNodes: true
      });
    } // In TypeScript there are a number of function-like constructs which have no body,
    // so check it exists before traversing


    if (node.body) {
      // Skip BlockStatement to prevent creating BlockStatement scope.
      if (node.body.type === 'BlockStatement') {
        this.visitChildren(node.body);
      } else {
        this.visit(node.body);
      }
    }

    this.close(node);
  }

  visitJSXTag(node) {
    const rootName = getJsxName(node.name);

    if (this._fbtSupport !== true || !FBT_NAMES.has(rootName)) {
      // <fbt /> does not reference the jsxPragma, but instead references the fbt import
      this._referenceJsxPragma();
    }

    switch (node.name.type) {
      case 'JSXIdentifier':
        if (rootName[0].toUpperCase() === rootName[0] || this._fbtSupport === true && FBT_NAMES.has(rootName)) {
          // lower cased component names are always treated as "intrinsic" names, and are converted to a string,
          // not a variable by JSX transforms:
          // <div /> => React.createElement("div", null)
          this.visit(node.name);
        }

        break;

      case 'JSXMemberExpression':
      case 'JSXNamespacedName':
        // special case for <this.Foo /> - we don't want to create an unclosed
        // and impossible-to-resolve reference to a variable called `this`.
        if (rootName !== 'this') {
          this.visit(node.name);
        }

        break;
    }
  } /////////////////////
  // Visit selectors //
  /////////////////////


  ArrowFunctionExpression(node) {
    this.visitFunction(node);
  }

  AsExpression(node) {
    this.visit(node.expression);
    this.visitType(node.typeAnnotation);
  }

  AssignmentExpression(node) {
    const left = node.left;

    if (_PatternVisitor.PatternVisitor.isPattern(left)) {
      if (node.operator === '=') {
        this.visitPattern(left, (pattern, info) => {
          const maybeImplicitGlobal = !this.currentScope().isStrict ? {
            pattern,
            node
          } : null;
          this.referencingDefaultValue(pattern, info.assignments, maybeImplicitGlobal, false);
          this.currentScope().referenceValue(pattern, _Reference.ReferenceFlag.Write, node.right, maybeImplicitGlobal, false);
        }, () => {}, {
          processRightHandNodes: true
        });
      } else if (left.type === 'Identifier') {
        this.currentScope().referenceValue(left, _Reference.ReferenceFlag.ReadWrite, node.right);
      }
    } else {
      this.visit(left);
    }

    this.visit(node.right);
  }

  BlockStatement(node) {
    if (this.scopeManager.isES6()) {
      this.scopeManager.nestBlockScope(node);
    }

    this.visitChildren(node);
    this.close(node);
  }

  BreakStatement(_) {// don't reference the break statement's label
  }

  CallExpression(node) {
    this.visitChildren(node, ['typeArguments']);
    this.visitType(node.typeArguments);
  }

  CatchClause(node) {
    this.scopeManager.nestCatchScope(node);

    if (node.param) {
      const param = node.param;
      this.visitPattern(param, (pattern, info) => {
        this.currentScope().defineIdentifier(pattern, new _definition.CatchClauseDefinition(param, node));
        this.referencingDefaultValue(pattern, info.assignments, null, true);
      }, typeAnnotation => {
        this.visitType(typeAnnotation);
      }, {
        processRightHandNodes: true
      });
    }

    this.visit(node.body);
    this.close(node);
  }

  ClassExpression(node) {
    this.visitClass(node);
  }

  ClassDeclaration(node) {
    this.visitClass(node);
  }

  ContinueStatement(_) {// don't reference the continue statement's label
  }

  EnumDeclaration(node) {
    this.currentScope().defineIdentifier(node.id, new _definition.EnumDefinition(node.id, node)); // Enum body cannot contain identifier references, so no need to visit body.
  }

  ExportAllDeclaration(_) {// this defines no local variables
  }

  ExportDefaultDeclaration(node) {
    if (node.declaration.type === 'Identifier') {
      _ExportVisitor.ExportVisitor.visit(this, node);
    } else {
      this.visit(node.declaration);
    }
  }

  ExportNamedDeclaration(node) {
    if (node.declaration) {
      this.visit(node.declaration);
    } else {
      _ExportVisitor.ExportVisitor.visit(this, node);
    }
  }

  ForInStatement(node) {
    this.visitForIn(node);
  }

  ForOfStatement(node) {
    this.visitForIn(node);
  }

  ForStatement(node) {
    // Create ForStatement declaration.
    // NOTE: In ES6, ForStatement dynamically generates per iteration environment. However, this is
    // a static analyzer, we only generate one scope for ForStatement.
    if (node.init && node.init.type === 'VariableDeclaration' && node.init.kind !== 'var') {
      this.scopeManager.nestForScope(node);
    }

    this.visitChildren(node);
    this.close(node);
  }

  ComponentDeclaration(node) {
    const id = node.id; // id is defined in upper scope

    this.currentScope().defineIdentifier(id, new _definition.ComponentNameDefinition(id, node));
    this.scopeManager.nestComponentScope(node); // component type parameters can be referenced by component params, so have to be declared first

    this.visitType(node.typeParameters); // Renders type may reference type parameters but not component parameters, so visit it before the parameters

    this.visitType(node.rendersType); // Process parameter declarations.

    for (const param of node.params) {
      const paramPattern = (() => {
        switch (param.type) {
          case 'ComponentParameter':
            return param.local;

          case 'RestElement':
            return param;
        }
      })();

      this.visitPattern(paramPattern, (pattern, info) => {
        this.currentScope().defineIdentifier(pattern, new _definition.ParameterDefinition(pattern, node, info.rest));
        this.referencingDefaultValue(pattern, info.assignments, null, true);
      }, typeAnnotation => {
        this.visitType(typeAnnotation);
      }, {
        processRightHandNodes: true
      });
    }

    this.visitChildren(node.body);
    this.close(node);
  }

  HookDeclaration(node) {
    const id = node.id; // id is defined in upper scope

    this.currentScope().defineIdentifier(id, new _definition.HookNameDefinition(id, node));
    this.scopeManager.nestHookScope(node); // hook type parameters can be referenced by hook params, so have to be declared first

    this.visitType(node.typeParameters); // Return type may reference type parameters but not hook parameters, so visit it before the parameters

    this.visitType(node.returnType); // Process parameter declarations.

    for (const param of node.params) {
      this.visitPattern(param, (pattern, info) => {
        this.currentScope().defineIdentifier(pattern, new _definition.ParameterDefinition(pattern, node, info.rest));
        this.referencingDefaultValue(pattern, info.assignments, null, true);
      }, typeAnnotation => {
        this.visitType(typeAnnotation);
      }, {
        processRightHandNodes: true
      });
    }

    this.visitChildren(node.body);
    this.close(node);
  }

  FunctionDeclaration(node) {
    this.visitFunction(node);
  }

  FunctionExpression(node) {
    this.visitFunction(node);
  }

  Identifier(node) {
    this.currentScope().referenceValue(node);
    this.visitType(node.typeAnnotation);
  }

  ImportAttribute(_) {// import assertions are module metadata and thus have no variables to reference
  }

  ImportDeclaration(node) {
    if (!this.scopeManager.isES6() || !this.scopeManager.isModule()) {
      throw new Error('ImportDeclaration should appear when the mode is ES6 and in the module context.');
    }

    _ImportVisitor.ImportVisitor.visit(this, node);
  }

  JSXAttribute(node) {
    this.visit(node.value);
  }

  JSXClosingElement(node) {
    /**
     * Note that this was not previously considered to be a reference and that
     * other scope analyzers do not count them either: e.g. TypeScript-eslint
     * https://fburl.com/4q93a3x3
     *
     * We are considering this a reference because it technically includes an
     * identifier that refers to a defined variable. So, if you want to answer:
     * "what are all of the references to this variable?", the closing element
     * should be included.
     */
    this.visitJSXTag(node);
  }

  JSXFragment(node) {
    this._referenceJsxPragma();

    this._referenceJsxFragment();

    this.visitChildren(node);
  }

  JSXIdentifier(node) {
    this.currentScope().referenceValue(node);
  }

  JSXMemberExpression(node) {
    this.visit(node.object); // we don't ever reference the property as it's always going to be a property on the thing
  }

  JSXNamespacedName(node) {
    this.visit(node.namespace); // namespace:name
    // the "name" doesn't reference a variable so it should be ignored, like a member expression
  }

  JSXOpeningElement(node) {
    this.visitJSXTag(node); // the opening tag may also have type args and attributes

    this.visitType(node.typeArguments);

    for (const attr of node.attributes) {
      this.visit(attr);
    }
  }

  LabeledStatement(node) {
    this.visit(node.body);
  }

  MemberExpression(node) {
    this.visit(node.object);

    if (node.computed === true) {
      this.visit(node.property);
    }
  }

  MetaProperty(_) {// meta properties all builtin globals
  }

  NewExpression(node) {
    this.visitChildren(node, ['typeArguments']);
    this.visitType(node.typeArguments);
  }

  PrivateIdentifier(_) {// private names can only reference class properties
  }

  Program(node) {
    this.scopeManager.nestGlobalScope(node);

    if (this.scopeManager.isGlobalReturn()) {
      // Force strictness of GlobalScope to false when using node.js scope.
      this.currentScope().isStrict = false;
      this.scopeManager.nestFunctionScope(node, false);
    }

    if (this.scopeManager.isES6() && this.scopeManager.isModule()) {
      this.scopeManager.nestModuleScope(node);
    }

    if (this.scopeManager.isStrictModeSupported() && this.scopeManager.isImpliedStrict()) {
      this.currentScope().isStrict = true;
    }

    this.visitChildren(node);
    this.close(node);
  }

  Property(node) {
    if (node.computed) {
      this.visit(node.key);
    }

    this.visit(node.value);
  }

  SwitchStatement(node) {
    this.visit(node.discriminant);

    if (this.scopeManager.isES6()) {
      this.scopeManager.nestSwitchScope(node);
    }

    for (const switchCase of node.cases) {
      this.visit(switchCase);
    }

    this.close(node);
  }

  TaggedTemplateExpression(node) {
    this.visit(node.tag);
    this.visit(node.quasi);
  }

  UpdateExpression(node) {
    if (_PatternVisitor.PatternVisitor.isPattern(node.argument)) {
      this.visitPattern(node.argument, pattern => {
        this.currentScope().referenceValue(pattern, _Reference.ReferenceFlag.ReadWrite, null);
      }, () => {});
    } else {
      this.visitChildren(node);
    }
  }

  VariableDeclaration(node) {
    const variableTargetScope = node.kind === 'var' ? this.currentScope().variableScope : this.currentScope();

    for (const decl of node.declarations) {
      const init = decl.init;
      this.visitPattern(decl.id, (pattern, info) => {
        variableTargetScope.defineIdentifier(pattern, new _definition.VariableDefinition(pattern, decl, node));
        this.referencingDefaultValue(pattern, info.assignments, null, true);

        if (init) {
          this.currentScope().referenceValue(pattern, _Reference.ReferenceFlag.Write, init, null, true);
        }
      }, typeAnnotation => {
        this.visitType(typeAnnotation);
      }, {
        processRightHandNodes: true
      });

      if (decl.init) {
        this.visit(decl.init);
      }
    }
  }

  WithStatement(node) {
    this.visit(node.object); // Then nest scope for WithStatement.

    this.scopeManager.nestWithScope(node);
    this.visit(node.body);
    this.close(node);
  } //
  // Type node passthrough visitors
  //


  DeclareClass(node) {
    this.visitType(node);
  }

  DeclareVariable(node) {
    this.visitType(node);
  }

  DeclareEnum(node) {
    this.currentScope().defineIdentifier(node.id, new _definition.EnumDefinition(node.id, node)); // Enum body cannot contain identifier references, so no need to visit body.
  }

  DeclareComponent(node) {
    this.visitType(node);
  }

  DeclareFunction(node) {
    this.visitType(node);
  }

  DeclareHook(node) {
    this.visitType(node);
  }

  DeclareModule(node) {
    this.visitType(node);
  }

  DeclareModuleExports(node) {
    this.visitType(node);
  }

  DeclareNamespace(node) {
    this.visitType(node);
  }

  DeclareInterface(node) {
    this.visitType(node);
  }

  DeclareTypeAlias(node) {
    this.visitType(node);
  }

  DeclareOpaqueType(node) {
    this.visitType(node);
  }

  DeclareExportAllDeclaration(node) {
    this.visitType(node);
  }

  DeclareExportDeclaration(node) {
    this.visitType(node);
  }

  InterfaceDeclaration(node) {
    this.visitType(node);
  }

  OpaqueType(node) {
    this.visitType(node);
  }

  TypeAlias(node) {
    this.visitType(node);
  }

  TypeCastExpression(node) {
    this.visit(node.expression);
    this.visitType(node.typeAnnotation);
  }

}

exports.Referencer = Referencer;