/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */
'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ScopeBase = void 0;

var _hermesEstree = require("hermes-estree");

var _ScopeType = require("./ScopeType");

var _definition = require("../definition");

var _ID = require("../ID");

var _Reference = require("../referencer/Reference");

var _variable2 = require("../variable");

/**
 * Test if scope is strict
 */
function isStrictScope(scope, isMethodDefinition) {
  // When upper scope exists and is strict, inner scope is also strict.
  if (scope.upper && scope.upper.isStrict) {
    return true;
  }

  if (isMethodDefinition) {
    return true;
  }

  if (scope.type === _ScopeType.ScopeType.Class || scope.type === _ScopeType.ScopeType.Module) {
    return true;
  }

  if (scope.type === _ScopeType.ScopeType.Block || scope.type === _ScopeType.ScopeType.Switch) {
    return false;
  }

  const body = (() => {
    if (scope.type === _ScopeType.ScopeType.Function) {
      if (scope.block.type === 'ArrowFunctionExpression') {
        if (scope.block.body.type !== 'BlockStatement') {
          return false;
        } else {
          return scope.block.body;
        }
      } else if (scope.block.type === 'Program') {
        return scope.block;
      } else {
        return scope.block.body;
      }
    } else if (scope.type === _ScopeType.ScopeType.Global) {
      return scope.block;
    } else {
      return false;
    }
  })();

  if (body === false) {
    return false;
  } // Search 'use strict' directive.


  for (let i = 0, iz = body.body.length; i < iz; ++i) {
    const stmt = body.body[i];

    if (stmt.type !== 'ExpressionStatement') {
      break;
    }

    const expr = stmt.expression;

    if (!(0, _hermesEstree.isStringLiteral)(expr)) {
      break;
    }

    return expr.value === 'use strict';
  }

  return false;
}
/**
 * Register scope
 */


function registerScope(scopeManager, scope) {
  scopeManager.scopes.push(scope);
  const scopes = scopeManager.nodeToScope.get(scope.block);

  if (scopes) {
    scopes.push(scope);
  } else {
    scopeManager.nodeToScope.set(scope.block, [scope]);
  }
}

function asScope(scope) {
  // $FlowExpectedError[incompatible-return] - it's impossible to tell flow this is safe
  return scope;
}

const generator = (0, _ID.createIdGenerator)();

/* abstract */
class ScopeBase {
  /**
   * A unique ID for this instance - primarily used to help debugging and testing
   */

  /**
   * The AST node which created this scope.
   * @public
   */

  /**
   * The array of child scopes. This does not include grandchild scopes.
   * @public
   */

  /**
   * A map of the variables for each node in this scope.
   * This is map is a pointer to the one in the parent ScopeManager instance
   */

  /**
   * Generally, through the lexical scoping of JS you can always know which variable an identifier in the source code
   * refers to. There are a few exceptions to this rule. With `global` and `with` scopes you can only decide at runtime
   * which variable a reference refers to.
   * All those scopes are considered "dynamic".
   */

  /**
   * Whether this scope is created by a FunctionExpression.
   * @public
   */

  /**
   * Whether 'use strict' is in effect in this scope.
   * @public
   */

  /**
   * List of {@link Reference}s that are left to be resolved (i.e. which
   * need to be linked to the variable they refer to).
   */

  /**
   * Any variable {@link Reference} found in this scope.
   * This includes occurrences of local variables as well as variables from parent scopes (including the global scope).
   * For local variables this also includes defining occurrences (like in a 'var' statement).
   * In a 'function' scope this does not include the occurrences of the formal parameter in the parameter list.
   * @public
   */

  /**
   * The map from variable names to variable objects.
   * @public
   */

  /**
   * The {@link Reference}s that are not resolved with this scope.
   * @public
   */

  /**
   * The type of scope
   * @public
   */

  /**
   * Reference to the parent {@link Scope}.
   * @public
   */

  /**
   * The scoped {@link Variable}s of this scope.
   * In the case of a 'function' scope this includes the automatic argument `arguments` as its first element, as well
   * as all further formal arguments.
   * This does not include variables which are defined in child scopes.
   * @public
   */

  /**
   * For scopes that can contain variable declarations, this is a self-reference.
   * For other scope types this is the *variableScope* value of the parent scope.
   * @public
   */

  /**
   * The names that are indirectly referenced within this scope.
   * @private
   */
  constructor(_scopeManager, type, upperScope, block, isMethodDefinition) {
    this.$id = generator();
    this.block = void 0;
    this.childScopes = [];
    this.__declaredVariables = void 0;
    this.__dynamic = void 0;
    this.functionExpressionScope = false;
    this.isStrict = void 0;
    this.__referencesLeftToResolve = [];
    this.references = [];
    this.set = new Map();
    this.through = [];
    this.type = void 0;
    this.upper = void 0;
    this.variables = [];
    this.variableScope = void 0;
    this.__indirectReferences = new Set();

    this._staticCloseRef = (ref, _) => {
      const resolve = () => {
        const name = ref.identifier.name;
        const variable = this.set.get(name);

        if (!variable) {
          return false;
        }

        if (!this.__isValidResolution(ref, variable)) {
          return false;
        } // make sure we don't match a type reference to a value variable


        const isValidTypeReference = ref.isTypeReference && variable.isTypeVariable;
        const isValidValueReference = ref.isValueReference && variable.isValueVariable;

        if (!isValidTypeReference && !isValidValueReference) {
          return false;
        }

        variable.references.push(ref);
        ref.resolved = variable;
        return true;
      };

      if (!resolve()) {
        this.__delegateToUpperScope(ref);
      }
    };

    this._dynamicCloseRef = (ref, _) => {
      // notify all names are through to global
      let current = asScope(this);

      while (current) {
        current.through.push(ref);
        current = current.upper;
      }
    };

    this._globalCloseRef = (ref, scopeManager) => {
      // let/const/class declarations should be resolved statically.
      // others should be resolved dynamically.
      if (this._shouldStaticallyCloseForGlobal(ref, scopeManager)) {
        this._staticCloseRef(ref);
      } else {
        this._dynamicCloseRef(ref);
      }
    };

    this.type = type;
    this.__dynamic = this.type === _ScopeType.ScopeType.Global || this.type === _ScopeType.ScopeType.With;
    this.block = block;
    this.variableScope = this.type === _ScopeType.ScopeType.ClassFieldInitializer || this.type === _ScopeType.ScopeType.ClassStaticBlock || this.type === _ScopeType.ScopeType.Function || this.type === _ScopeType.ScopeType.Global || this.type === _ScopeType.ScopeType.Module || this.type === _ScopeType.ScopeType.DeclareModule || this.type === _ScopeType.ScopeType.DeclareNamespace ? // $FlowFixMe[incompatible-type] not possible to teach flow this is safe
    this : // $FlowFixMe[incompatible-use] upperScope can only be null for Global scope
    upperScope.variableScope;
    this.upper = upperScope;
    this.isStrict = isStrictScope(asScope(this), isMethodDefinition);

    if (this.upper) {
      this.upper.childScopes.push(asScope(this));
    }

    this.__declaredVariables = _scopeManager.declaredVariables;
    registerScope(_scopeManager, asScope(this));
  }

  shouldStaticallyClose() {
    return !this.__dynamic;
  }

  _shouldStaticallyCloseForGlobal(ref, scopeManager) {
    // On global scope, let/const/class declarations should be resolved statically.
    const name = ref.identifier.name;
    const variable = this.set.get(name);

    if (!variable) {
      return false;
    } // variable exists on the scope
    // in module mode, we can statically resolve everything, regardless of its decl type


    if (scopeManager.isModule()) {
      return true;
    } // in script mode, only certain cases should be statically resolved
    // Example:
    // a `var` decl is ignored by the runtime if it clashes with a global name
    // this means that we should not resolve the reference to the variable


    const defs = variable.defs;
    return defs.length > 0 && defs.every(def => {
      var _def$parent;

      if (def.type === _definition.DefinitionType.Variable && ((_def$parent = def.parent) == null ? void 0 : _def$parent.type) === 'VariableDeclaration' && def.parent.kind === 'var') {
        return false;
      }

      return true;
    });
  }

  close(scopeManager) {
    let closeRef;

    if (this.shouldStaticallyClose()) {
      closeRef = this._staticCloseRef;
    } else if (this.type !== 'global') {
      closeRef = this._dynamicCloseRef;
    } else {
      closeRef = this._globalCloseRef;
    } // Try Resolving all references in this scope.


    if (this.__referencesLeftToResolve == null) {
      throw new Error('__referencesLeftToResolve was unexpectedly null.');
    }

    for (let i = 0; i < this.__referencesLeftToResolve.length; ++i) {
      const ref = this.__referencesLeftToResolve[i];
      closeRef(ref, scopeManager);
    }

    this.__referencesLeftToResolve = null;

    if (this.__indirectReferences.size > 0) {
      const upper = this.upper;

      for (const name of this.__indirectReferences) {
        const variable = this.set.get(name);

        if (variable) {
          variable.eslintUsed = true;

          this.__indirectReferences.delete(name);

          continue;
        } // delegate it to the upper scope


        if (upper) {
          upper.__indirectReferences.add(name);

          this.__indirectReferences.delete(name);
        }
      }
    }

    return this.upper;
  }
  /**
   * To override by function scopes.
   * References in default parameters isn't resolved to variables which are in their function body.
   */


  __isValidResolution(_ref, _variable) {
    return true;
  }

  __delegateToUpperScope(ref) {
    var _this$upper;

    if ((_this$upper = this.upper) != null && _this$upper.__referencesLeftToResolve) {
      this.upper.__referencesLeftToResolve.push(ref);
    }

    this.through.push(ref);
  }

  _addDeclaredVariablesOfNode(variable, node) {
    if (node == null) {
      return;
    }

    let variables = this.__declaredVariables.get(node);

    if (variables == null) {
      variables = [];

      this.__declaredVariables.set(node, variables);
    }

    if (!variables.includes(variable)) {
      variables.push(variable);
    }
  }

  __defineVariable(nameOrVariable, set, variables, node, def) {
    const name = typeof nameOrVariable === 'string' ? nameOrVariable : nameOrVariable.name;
    let variable = set.get(name);

    if (!variable) {
      variable = typeof nameOrVariable === 'string' ? new _variable2.Variable(name, asScope(this)) : nameOrVariable;
      set.set(name, variable);
      variables.push(variable);
    }

    if (def) {
      variable.defs.push(def);

      this._addDeclaredVariablesOfNode(variable, def.node);

      this._addDeclaredVariablesOfNode(variable, def.parent);
    }

    if (node) {
      variable.identifiers.push(node);
    }
  }

  defineIdentifier(node, def) {
    this.__defineVariable(node.name, this.set, this.variables, node, def);
  }

  defineLiteralIdentifier(node, def) {
    this.__defineVariable(node.value, this.set, this.variables, null, def);
  }

  referenceValue(node, assign = _Reference.ReferenceFlag.Read, writeExpr, maybeImplicitGlobal, init = false) {
    var _this$__referencesLef;

    const ref = new _Reference.Reference(node, asScope(this), assign, writeExpr, maybeImplicitGlobal, init, _Reference.ReferenceTypeFlag.Value);
    this.references.push(ref);
    (_this$__referencesLef = this.__referencesLeftToResolve) == null ? void 0 : _this$__referencesLef.push(ref);
  }
  /**
   * Creates an indirect reference to a given `name` `from` the given location
   * This is useful when a build process is expected to create a reference to
   * the name, for example - the JSX transform that references a JSX pragma (React)
   */


  indirectlyReferenceValue(name) {
    this.__indirectReferences.add(name);
  }

  referenceType(node) {
    var _this$__referencesLef2;

    const ref = new _Reference.Reference(node, asScope(this), _Reference.ReferenceFlag.Read, null, null, false, _Reference.ReferenceTypeFlag.Type);
    this.references.push(ref);
    (_this$__referencesLef2 = this.__referencesLeftToResolve) == null ? void 0 : _this$__referencesLef2.push(ref);
  }

  referenceDualValueType(node) {
    var _this$__referencesLef3;

    const ref = new _Reference.Reference(node, asScope(this), _Reference.ReferenceFlag.Read, null, null, false, _Reference.ReferenceTypeFlag.ValueAndType);
    this.references.push(ref);
    (_this$__referencesLef3 = this.__referencesLeftToResolve) == null ? void 0 : _this$__referencesLef3.push(ref);
  }

}

exports.ScopeBase = ScopeBase;