import socket
from enum import Enum
from typing import Tuple, Optional

class DescriptorPortType(Enum):
    """
    @brief Types of descriptor ports.

    This enum defines the types of descriptor ports used in the system.

    Example usage:
    @code
    port_type = DescriptorPortType.DATA
    if port_type == DescriptorPortType.DATA:
        # Handle data port
    elif port_type == DescriptorPortType.MGMT:
        # Handle management port
    elif port_type == DescriptorPortType.STATUS:
        # Handle status port
    @endcode
    """
    DATA: int
    MGMT: int
    STATUS: int

DEFAULT_DATA_PORT = 9100  #: @brief Port for sending print data.
DEFAULT_MGMT_PORT = 3007  #: @brief Port used to control and manage printers.
DEFAULT_STATUS_PORT = 3002  #: @brief Printer may be configured to send status messages to different ports based on the "Ret. Status Port" setting on the printer.

class TcpComm:
    """
    @class TcpComm
    @brief A class for TCP communication.
    """
    
    def __init__(self, ipAddress: str, port: int) -> None:
        """
        @brief Initializes a new instance with the given IP address and port.

        @param ipAddress The IP address of the TCP server.
        @param port The port number of the TCP server.
        """
        ...
    
    @classmethod
    def from_descriptor(cls, descriptor: str, port_type: DescriptorPortType) -> "TcpComm":
        """
        @brief Creates a TcpComm object from a descriptor string and port type.

        @param descriptor The descriptor string representing the connection.
        @param port_type The type of port to use for the connection.
        @return An instance of the TcpComm class.
        """
        ...
    
    @staticmethod
    def parse_descriptor(descriptor: str, port_type: DescriptorPortType) -> Tuple[str, int]:
        """
        @brief Parses a descriptor string into an IP address and port number.

        @param descriptor The descriptor string representing the connection.
        @param port_type The type of port to use for the connection.
        @return A tuple containing the IP address and port number.
        """
        ...
    
    @staticmethod
    def descriptor_validate(descriptor_hint: str, port_type_hint: DescriptorPortType) -> str:
        """
        @brief Validates the descriptor with the specified hint and port type.

        @param descriptor_hint A hint for the descriptor.
        @param port_type_hint The type of port to validate.
        @return A validated descriptor string.
        """
        ...
    
    @staticmethod
    def descriptor_use_ip(ip: str, port: int) -> str:
        """
        @brief Returns a descriptor string using the specified IP and port.

        @param ip The IP address to use in the descriptor.
        @param port The port to use in the descriptor.
        @return A descriptor string.
        """
        ...
    
    def open(self) -> None:
        """
        @brief Opens the connection to the TCP server.
        """
        ...
    
    def get_ip_address(self) -> str:
        """
        @brief Gets the IP address of the remote server.

        @return The IP address of the remote server.
        """
        ...
    
    def connection_settings(self, ip_address: str, port: int) -> None:
        """!
        @brief Configures connection settings with the specified IP address and port.

        @param ip_address The IP address of the remote server.
        @param port The port number to connect to on the remote server.
        """
        ...
    
    def is_connected(self) -> bool:
        """
        @brief A flag indicating whether the connection is established.

        @return True if the connection is open, False otherwise.
        """
        ...
    
    def close(self) -> None:
        """
        @brief Closes the connection to the TCP server.
        """
        ...
    
    def write(self, text: bytes) -> None:
        """
        @brief Writes data to the TCP server.

        @param text The data to write to the server.
        """
        ...
    
    def read(self) -> Optional[bytes]:
        """
        @brief Reads data from the TCP server.

        @return The data read from the server.
        """
        ...
    
    def wait_for_data(self, msTimeout: int) -> None:
        """
        @brief Waits for data to be available on the TCP connection.

        @param msTimeout The timeout interval in milliseconds.
        """
        ...
    
    def write_and_wait_for_response(self, text: bytes, responseStartTimeout: int, responseEndTimeout: int, completionToken: str = "") -> Optional[bytes]:
        """
        @brief Writes data to the TCP server and waits for a response.

        @param text The data to write to the server.
        @param responseStartTimeout The timeout interval for the start of the response.
        @param responseEndTimeout The timeout interval for the end of the response.
        @param completionToken A token indicating the completion of the response.
        @return The data received in response from the server.
        """
        ...
    
    def write_and_wait_for_response_json(self, text: bytes, responseStartTimeout: int, responseEndTimeout: int, completionToken: str = "") -> Optional[bytes]:
        """
        @brief Writes JSON data to the TCP server and waits for a response.

        @param text The JSON data to write to the server.
        @param responseStartTimeout The timeout interval for the start of the response.
        @param responseEndTimeout The timeout interval for the end of the response.
        @param completionToken A token indicating the completion of the response.
        @return The data received in response from the server.
        """
        ...
    
    def write_printer_file(self, fileName: str) -> None:
        """
        @brief Writes a file to the printer.

        @param fileName The name of the file to write to the printer.
        """
        ...
    
    def send_printer_file(self, ipAddress: str, fileName: str) -> None:
        """
        @brief Sends a print file to the specified IP address.

        @param ipAddress The IP address of the printer.
        @param fileName The name of the file to print.
        """
        ...
    
    def send_print_string(self, ipAddress: str, data: bytes) -> None:
        """
        @brief Sends a print string using **TSPL** or **PGL** to the specified IP address.

        @param ipAddress The IP address of the printer.
        @param data The data to print.
        """
        ...
