//
//  UsbComm.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/3/12.
//

/**@cond**/


#import <Foundation/Foundation.h>
#import <IOKit/usb/IOUSBLib.h>
#import <UniPRT/UsbInterfaceInfo.h>
#import <UniPRT/AComm.h>

/**
 * \~English
 * @class UsbComm
 * @brief Class that implements the USB specific communication and extends the AComm class.
 * @ingroup Comm
 * \~Chinese
 * @class UsbComm
 * @brief 实现USB特定通信并继承AComm类的类。
 * @ingroup Comm
 * \~Chinese-Traditional
 * @class UsbComm
 * @brief 實現USB特定通信並繼承AComm類的類。
 * @ingroup Comm
 */
@interface UsbComm: AComm

@property (nonatomic, strong, readonly) NSMutableArray<UsbInterfaceInfo *> *discoveredInterfaces;
@property (readonly, nonatomic) NSMutableDictionary *deviceList;
@property (nonatomic, readonly) UsbInterfaceInfo *matchedInterfaces;

/** @cond */
@property (nonatomic, readonly) BOOL shouldStopReadLoop;
/** @endcond */

/**
 * \~English
 * @brief Initializes a new instance with the given vendor ID and product ID.
 *
 * @param vendorId The vendor ID for the instance.
 * @param productId The product ID for the instance.
 * @return An initialized instance of the class, or nil if the object could not be created.
 * \~Chinese
 * @brief 使用给定的厂商ID和产品ID初始化一个新实例。
 *
 * @param vendorId 实例的厂商ID。
 * @param productId 实例的产品ID。
 * @return 初始化后的类实例，如果无法创建则返回nil。
 * \~Chinese-Traditional
 * @brief 使用給定的廠商ID和產品ID初始化一個新實例。
 *
 * @param vendorId 實例的廠商ID。
 * @param productId 實例的產品ID。
 * @return 初始化後的類實例，如果無法創建則返回nil。
 */
- (instancetype)initWithVendorId:(NSString *)vendorId productId:(NSString *)productId;

/**
 * \~English
 * @brief Opens the connection to the device.
 * \~Chinese
 * @brief 打开设备连接。
 * \~Chinese-Traditional
 * @brief 打開設備連接。
 */
- (void)open;

/**
 * \~English
 * @brief Closes the connection to the device.
 * \~Chinese
 * @brief 关闭设备连接。
 * \~Chinese-Traditional
 * @brief 關閉設備連接。
 */
- (void)close;

/**
 * \~English
 * @brief Writes data to the device.
 *
 * @param text The data to write to the device.
 * \~Chinese
 * @brief 向设备写入数据。
 *
 * @param text 要写入设备的数据。
 * \~Chinese-Traditional
 * @brief 向設備寫入數據。
 *
 * @param text 要寫入設備的數據。
 */
- (void)write:(NSData *)text;

/**
 * \~English
 * @brief Reads data from the device.
 *
 * @return The data read from the device.
 * \~Chinese
 * @brief 从设备读取数据。
 *
 * @return 从设备读取的数据。
 * \~Chinese-Traditional
 * @brief 從設備讀取數據。
 *
 * @return 從設備讀取的數據。
 */
- (NSData*)read;

/**
 * \~English
 * @brief Writes data to the device and waits for a response.
 *
 * @param text The data to write to the device.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the device.
 * \~Chinese
 * @brief 向设备写入数据并等待响应。
 *
 * @param text 要写入设备的数据。
 * @param responseStartTimeOut 响应开始的超时时间。
 * @param responseEndTimeOut 响应结束的超时时间。
 * @param completionToken 表示响应完成的标记。
 * @return 从设备收到的响应数据。
 * \~Chinese-Traditional
 * @brief 向設備寫入數據並等待響應。
 *
 * @param text 要寫入設備的數據。
 * @param responseStartTimeOut 響應開始的超時時間。
 * @param responseEndTimeOut 響應結束的超時時間。
 * @param completionToken 表示響應完成的標記。
 * @return 從設備收到的響應數據。
 */
- (NSData*)writeAndWaitForResponse:(NSData *)text
              responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                   completionToken:(NSString *)completionToken;

/**
 * \~English
 * @brief Asynchronously listens for and reads data from the device.
 * \~Chinese
 * @brief 异步监听并读取设备数据。
 * \~Chinese-Traditional
 * @brief 異步監聽並讀取設備數據。
 */
- (void)asyncListenRead;

/**
 * \~English
 * @brief Stops asynchronous reading from the device.
 * \~Chinese
 * @brief 停止从设备的异步读取。
 * \~Chinese-Traditional
 * @brief 停止從設備的異步讀取。
 */
- (void)stopListenRead;

@end

/**@endcond**/
