//
//  Pdf417Barcode.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/16.
//

#ifndef Pdf417Barcode_h
#define Pdf417Barcode_h

#import <Foundation/Foundation.h>
#import <UniPRT/APdf417.h>
#import <UniPRT/IPoint.h>

/**
 * \~English
 * @class Pdf417Barcode
 * @brief A class representing a PDF417 barcode for TSPL.
 *
 * This class inherits from APdf417 and provides functionality for initializing and describing PDF417 barcodes in TSPL.
 *
 * @ingroup TSPL
 *
 * \~Chinese
 * @class Pdf417Barcode
 * @brief 表示 TSPL 中 PDF417 条形码的类。
 *
 * 该类继承自 APdf417，提供了在 TSPL 中初始化和描述 PDF417 条形码的功能。
 *
 * @ingroup TSPL
 *
 * \~Chinese-Traditional
 * @class Pdf417Barcode
 * @brief 表示 TSPL 中 PDF417 條形碼的類。
 *
 * 該類繼承自 APdf417，提供了在 TSPL 中初始化和描述 PDF417 條形碼的功能。
 *
 * @ingroup TSPL
 */
@interface Pdf417Barcode : APdf417

/**
 * \~English
 * @brief The starting point of the barcode.
 *
 * \~Chinese
 * @brief 条形码的起始点。
 *
 * \~Chinese-Traditional
 * @brief 條形碼的起始點。
 */
@property (nonatomic, strong) id<IPoint> start;

/**
 * \~English
 * @brief The data to be encoded in the barcode.
 *
 * \~Chinese
 * @brief 要在条形码中编码的数据。
 *
 * \~Chinese-Traditional
 * @brief 要在條形碼中編碼的數據。
 */
@property (nonatomic, strong) NSString *data;

/**
 * \~English
 * @brief Initializes a new instance of the Pdf417Barcode class.
 *
 * @return A new instance of Pdf417Barcode.
 *
 * \~Chinese
 * @brief 初始化 Pdf417Barcode 类的新实例。
 *
 * @return Pdf417Barcode 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 初始化 Pdf417Barcode 類的新實例。
 *
 * @return Pdf417Barcode 的新實例。
 */
- (instancetype)init;

/**
 * \~English
 * @brief Initializes a new instance of the Pdf417Barcode class with the specified start point and data.
 *
 * @param start The starting point of the barcode.
 * @param data The data to be encoded in the barcode.
 * @return A new instance of Pdf417Barcode.
 *
 * \~Chinese
 * @brief 使用指定的起点和数据初始化 Pdf417Barcode 类的新实例。
 *
 * @param start 条形码的起始点。
 * @param data 要在条形码中编码的数据。
 * @return Pdf417Barcode 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的起點和數據初始化 Pdf417Barcode 類的新實例。
 *
 * @param start 條形碼的起始點。
 * @param data 要在條形碼中編碼的數據。
 * @return Pdf417Barcode 的新實例。
 */
- (instancetype)initWithStart:(id<IPoint>)start data:(NSString *)data;

/**
 * \~English
 * @brief Returns a string that can be sent to the printer.
 *
 * This method generates a string representation of the Pdf417Barcode instance that
 * is formatted and ready to be sent to a printer. The string includes all the
 * necessary commands and data required for printing in TSPL.
 *
 * @return A string description of the Pdf417Barcode instance.
 *
 * \~Chinese
 * @brief 返回可以发送到打印机的字符串。
 *
 * 该方法生成 Pdf417Barcode 实例的字符串表示形式，
 * 格式化并准备好发送到打印机。字符串包括在 TSPL 中打印所需的所有命令和数据。
 *
 * @return Pdf417Barcode 实例的字符串描述。
 *
 * \~Chinese-Traditional
 * @brief 返回可以發送到打印機的字符串。
 *
 * 該方法生成 Pdf417Barcode 實例的字符串表示形式，
 * 格式化並準備好發送到打印機。字符串包括在 TSPL 中打印所需的所有命令和數據。
 *
 * @return Pdf417Barcode 實例的字符串描述。
 */
- (NSString *)description;

@end

#endif /* Pdf417Barcode_h */
