//
//  AComm.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/3/17.
//

#import <Foundation/Foundation.h>
#import <AutoIdSDK/IComm.h>
/**
 * @class AComm
 * @brief A base class for communication interfaces, conforming to the IComm protocol.
 * @ingroup Comm
 */
@interface AComm : NSObject <IComm>

/**
 * @brief The maximum packet size for communication.
 */
@property (nonatomic, readonly) int maxPacketSize;

/**
 * @brief The descriptor for the communication interface.
 */
@property (nonatomic, readonly) NSString *descriptor;

/**
 * @brief The number of bytes available to read.
 */
@property (nonatomic, readonly) int bytesAvailable;

/**
 * @brief A flag indicating whether the connection is established.
 */
@property (nonatomic) BOOL isConnected;

/**
 * @brief Closes the communication interface.
 */
- (void)close;

/**
 * @brief Opens the communication interface.
 */
- (void)open;

/**
 * @brief Reads data from the communication interface.
 *
 * @return The data read from the interface.
 */
- (NSData *)read;

/**
 * @brief Writes data to the communication interface.
 *
 * @param dataOut The data to write to the interface.
 */
- (void)write:(NSData *)dataOut;

/**
 * @brief Reads data into a binary writer (output stream).
 *
 * @param binDataIn The output stream to write the data into.
 */
- (void)readToBinaryWriter:(NSOutputStream *)binDataIn;

/**
 * @brief Waits for data to be available within a specified timeout.
 *
 * @param msTimeOut The timeout interval in milliseconds.
 */
- (void)waitForDataWithTimeout:(NSTimeInterval)msTimeOut;

/**
 * @brief Writes data from a binary reader (input stream) to the communication interface.
 *
 * @param binReader The input stream to read the data from.
 */
- (void)writeFromBinaryReader:(NSInputStream *)binReader;

/**
 * @brief Writes data to the communication interface and waits for a response, using binary streams.
 *
 * @param binDataIn The output stream to write the data into.
 * @param binDataOut The input stream to read the response from.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 */
- (void)writeAndWaitForResponseToBinaryWriter:(NSOutputStream *)binDataIn
                              fromBinaryReader:(NSInputStream *)binDataOut
                          responseStartTimeout:(int)responseStartTimeOut
                            responseEndTimeout:(int)responseEndTimeOut
                               completionToken:(NSString *)completionToken;

/**
 * @brief Writes data to the communication interface and waits for a response.
 *
 * @param text The data to write to the interface.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the interface.
 */
- (NSData *)writeAndWaitForResponse:(NSData *)text
               responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                 responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                    completionToken:(NSString *)completionToken;

@end
