//
//  IComm.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/1.
//

#import <Foundation/Foundation.h>

/**
 * @protocol IComm
 * @brief A protocol defining the interface for communication classes.
 * @ingroup Comm
 */
@protocol IComm <NSObject>

/**
 * @brief The descriptor for the communication interface.
 */
@property (nonatomic, readonly) NSString *descriptor;

/**
 * @brief The number of bytes available to read.
 */
@property (nonatomic, readonly) int bytesAvailable;

/**
 * @brief A flag indicating whether the connection is established.
 */
@property (nonatomic, readonly) BOOL isConnected;

/**
 * @brief Closes the communication interface.
 */
- (void)close;

/**
 * @brief Opens the communication interface.
 */
- (void)open;

/**
 * @brief Reads data from the communication interface.
 *
 * @return The data read from the interface.
 */
- (NSData *)read;

/**
 * @brief Writes data from a binary reader (input stream) to the output stream.
 *
 * @param binDataIn The output stream to write the data into.
 */
- (void)readToBinaryWriter:(NSOutputStream *)binDataIn;

/**
 * @brief Waits until data is available in the current thread. The current thread sleeps until data is received or the timeout is reached. This is a blocking call.
 *
 * @param msTimeOut The timeout interval in milliseconds before returning if no data is received.
 */
- (void)waitForDataWithTimeout:(NSTimeInterval)msTimeOut;

/**
 * @brief Writes all bytes from the array passed in.
 *
 * @param binReader The input stream to read the data from.
 */
- (void)writeFromBinaryReader:(NSInputStream *)binReader;

/**
 * @brief Writes data to the communication interface.
 *
 * @param dataOut The data to write to the interface.
 */
- (void)write:(NSData *)dataOut;

/**
 * @brief Writes data from the input stream to the output stream and returns the data received in the output stream. The data returned is any data received or up to the completion token if received. Waits for a response timeout before returning.
 *
 * @param binDataIn The output stream to store the response received from the connection.
 * @param binDataOut The input stream data to write to the connection.
 * @param responseStartTimeOut The timeout interval to wait for the first data to be received (time to first byte).
 * @param responseEndTimeOut The timeout interval after the last byte if no new data is received (time after last byte).
 * @param completionToken The string token denoting the end of the expected response. This can be an empty string if simply waiting for a response after writing data to the connection.
 * @return The data received from the connection.
 */
- (NSData *)writeAndWaitForResponseToBinaryWriter:(NSOutputStream *)binDataIn
                                  fromBinaryReader:(NSInputStream *)binDataOut
                              responseStartTimeout:(int)responseStartTimeOut
                                responseEndTimeout:(int)responseEndTimeOut
                                   completionToken:(NSString *)completionToken;

/**
 * @brief Writes byte data to the output stream and returns the data received. The data returned is any data received or up to the completion token if received. Waits for a response timeout before returning.
 *
 * @param text The source byte array to write to the connection.
 * @param responseStartTimeOut The timeout interval to wait for the first data to be received (time to first byte).
 * @param responseEndTimeOut The timeout interval after the last byte if no new data is received (time after last byte).
 * @param completionToken The string token denoting the end of the expected response. This can be an empty string if simply waiting for a response after writing data to the connection.
 * @return The array of bytes received from the connection.
 */
- (NSData *)writeAndWaitForResponse:(NSData *)text
               responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                 responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                    completionToken:(NSString *)completionToken;

@end

