//
//  JsonMessenger.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/8.
//

#ifndef JsonMessenger_h
#define JsonMessenger_h


#endif /* JsonMessenger_h */

#import <Foundation/Foundation.h>
#import <AutoIdSDK/JsonComm.h>
#import <AutoIdSDK/JsonMng.h>
#import <AutoIdSDK/TcpComm.h>
#import <AutoIdSDK/JsonCommController.h>
#import <AutoIdSDK/UsbComm.h>


@class TcpComm;
@class BtComm;
@class UsbComm;
@class JsonMng;
@class JsonComm;

/**
 * @class JsonMessenger
 * @brief A class for managing JSON-based messaging.
 * @ingroup JSONMng
 */
@interface JsonMessenger : NSObject {
    BOOL _usingDataPort;              /**< A flag indicating whether the data port is used. */
    BOOL _disposed;                   /**< A flag indicating whether the object is disposed. */
    TcpComm *_tcpComm;                /**< TCP communication instance. */
    JsonComm *_MgmtComm;              /**< Management JSON communication instance. */
    JsonComm *_MgmtCommPtr;           /**< Pointer to the management JSON communication instance. */
    ImpChannel *_MsgsFromPtr;         /**< Channel for messages from the pointer. */
    
    NSString *_myMgmtId;              /**< Management ID. */
    long _TrackCount;                 /**< Track count. */
    BOOL _sdkManagedComm;             /**< A flag indicating whether the SDK manages communication. */
    NSString *_managedDescriptor;     /**< Descriptor managed by the SDK. */
}

/**
 * @brief Initializes a new instance with the given communication pointer, communication type, maximum input message capacity, and data port usage flag.
 *
 * @param commToPtr The pointer to the communication instance.
 * @param iCommTyp The type of communication.
 * @param maxInputMsgCapacity The maximum input message capacity.
 * @param usingDataPort A flag indicating whether to use the data port.
 * @return An initialized instance of JsonMessenger, or nil if the object could not be created.
 */
- (instancetype)initWithCommToPtr:(AComm *)commToPtr iCommTyp:(CommType)iCommTyp maxInputMsgCapacity:(int)maxInputMsgCapacity usingDataPort:(BOOL)usingDataPort;

/**
 * @brief Sends a message with the specified command, content, and track number.
 *
 * @param command The command of the message.
 * @param content The content of the message.
 * @param trackNo The track number of the message.
 * @return A string representing the result of the send operation.
 */
- (NSString *)sendMsgWithCommand:(NSString *)command content:(NSString *)content trackNo:(NSString *)trackNo;

/**
 * @brief Sends a message and waits for a response with the specified command, content, and maximum wait time.
 *
 * @param command The command of the message.
 * @param content The content of the message.
 * @param maxWaitTimeSecs The maximum wait time in seconds.
 * @return A JsonMng instance representing the response.
 */
- (JsonMng *)sendMsgAndWaitForResponseWithCommand:(NSString *)command content:(NSString *)content maxWaitTimeSecs:(int)maxWaitTimeSecs;

/**
 * @brief Reads the next message.
 *
 * @return A JsonMng instance representing the next message.
 */
- (JsonMng *)readNextMsg;

/**
 * @brief Sends a raw message with the specified data.
 *
 * @param dataToSend The data to send.
 */
- (void)sendMsgRawWithDataToSend:(NSString *)dataToSend;

/**
 * @brief Disposes the object, releasing any resources.
 */
- (void)dispose;

@end

