//
//  SettingsReadWrite.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/14.
//

#ifndef SettingsReadWrite_h
#define SettingsReadWrite_h


#endif /* SettingsReadWrite_h */

#import <Foundation/Foundation.h>
#import <UniPRT/TcpComm.h>
#import <UniPRT/JsonComm.h>
#import <UniPRT/Setting.h>
#import <UniPRT/JsonCommController.h>
#import <UniPRT/Debug.h>
#import <UniPRT/JsonStringBuilder.h>
#import <UniPRT/JsonStringTokenizer.h>

/**
 * \~English
 * @class SettingsReadWrite
 * @brief Support for reading and writing printer settings.
 *
 * For list of valid keys and values refer to: Settings Key/Value Items.
 * @ingroup Settings
 *
 * \~Chinese
 * @class SettingsReadWrite
 * @brief 支持读取和写入打印机设置。
 *
 * 有关有效键和值的列表，请参阅：Settings Key/Value Items。
 * @ingroup Settings
 *
 * \~Chinese-Traditional
 * @class SettingsReadWrite
 * @brief 支持讀取和寫入打印機設置。
 *
 * 有關有效鍵和值的列表，請參閱：Settings Key/Value Items。
 * @ingroup Settings
 */

/**
 * \~English
 * @brief Command item constant.
 *
 * \~Chinese
 * @brief 命令项常量。
 *
 * \~Chinese-Traditional
 * @brief 命令項常量。
 */
extern NSString *const CMD_ITEM;

/**
 * \~English
 * @brief Command property constant.
 *
 * \~Chinese
 * @brief 命令属性常量。
 *
 * \~Chinese-Traditional
 * @brief 命令屬性常量。
 */
extern NSString *const CMD_PROP;

@interface SettingsReadWrite : NSObject

/**
 * \~English
 * @brief Must pass an opened TcpComm connection that can remain opened if listening for unsolicited msgs from printer.
 *
 * A connection is necessary to send/receive printer status msgs.
 *
 * @param connection The TCP communication object.
 * @return An initialized instance of SettingsReadWrite, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 必须传递一个已打开的 TcpComm 连接，如果监听来自打印机的非请求消息，该连接可以保持打开状态。
 *
 * 发送/接收打印机状态消息需要连接。
 *
 * @param connection TCP 通信对象。
 * @return 一个初始化的 SettingsReadWrite 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 必須傳遞一個已打開的 TcpComm 連接，如果監聽來自打印機的非請求消息，該連接可以保持打開狀態。
 *
 * 發送/接收打印機狀態消息需要連接。
 *
 * @param connection TCP 通信對象。
 * @return 一個初始化的 SettingsReadWrite 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithTcpComm:(TcpComm *)connection;

/**
 * \~English
 * @brief Initializes a new instance with the given JSON communication object.
 *
 * @param mgmtComm The JSON communication object.
 * @return An initialized instance of SettingsReadWrite, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用给定的 JSON 通信对象初始化一个新实例。
 *
 * @param mgmtComm JSON 通信对象。
 * @return 一个初始化的 SettingsReadWrite 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用給定的 JSON 通信對象初始化一個新實例。
 *
 * @param mgmtComm JSON 通信對象。
 * @return 一個初始化的 SettingsReadWrite 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithJsonComm:(JsonComm *)mgmtComm;

/**
 * \~English
 * @brief Constructor called when creating a connection managed by the SDK.
 *
 * commDescriptor is descriptor for the communication used to connect with printer.
 *
 * TCP descriptor format: "ip" or "ip:port"
 *
 * e.g. "127.0.0.1" or "127.0.0.1:3007"
 *
 * If port is not provided, default value is used.
 *
 * @param commDescriptor The communication descriptor.
 * @return An initialized instance of SettingsReadWrite, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 创建由 SDK 管理的连接时调用的构造函数。
 *
 * commDescriptor 是用于连接打印机的通信描述符。
 *
 * TCP 描述符格式："ip" 或 "ip:port"
 *
 * 例如 "127.0.0.1" 或 "127.0.0.1:3007"
 *
 * 如果未提供端口，则使用默认值。
 *
 * @param commDescriptor 通信描述符。
 * @return 一个初始化的 SettingsReadWrite 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 創建由 SDK 管理的連接時調用的構造函數。
 *
 * commDescriptor 是用於連接打印機的通信描述符。
 *
 * TCP 描述符格式："ip" 或 "ip:port"
 *
 * 例如 "127.0.0.1" 或 "127.0.0.1:3007"
 *
 * 如果未提供端口，則使用默認值。
 *
 * @param commDescriptor 通信描述符。
 * @return 一個初始化的 SettingsReadWrite 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithCommDescriptor:(NSString *)commDescriptor;

/**
 * \~English
 * @brief Read all settings key/value pairs.
 *
 * Implemented in SettingsReadWrite.
 *
 * @return A dictionary containing all settings keys and their corresponding values.
 *
 * \~Chinese
 * @brief 读取所有设置的键/值对。
 *
 * 在 SettingsReadWrite 中实现。
 *
 * @return 包含所有设置键及其相应值的字典。
 *
 * \~Chinese-Traditional
 * @brief 讀取所有設置的鍵/值對。
 *
 * 在 SettingsReadWrite 中實現。
 *
 * @return 包含所有設置鍵及其相應值的字典。
 */
- (NSDictionary<NSString *, NSString *> *)getAllValues;

/**
 * \~English
 * @brief Read setting associated with given key.
 *
 * Implemented in SettingsReadWrite.
 *
 * @param key The key for which to get the value.
 * @return The value associated with the specified key.
 *
 * \~Chinese
 * @brief 读取与给定键关联的设置。
 *
 * 在 SettingsReadWrite 中实现。
 *
 * @param key 要获取值的键。
 * @return 与指定键关联的值。
 *
 * \~Chinese-Traditional
 * @brief 讀取與給定鍵關聯的設置。
 *
 * 在 SettingsReadWrite 中實現。
 *
 * @param key 要獲取值的鍵。
 * @return 與指定鍵關聯的值。
 */
- (NSString *)getValueForKey:(NSString *)key;

/**
 * \~English
 * @brief Read multiple settings associated with given list of keys.
 *
 * Implemented in SettingsReadWrite.
 *
 * @param keys An array of keys for which to get the values.
 * @return A dictionary containing the specified keys and their corresponding values.
 *
 * \~Chinese
 * @brief 读取与给定键列表关联的多个设置。
 *
 * 在 SettingsReadWrite 中实现。
 *
 * @param keys 要获取值的键数组。
 * @return 包含指定键及其相应值的字典。
 *
 * \~Chinese-Traditional
 * @brief 讀取與給定鍵列表關聯的多個設置。
 *
 * 在 SettingsReadWrite 中實現。
 *
 * @param keys 要獲取值的鍵數組。
 * @return 包含指定鍵及其相應值的字典。
 */
- (NSDictionary<NSString *, NSString *> *)getValuesForKeys:(NSArray<NSString *> *)keys;

/**
 * \~English
 * @brief Write a setting value associated with given key. Returns true if write successful.
 *
 * Note: A value that is intended to be processed as strings should be quoted. e.g. value = SomeString and value = "SomeString" will be sent exactly as received but may be handled differently by printer.
 *
 * Implemented in SettingsReadWrite.
 *
 * @param value The value to set.
 * @param key The key for which to set the value.
 * @return YES if the value was successfully set, NO otherwise.
 *
 * \~Chinese
 * @brief 写入与给定键关联的设置值。如果写入成功，则返回 true。
 *
 * 注意：要作为字符串处理的值应加引号。例如，value = SomeString 和 value = "SomeString" 将按接收到的内容发送，但打印机可能会以不同方式处理。
 *
 * 在 SettingsReadWrite 中实现。
 *
 * @param value 要设置的值。
 * @param key 要设置值的键。
 * @return 如果值已成功设置，则返回 YES，否则返回 NO。
 *
 * \~Chinese-Traditional
 * @brief 寫入與給定鍵關聯的設置值。如果寫入成功，則返回 true。
 *
 * 注意：要作為字符串處理的值應加引號。例如，value = SomeString 和 value = "SomeString" 將按接收到的內容發送，但打印機可能會以不同方式處理。
 *
 * 在 SettingsReadWrite 中實現。
 *
 * @param value 要設置的值。
 * @param key 要設置值的鍵。
 * @return 如果值已成功設置，則返回 YES，否則返回 NO。
 */
- (BOOL)setValue:(NSString *)value forKey:(NSString *)key;

/**
 * \~English
 * @brief Write multiple settings associated with given keys. Returns true if write successful.
 *
 * Note: Values that are intended to be processed as strings should be quoted. e.g. value = SomeString and value = "SomeString" will be sent exactly as received but may be handled differently by printer.
 *
 * Implemented in SettingsReadWrite.
 *
 * @param keysToSet A dictionary containing the keys and values to set.
 * @return YES if the values were successfully set, NO otherwise.
 *
 * \~Chinese
 * @brief 写入与给定键关联的多个设置。如果写入成功，则返回 true。
 *
 * 注意：要作为字符串处理的值应加引号。例如，value = SomeString 和 value = "SomeString" 将按接收到的内容发送，但打印机可能会以不同方式处理。
 *
 * 在 SettingsReadWrite 中实现。
 *
 * @param keysToSet 包含要设置的键和值的字典。
 * @return 如果值已成功设置，则返回 YES，否则返回 NO。
 *
 * \~Chinese-Traditional
 * @brief 寫入與給定鍵關聯的多個設置。如果寫入成功，則返回 true。
 *
 * 注意：要作為字符串處理的值應加引號。例如，value = SomeString 和 value = "SomeString" 將按接收到的內容發送，但打印機可能會以不同方式處理。
 *
 * 在 SettingsReadWrite 中實現。
 *
 * @param keysToSet 包含要设置的键和值的字典。
 * @return 如果值已成功设置，则返回 YES，否则返回 NO。
 *
 * \~Chinese-Traditional
 * @param keysToSet 包含要設置的鍵和值的字典。
 * @return 如果值已成功設置，則返回 YES，否則返回 NO。
 */
- (BOOL)setValues:(NSDictionary<NSString *, NSString *> *)keysToSet;

/**
 * \~English
 * @brief Read multiple settings associated with given list of keys.
 *
 * Implemented in SettingsReadWrite.
 *
 * @param key The key for which to get the properties.
 * @return A Setting object containing the properties for the specified key.
 *
 * \~Chinese
 * @brief 读取与给定键列表关联的多个设置。
 *
 * 在 SettingsReadWrite 中实现。
 *
 * @param key 要获取属性的键。
 * @return 包含指定键属性的 Setting 对象。
 *
 * \~Chinese-Traditional
 * @brief 讀取與給定鍵列表關聯的多個設置。
 *
 * 在 SettingsReadWrite 中實現。
 *
 * @param key 要獲取屬性的鍵。
 * @return 包含指定鍵屬性的 Setting 對象。
 */
- (Setting *)getPropertiesForKey:(NSString *)key;

/**
 * \~English
 * @brief Read settings associated with given list of keys.
 *
 * Implemented in SettingsReadWrite.
 *
 * @param keys An array of keys for which to get the properties.
 * @return A dictionary containing the specified keys and their corresponding Setting objects.
 *
 * \~Chinese
 * @brief 读取与给定键列表关联的设置。
 *
 * 在 SettingsReadWrite 中实现。
 *
 * @param keys 要获取属性的键数组。
 * @return 包含指定键及其相应属性的字典。
 *
 * \~Chinese-Traditional
 * @brief 讀取與給定鍵列表關聯的設置。
 *
 * 在 SettingsReadWrite 中實現。
 *
 * @param keys 要獲取屬性的鍵數組。
 * @return 包含指定鍵及其相應屬性的字典。
 */
- (NSDictionary<NSString *, Setting *> *)getPropertiesForKeys:(NSArray<NSString *> *)keys;

/**
 * \~English
 * @brief Read all settings.
 *
 * Implemented in SettingsReadWrite.
 *
 * @return A dictionary containing all keys and their corresponding Setting objects.
 *
 * \~Chinese
 * @brief 读取所有设置。
 *
 * 在 SettingsReadWrite 中实现。
 *
 * @return 包含所有键及其相应属性的字典。
 *
 * \~Chinese-Traditional
 * @brief 讀取所有設置。
 *
 * 在 SettingsReadWrite 中實現。
 *
 * @return 包含所有鍵及其相應屬性的字典。
 */
- (NSDictionary<NSString *, Setting *> *)getAllProperties;

@end
