//
//  Label.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/16.
//

#ifndef Label_h
#define Label_h

#import <Foundation/Foundation.h>
#import <UniPRT/ILabel.h>
#import <UniPRT/TsplForm.h>
#import <UniPRT/Scale.h>
#import <UniPRT/Defaults.h>

/**
 * \~English
 * @class Label
 * @brief A class representing a label for TSPL.
 *
 * This class implements the ILabel protocol and provides functionality for initializing, adding objects and raw content, and describing labels in TSPL.
 *
 * @ingroup TSPL
 *
 * \~Chinese
 * @class Label
 * @brief 表示 TSPL 标签的类。
 *
 * 该类实现了 ILabel 协议，提供了在 TSPL 中初始化、添加对象和原始内容以及描述标签的功能。
 *
 * @ingroup TSPL
 *
 * \~Chinese-Traditional
 * @class Label
 * @brief 表示 TSPL 標籤的類。
 *
 * 該類實現了 ILabel 協議，提供了在 TSPL 中初始化、添加對象和原始內容以及描述標籤的功能。
 *
 * @ingroup TSPL
 */
@interface Label : NSObject <ILabel>

/**
 * \~English
 * @brief The form associated with the label.
 *
 * \~Chinese
 * @brief 与标签关联的格式。
 *
 * \~Chinese-Traditional
 * @brief 與標籤關聯的格式。
 */
@property (nonatomic, strong) TsplForm *form;

/**
 * \~English
 * @brief The scale associated with the label.
 *
 * \~Chinese
 * @brief 与标签关联的比例。
 *
 * \~Chinese-Traditional
 * @brief 與標籤關聯的比例。
 */
@property (nonatomic, strong) Scale *scale;

/**
 * \~English
 * @brief The name of the label.
 *
 * \~Chinese
 * @brief 标签的名称。
 *
 * \~Chinese-Traditional
 * @brief 標籤的名稱。
 */
@property (nonatomic, strong) NSString *name;

/**
 * \~English
 * @brief Initializes a new instance of the Label class with the specified name.
 *
 * @param name The name of the label.
 * @return A new instance of Label.
 *
 * \~Chinese
 * @brief 使用指定的名称初始化 Label 类的新实例。
 *
 * @param name 标签的名称。
 * @return Label 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的名稱初始化 Label 類的新實例。
 *
 * @param name 標籤的名稱。
 * @return Label 的新實例。
 */
- (instancetype)initWithName:(NSString *)name;

/**
 * \~English
 * @brief Adds an object that can be converted to printer language syntax to the label.
 *
 * @param addObject The object to be added to the label.
 *
 * \~Chinese
 * @brief 向标签添加一个可以转换为打印机语言语法的对象。
 *
 * @param addObject 要添加到标签的对象。
 *
 * \~Chinese-Traditional
 * @brief 向標籤添加一個可以轉換為打印機語言語法的對象。
 *
 * @param addObject 要添加到標籤的對象。
 */
- (void)addObject:(id)addObject;

/**
 * \~English
 * @brief Adds raw content to the label for cases where custom content needs to be added.
 *
 * Adding raw content allows the addition of printer language commands or other features that may not be available.
 * No formatting of the content is done; the caller is responsible for making sure the content follows printer language syntax.
 *
 * @param rawContent The raw content to be added to the label.
 *
 * \~Chinese
 * @brief 向标签添加原始内容，以便在需要添加自定义内容的情况下使用。
 *
 * 添加原始内容允许添加打印机语言命令或其他可能不可用的功能。
 * 不对内容进行任何格式化；调用者负责确保内容符合打印机语言语法。
 *
 * @param rawContent 要添加到标签的原始内容。
 *
 * \~Chinese-Traditional
 * @brief 向標籤添加原始內容，以便在需要添加自定義內容的情況下使用。
 *
 * 添加原始內容允許添加打印機語言命令或其他可能不可用的功能。
 * 不對內容進行任何格式化；調用者負責確保內容符合打印機語言語法。
 *
 * @param rawContent 要添加到標籤的原始內容。
 */
- (void)addRawContent:(NSString *)rawContent;

/**
 * \~English
 * @brief Returns a string that can be sent to the printer.
 *
 * This method generates a string representation of the Label instance that
 * is formatted and ready to be sent to a printer. The string includes all the
 * necessary commands and data required for printing in TSPL.
 *
 * @return A string description of the Label instance.
 *
 * \~Chinese
 * @brief 返回可以发送到打印机的字符串。
 *
 * 该方法生成 Label 实例的字符串表示形式，
 * 格式化并准备好发送到打印机。字符串包括在 TSPL 中打印所需的所有命令和数据。
 *
 * @return Label 实例的字符串描述。
 *
 * \~Chinese-Traditional
 * @brief 返回可以發送到打印機的字符串。
 *
 * 該方法生成 Label 實例的字符串表示形式，
 * 格式化並準備好發送到打印機。字符串包括在 TSPL 中打印所需的所有命令和數據。
 *
 * @return Label 實例的字符串描述。
 */
- (NSString *)description;

/**
 * \~English
 * @brief Sets the name of the label.
 *
 * @param name The new name for the label.
 *
 * \~Chinese
 * @brief 设置标签的名称。
 *
 * @param name 标签的新名称。
 *
 * \~Chinese-Traditional
 * @brief 設置標籤的名稱。
 *
 * @param name 標籤的新名稱。
 */
- (void)SetName:(NSString *)name;

@end

#endif /* Label_h */
