//
//  PglLabel.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/24.
//

#ifndef PglLabel_h
#define PglLabel_h

#import <Foundation/Foundation.h>
#import <UniPRT/PglForm.h>
#import <UniPRT/PglScale.h>

/**
 * \~English
 * @class PglLabel
 * @brief A class representing a label that can be converted to printer language syntax.
 *
 * This class provides functionality for managing the form, scale, and name of a label, as well as adding objects and raw content to the label.
 *
 * @ingroup PGL
 *
 * \~Chinese
 * @class PglLabel
 * @brief 表示一个可以转换为打印机语言语法的标签类。
 *
 * 该类提供了管理标签的格式、比例和名称的功能，以及向标签添加对象和原始内容的功能。
 *
 * @ingroup PGL
 *
 * \~Chinese-Traditional
 * @class PglLabel
 * @brief 表示一個可以轉換為打印機語言語法的標籤類。
 *
 * 該類提供了管理標籤的格式、比例和名稱的功能，以及向標籤添加對象和原始內容的功能。
 *
 * @ingroup PGL
 */
@interface PglLabel : NSObject

/**
 * \~English
 * @brief The form associated with the label.
 *
 * \~Chinese
 * @brief 与标签相关联的格式。
 *
 * \~Chinese-Traditional
 * @brief 與標籤相關聯的格式。
 */
@property (nonatomic, strong) PglForm *form;

/**
 * \~English
 * @brief The scale associated with the label.
 *
 * \~Chinese
 * @brief 与标签相关联的比例。
 *
 * \~Chinese-Traditional
 * @brief 與標籤相關聯的比例。
 */
@property (nonatomic, strong) PglScale *scale;

/**
 * \~English
 * @brief The name of the label.
 *
 * \~Chinese
 * @brief 标签的名称。
 *
 * \~Chinese-Traditional
 * @brief 標籤的名稱。
 */
@property (nonatomic, copy) NSString *name;

/**
 * \~English
 * @brief Initializes a new instance of the PglLabel class with the specified name.
 *
 * @param name The name of the label.
 * @return A new instance of PglLabel.
 *
 * \~Chinese
 * @brief 使用指定的名称初始化 PglLabel 类的新实例。
 *
 * @param name 标签的名称。
 * @return PglLabel 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的名稱初始化 PglLabel 類的新實例。
 *
 * @param name 標籤的名稱。
 * @return PglLabel 的新實例。
 */
- (instancetype)initWithName:(NSString *)name;

/**
 * \~English
 * @brief Adds an object that can be converted to printer language syntax to the label.
 *
 * @param addObject The object to be added to the label.
 *
 * \~Chinese
 * @brief 向标签添加可以转换为打印机语言语法的对象。
 *
 * @param addObject 要添加到标签的对象。
 *
 * \~Chinese-Traditional
 * @brief 向標籤添加可以轉換為打印機語言語法的對象。
 *
 * @param addObject 要添加到標籤的對象。
 */
- (void)addObject:(id)addObject;

/**
 * \~English
 * @brief Adds raw content to the label for cases where custom content needs to be added.
 *
 * Adding raw content allows the addition of printer language commands or other features that may not be available.
 * No formatting of the content is done; the caller is responsible for making sure the content follows printer language syntax.
 *
 * @param rawContent The raw content to be added to the label.
 *
 * \~Chinese
 * @brief 向标签添加原始内容，用于需要添加自定义内容的情况。
 *
 * 添加原始内容允许添加打印机语言命令或其他可能不可用的功能。
 * 不会对内容进行格式化；调用者负责确保内容遵循打印机语言语法。
 *
 * @param rawContent 要添加到标签的原始内容。
 *
 * \~Chinese-Traditional
 * @brief 向標籤添加原始內容，用於需要添加自定義內容的情況。
 *
 * 添加原始內容允許添加打印機語言命令或其他可能不可用的功能。
 * 不會對內容進行格式化；調用者負責確保內容遵循打印機語言語法。
 *
 * @param rawContent 要添加到標籤的原始內容。
 */
- (void)addRawContent:(NSString *)rawContent;

/**
 * \~English
 * @brief Returns a string that can be sent to the printer.
 *
 * @return A string description of the PglLabel instance.
 *
 * \~Chinese
 * @brief 返回可以发送到打印机的字符串。
 *
 * @return PglLabel 实例的字符串描述。
 *
 * \~Chinese-Traditional
 * @brief 返回可以發送到打印機的字符串。
 *
 * @return PglLabel 實例的字符串描述。
 */
- (NSString *)description;

@end

#endif /* PglLabel_h */
