﻿#include <cstdio>
#include <iostream>
#include <cstring>
#include <vector>
#include <string>
#include <dlfcn.h> // 包含動態加載庫的頭文件
#include <sstream>
#include <thread>
#include <chrono>
#include <mutex>



#define TSC_USB_VID 0x1203
#define PTX_USB_VID 0x14ae

// 定義 Tuple 的結構
struct Tuple_c {
    short vendorId;
    short productId;
};

enum DescriptorPortType {

    // Data
    DATA,
    // Management
    MGMT,
    // Status
    STATUS
};

enum class Brand {
    TSC = 1,
    PTX
};

typedef unsigned char byte;

void* handleLib;
Tuple_c* devices = nullptr;

typedef int (*BytesAvailableFunc)();
typedef bool (*ConnectedFunc)();
typedef void (*ReadFunc)(byte* buffer, int size);
typedef void (*WriteFunc)(byte* dataOut, int iDataSize);
typedef byte* (*WriteAndWaitForResponseFunc)(byte* dataOut, int iDataSize, int responseStartTimeOut, int responseEndTimeOut, char* completetionToken);
typedef void (*CloseFunc)();
typedef void (*OpenFunc)();
typedef int (*GetAvailableDevicesFunc)(Tuple_c**, int*);
typedef void (*UsbConnectFunc)(Tuple_c device);
typedef void (*TcpConnectFunc)(char* ipAddress, int port);
typedef void (*BtConnectFunc)(uint64_t macAddress);
typedef void (*ComConnectFunc)(std::string ComPort, int BaudRate);






GetAvailableDevicesFunc pGetAvailableDevices = NULL;
BytesAvailableFunc pBytesAvailable = NULL;
ConnectedFunc pConnected = NULL;
ReadFunc pRead = NULL;
WriteFunc pWrite = NULL;
WriteAndWaitForResponseFunc pWriteAndWaitForResponse = NULL;
CloseFunc pClose = NULL;
OpenFunc pOpen = NULL;
UsbConnectFunc pUsbConnect = NULL;
TcpConnectFunc pTcpConnect = NULL;
BtConnectFunc pBtConnect = NULL;
ComConnectFunc pComConnect = NULL;



bool _bAsyncListening = false;
bool _bConnected = false;
std::mutex _mutex;
std::thread listenerThread_;
std::string currentText;



void loadListOfUsbPrinters() {
    
    int count = 0;

    pGetAvailableDevices(&devices, &count);
    std::cout << "Number of devices: " << count << std::endl;

    if (count > 0) {
        bool fDevFound = false;
        for (int i = 0; i < count; i++) {
            std::cout << "Device " << i << ": VendorID= 0x" << std::hex << devices[i].vendorId
                << " ProductID= 0x" << devices[i].productId << std::endl;

            std::string usbItemDesc;
            if (TSC_USB_VID == devices[i].vendorId) {
                usbItemDesc = "TSC";
                fDevFound = true;
            }
            else if (PTX_USB_VID == devices[i].vendorId) {
                usbItemDesc = "Printronix";
                fDevFound = true;
            }

            
            if (fDevFound) {
                std::ostringstream oss;
                oss << std::hex << devices[i].vendorId;
                std::string vendorIdHex = oss.str();
                oss.str("");  // 清除 oss 的內容以供重用
                oss << std::hex << devices[i].productId;
                std::string productIdHex = oss.str();

                usbItemDesc += ", 0x" + vendorIdHex + ", 0x" + productIdHex;
                std::cout << usbItemDesc << std::endl;
            }
        }
    }

    
    
}


void ListenerAsync() 
{
    while (_bAsyncListening && pConnected()) {
        std::this_thread::sleep_for(std::chrono::milliseconds(500));

        if (!_bAsyncListening)
            break;

        std::lock_guard<std::mutex> lock(_mutex);

        
        int iBytes = pBytesAvailable();
        byte* pBuf = nullptr; 

        if (!_bAsyncListening)
            break;

        if (iBytes > 0) {
           
            pBuf = new byte[iBytes];
            pRead(pBuf, iBytes);

            if (!_bAsyncListening)
                break;

            std::string str(reinterpret_cast<char*>(pBuf), iBytes);
            currentText += str;
            std::cout << std::endl;
            std::cout << std::endl;
            std::cout << "Response:";
            std::cout << currentText << std::endl;

            delete[] pBuf;
        }
        else {
            std::this_thread::sleep_for(std::chrono::milliseconds(10));
        }
    }

}




void RunAsyncListen() 
{

    if (_bAsyncListening) {
        std::cout << "---Deactivate Listener" << std::endl;
        _bAsyncListening = false;
        if (listenerThread_.joinable()) {
            listenerThread_.join();
        }
    }
    else {
        if (!pConnected()) {
            std::cerr << "Connect first. Can't listen without a valid open/active connection." << std::endl;
            return;
        }
        std::cout << "Listening....." << std::endl;
        _bAsyncListening = true;
        listenerThread_ = std::thread(ListenerAsync);
        listenerThread_.detach();
        std::this_thread::sleep_for(std::chrono::seconds(1));
    }

}



int main()
{
    //printf("來自 %s 的問候!\n", "TestCommSDK");

    // 加載 libCommSDK.so 庫
    handleLib = dlopen("libCommSDK.so", RTLD_LAZY);
    if (!handleLib) {
        std::cerr << "Failed to load libCommSDK.so: " << dlerror() << std::endl;
        return 1;
    }

    pGetAvailableDevices = reinterpret_cast<GetAvailableDevicesFunc>(dlsym(handleLib, "GetAvailableDevices"));
    pBytesAvailable = reinterpret_cast<BytesAvailableFunc>(dlsym(handleLib, "BytesAvailable"));
    pConnected = reinterpret_cast<ConnectedFunc>(dlsym(handleLib, "Connected"));
    pRead = reinterpret_cast<ReadFunc>(dlsym(handleLib, "Read"));
    pWrite = reinterpret_cast<WriteFunc>(dlsym(handleLib, "Write"));
    pWriteAndWaitForResponse = reinterpret_cast<WriteAndWaitForResponseFunc>(dlsym(handleLib, "WriteAndWaitForResponse"));
    pClose = reinterpret_cast<CloseFunc>(dlsym(handleLib, "Close"));
    pOpen = reinterpret_cast<OpenFunc>(dlsym(handleLib, "Open"));
    pUsbConnect = reinterpret_cast<UsbConnectFunc>(dlsym(handleLib, "UsbConnect"));
    pTcpConnect = reinterpret_cast<TcpConnectFunc>(dlsym(handleLib, "TcpConnect"));
    pBtConnect = reinterpret_cast<BtConnectFunc>(dlsym(handleLib, "BtConnect"));
    pComConnect = reinterpret_cast<ComConnectFunc>(dlsym(handleLib, "ComConnect"));
   
    


    if ( !pGetAvailableDevices || !pBytesAvailable || !pConnected || !pRead || !pWrite || 
        !pWriteAndWaitForResponse || !pOpen || !pUsbConnect || !pTcpConnect || !pBtConnect || 
        !pComConnect)
    {
        std::cerr << "Failed to get address of libCommSDK.so API functions: " << dlerror() << std::endl;
        dlclose(handleLib);
        return 1;
    }



    loadListOfUsbPrinters();


    // USB
    if (devices != nullptr)
        pUsbConnect(devices[0]);
    else
    {
        std::cerr << "Failed to call UsbConnect API: " << dlerror() << std::endl;
        dlclose(handleLib);
        return 1;
    }
    
    
    
    
    // NET
    //pTcpConnect("192.168.1.191", 9100);
    //pTcpConnect("fe80::208:96ff:fe40:9b04%ens33", 9100);  // Link Local IPv6 Address
    //pTcpConnect("2001:b030:2219:c40:208:96ff:fe40:9b04", 9100); // Global Unicast IPv6 Address
    // BT
    //pBtConnect(0x44B7D02E6EB7);
    // Serial Port
    //pComConnect("/dev/ttyUSB0", 9600);



    pOpen();

    if (pConnected())
    {
        std::string content;

        
        // Print Job
        content += "SIZE 3,2\r\n";
        content += "GAP 0 mm, 0 mm\r\n";
        content += "DIRECTION 1\r\n";
        content += "CLS\r\n";
        content += "TEXT 10, 30, \"3\", 0, 1, 1, \"123456\"\r\n";
        content += "BARCODE 10, 100, \"EAN13\", 80, 1, 0, 2, 4, \"123456789012\"\r\n";
        content += "TEXT 10, 70, \"4\", 0, 1, 1, \"TEST PRINTOUT\"\r\n";
        content += "PRINT 1, 1\r\n";
        
        
        
        /*
        content += "!PTX_SETUP\r\n";
        content += "PRINTJOB - START; 1234\r\n";
        content += "PTX_END\r\n";

        content += "~NORMAL\r\n";

        content += "~CREATE; PTX; 432\r\n";
        content += "ALPHA\r\n";
        content += "5.7; 8; 1; 1; *PRINTRONIX*\r\n";
        content += "STOP\r\n";
        content += "BARCODE\r\n";
        content += "DATAMATRIX; 2; 10\r\n";
        content += "*PRINTRONIX*\r\n";
        content += "STOP\r\n";
        content += "END\r\n";
        content += "~EXECUTE; PTX; 1\r\n";

        content += "~NORMAL\r\n";

        content += "!PTX_SETUP\r\n";
        content += "PRINTJOB - END; 1234\r\n";
        content += "PTX_END\r\n";
        // Print Job
        */


       
        /*
        //AsyncListen
        RunAsyncListen();
        content += "A$=\"ABCDEFGHIJKLMNOPQRSTUVWXYZ\"\r\n";
        content += "OUT A$\r\n";
        //AsyncListen
        */
        

        
        std::cout << std::endl;
        std::cout << "Send to Printer:" << std::endl;
        std::cout << content;
        const char* pStr = content.c_str();
        pWrite((byte*)pStr, strlen(pStr));
        


        /*
        // WriteAndWaitForResponse
        content += "FEED 100\r\n";
        content += "OUT \"12345678\"\r\n";
        std::cout << std::endl;
        std::cout << "Send to Printer:" << std::endl;
        std::cout << content;
        const char* pStr = content.c_str();
        byte* pResponse = pWriteAndWaitForResponse((byte*)pStr, content.size(), 1000, 500, "\r\n");
        std::string str(reinterpret_cast<char*>(pResponse), strlen(reinterpret_cast<char*>(pResponse)));
        std::cout << std::endl;
        std::cout << std::endl;
        std::cout << "Response:";
        std::cout << str << std::endl;
        // WriteAndWaitForResponse
        */
       

    }
    else
    {
        std::cerr << "Failed to Connect Printer " << dlerror() << std::endl;
        dlclose(handleLib);
        return 1;
    }

    if (devices != nullptr)
        delete[] devices;



    int userInput;

    // 循環等待用戶輸入，直到用戶輸入0為止
    do {

        std::cout << std::endl;
        std::cout << std::endl;
        std::cout << "Press 0 to exit";
        std::cout << std::endl;
        std::cin >> userInput;
    } while (userInput != 0);

    // 使用者輸入0後程式結束
    std::cout << std::endl;
    std::cout << "Program exit" << std::endl;
    std::cout << std::endl;

    _bAsyncListening = false;

    pClose();
    // 釋放 libCommSDK.so 庫
    dlclose(handleLib);

    return 0;
}