﻿#include <cstdio>
#include <iostream>
#include <string>
#include <sstream>
#include <string>
#include <cstring>
#include <iomanip>
#include <fstream>
#include <vector>
#include <thread>
#include <mutex>
#include <atomic>
#include <chrono>
#include <condition_variable>
#include <dlfcn.h> // 包含動態加載庫的頭文件

#include "RfidReport.h"


enum COMM_TYP
{
	USB_COMM = 0,
	TCP_COMM,
};

// 定義 Tuple 的結構
//#pragma pack(1)
struct Tuple_c {
	short vendorId;
	short productId;
};
//#pragma pack()

// TCP
enum DescriptorPortType {

	// Data
	DATA,
	// Management
	MGMT,
	// Status
	STATUS
};


struct PrinterInfo {
	std::string Model;
	std::string SerialNumber;
	std::string FirmwarePartNumber;
	std::string FirmwareVersion;
	std::string PrintheadResolution;
	bool HasRfidOption;
	bool HasOdvOption;
};


enum INFO_TYP
{
	ODV_TYP = 0,
	RFID_TYP,

};


using byte = unsigned char;

// CommSDK. dll export
typedef int (*BytesAvailableFunc)();
typedef bool (*ConnectedFunc)();
typedef void (*ReadFunc)(byte* buffer, int size);
typedef void (*WriteFunc)(byte* dataOut, int iDataSize);
typedef byte* (*WriteAndWaitForResponseFunc)(byte* dataOut, int iDataSize, int responseStartTimeOut, int responseEndTimeOut, char* completetionToken);
typedef void (*CloseFunc)();
typedef void (*OpenFunc)();
typedef int (*GetAvailableDevicesFunc)(Tuple_c**, int*);
typedef void (*UsbConnectFunc)(Tuple_c device);
typedef void (*TcpConnectFunc)(char* ipAddress, int port);
typedef void (*BtConnectFunc)(uint64_t macAddress);
typedef void (*GetCommFunc)(void*& ptrComm); // for json


// Tool APIs
typedef void (*SendPrintFileFunc)(char* ipAddress, char* fileName);


// JsonSDK. dll export
typedef void (*MessengerGetFunc)(void* ptrComm, COMM_TYP commTyp, int maxInputMsgCapacity, bool usingDataPort);
typedef void (*MessengerReleaseFunc)();
typedef void (*MessengerSendMsgFunc)(char* strCommand, char* strContent);
typedef void (*MessengerReadNextMsgFunc)(const char** strNextMsg);
typedef void (*MessengerSendMsgAndWaitForResponseFunc)(char* strCommand, char* strContent, int maxWaitTimeSecs, const char** strResponse);
typedef int (*MessengerUnreadMsgCountFunc)();
typedef void (*RfidMonitorConnectionFunc)(char* lpAddress);
typedef void (*PrinterMonitorConnectionFunc)(char* lpAddress, INFO_TYP InfoTyp);
typedef void (*GetPrinterInfoFunc)(PrinterInfo* pPrtInfo, INFO_TYP InfoTyp);
typedef bool (*GetRfidReportListeningFunc)();
typedef void (*SetRfidReportListeningFunc)(bool value);
typedef void (*RfidMonitorDisposeFunc)();
typedef void (*PrinterMonitorDisposeFunc)(INFO_TYP InfoTyp);
typedef void (*SetRfidReportCallbackFunc)(void callback(const RfidReport& report));



GetAvailableDevicesFunc pGetAvailableDevices = NULL;
BytesAvailableFunc pBytesAvailable = NULL;
ConnectedFunc pConnected = NULL;
ReadFunc pRead = NULL;
WriteFunc pWrite = NULL;
WriteAndWaitForResponseFunc pWriteAndWaitForResponse = NULL;
CloseFunc pClose = NULL;
OpenFunc pOpen = NULL;
UsbConnectFunc pUsbConnect = NULL;
TcpConnectFunc pTcpConnect = NULL;
BtConnectFunc pBtConnect = NULL;
GetCommFunc pGetComm = NULL;
SendPrintFileFunc pSendPrintFile = NULL;


MessengerGetFunc pMessengerGet = NULL;
MessengerReleaseFunc pMessengerRelease = NULL;
MessengerSendMsgFunc pMessengerSendMsg = NULL;
MessengerReadNextMsgFunc pMessengerReadNextMsg = NULL;
MessengerSendMsgAndWaitForResponseFunc pMessengerSendMsgAndWaitForResponse = NULL;
MessengerUnreadMsgCountFunc pMessengerUnreadMsgCount = NULL;


RfidMonitorConnectionFunc pRfidMonitorConnection = NULL;
PrinterMonitorConnectionFunc pPrinterMonitorConnection = NULL;
GetPrinterInfoFunc pGetPrinterInfo = NULL;
GetRfidReportListeningFunc pGetRfidReportListening = NULL;
SetRfidReportListeningFunc pSetRfidReportListening = NULL;
RfidMonitorDisposeFunc pRfidMonitorDispose = NULL;
PrinterMonitorDisposeFunc pPrinterMonitorDispose = NULL;
SetRfidReportCallbackFunc pSetRfidReportCallback = NULL;


void* handleCommSDKLib;
void* handleJsonSDKLib;

void ShowPrinterInfo(PrinterInfo* pPrtInfo, INFO_TYP InfoTyp)
{
	pGetPrinterInfo(pPrtInfo, InfoTyp);

	std::cout << std::endl;
	std::cout << "Printer Model: " << pPrtInfo->Model << std::endl;
	std::cout << "Printer SN: " << pPrtInfo->SerialNumber << std::endl;
	std::cout << "Printer FW PN: " << pPrtInfo->FirmwarePartNumber << std::endl;
	std::cout << "Printer FW Ver: " << pPrtInfo->FirmwareVersion << std::endl;
	std::cout << "Printhead Resolution (Dots/Inch): " << pPrtInfo->PrintheadResolution << std::endl;
	std::cout << std::endl;
	std::cout << "Has RFID: " << (pPrtInfo->HasRfidOption ? "yes" : "no") << std::endl;
	

}


void RfidReportCallback(const RfidReport& report)
{
	if (report.Failed())
	{
		std::cout << "\nRFID Failed." << std::endl;
	}
	else
	{
		std::string memoryType = "";
		switch (report.DataType())
		{
		case RfidReport::RfidDataType::USR:
			memoryType = "USR";
			break;
		case RfidReport::RfidDataType::TID:
			memoryType = "TID";
			break;
		case RfidReport::RfidDataType::UNKNOWN:
			memoryType = "UNKNOWN";
			break;
		}

		std::cout << "\nRFID Passed." << std::endl;
		std::cout << "Write Action: " << ((report.IsWriteOperation()) ? "yes" : "no") << std::endl;
		std::cout << "Operation on EPC Memory: " << ((report.DataType() == RfidReport::RfidDataType::EPC) ? "yes" : "no") << std::endl;
		if (report.DataType() != RfidReport::RfidDataType::EPC)
		{
			std::cout << "  memory accessed: " << memoryType << std::endl;
		}

		std::cout << "Data: \n" << report.Data() << std::endl;
	}
}

int main()
{
	// 加載 libCommSDK.so 庫
	handleCommSDKLib = dlopen("libCommSDK.so", RTLD_LAZY);
	if (!handleCommSDKLib) {
		std::cerr << "Failed to load libCommSDK.so: " << dlerror() << std::endl;
		return 1;
	}

	// 加載 libJsonSDK.so 庫
	handleJsonSDKLib = dlopen("libJsonSDK.so", RTLD_LAZY);
	if (!handleJsonSDKLib) {
		std::cerr << "Failed to load libCommSDK.so: " << dlerror() << std::endl;
		return 1;
	}


	pGetAvailableDevices = reinterpret_cast<GetAvailableDevicesFunc>(dlsym(handleCommSDKLib, "GetAvailableDevices"));
	pBytesAvailable = reinterpret_cast<BytesAvailableFunc>(dlsym(handleCommSDKLib, "BytesAvailable"));
	pConnected = reinterpret_cast<ConnectedFunc>(dlsym(handleCommSDKLib, "Connected"));
	pRead = reinterpret_cast<ReadFunc>(dlsym(handleCommSDKLib, "Read"));
	pWrite = reinterpret_cast<WriteFunc>(dlsym(handleCommSDKLib, "Write"));
	pWriteAndWaitForResponse = reinterpret_cast<WriteAndWaitForResponseFunc>(dlsym(handleCommSDKLib, "WriteAndWaitForResponse"));
	pClose = reinterpret_cast<CloseFunc>(dlsym(handleCommSDKLib, "Close"));
	pOpen = reinterpret_cast<OpenFunc>(dlsym(handleCommSDKLib, "Open"));
	pUsbConnect = reinterpret_cast<UsbConnectFunc>(dlsym(handleCommSDKLib, "UsbConnect"));
	pTcpConnect = reinterpret_cast<TcpConnectFunc>(dlsym(handleCommSDKLib, "TcpConnect"));
	pBtConnect = reinterpret_cast<BtConnectFunc>(dlsym(handleCommSDKLib, "BtConnect"));
	pGetComm = reinterpret_cast<GetCommFunc>(dlsym(handleCommSDKLib, "GetComm"));
	pSendPrintFile = reinterpret_cast<SendPrintFileFunc>(dlsym(handleCommSDKLib, "SendPrintFile"));

	if (!pGetAvailableDevices || !pBytesAvailable || !pConnected || !pRead || !pWrite || 
		!pWriteAndWaitForResponse || !pOpen || !pUsbConnect || !pTcpConnect || !pBtConnect ||
		!pGetComm || !pSendPrintFile)
	{
		std::cerr << "Failed to get address of libCommSDK.so API functions: " << dlerror() << std::endl;
		dlclose(handleCommSDKLib);
		return 1;
	}


	pMessengerGet = reinterpret_cast<MessengerGetFunc>(dlsym(handleJsonSDKLib, "MessengerGet"));
	pMessengerRelease = reinterpret_cast<MessengerReleaseFunc>(dlsym(handleJsonSDKLib, "MessengerRelease"));
	pMessengerSendMsg = reinterpret_cast<MessengerSendMsgFunc>(dlsym(handleJsonSDKLib, "MessengerSendMsg"));
	pMessengerReadNextMsg = reinterpret_cast<MessengerReadNextMsgFunc>(dlsym(handleJsonSDKLib, "MessengerReadNextMsg"));
	pMessengerSendMsgAndWaitForResponse = reinterpret_cast<MessengerSendMsgAndWaitForResponseFunc>(dlsym(handleJsonSDKLib, "MessengerSendMsgAndWaitForResponse"));
	pMessengerUnreadMsgCount = reinterpret_cast<MessengerUnreadMsgCountFunc>(dlsym(handleJsonSDKLib, "MessengerUnreadMsgCount"));

	pRfidMonitorConnection = reinterpret_cast<RfidMonitorConnectionFunc>(dlsym(handleJsonSDKLib, "RfidMonitorConnection"));
	pPrinterMonitorConnection = reinterpret_cast<PrinterMonitorConnectionFunc>(dlsym(handleJsonSDKLib, "PrinterMonitorConnection"));
	pGetPrinterInfo = reinterpret_cast<GetPrinterInfoFunc>(dlsym(handleJsonSDKLib, "GetPrinterInfo"));
	pGetRfidReportListening = reinterpret_cast<GetRfidReportListeningFunc>(dlsym(handleJsonSDKLib, "GetRfidReportListening"));
	pSetRfidReportListening = reinterpret_cast<SetRfidReportListeningFunc>(dlsym(handleJsonSDKLib, "SetRfidReportListening"));
	pRfidMonitorDispose = reinterpret_cast<RfidMonitorDisposeFunc>(dlsym(handleJsonSDKLib, "RfidMonitorDispose"));
    pPrinterMonitorDispose = reinterpret_cast<PrinterMonitorDisposeFunc>(dlsym(handleJsonSDKLib, "PrinterMonitorDispose"));
	pSetRfidReportCallback = reinterpret_cast<SetRfidReportCallbackFunc>(dlsym(handleJsonSDKLib, "SetRfidReportCallback"));



	if (!pMessengerGet || !pMessengerRelease || !pMessengerSendMsg || !pMessengerReadNextMsg || !pMessengerSendMsgAndWaitForResponse || !pMessengerUnreadMsgCount ||
		!pRfidMonitorConnection || !pPrinterMonitorConnection || !pGetPrinterInfo || !pGetRfidReportListening || !pSetRfidReportListening || !pRfidMonitorDispose ||
		!pPrinterMonitorDispose || !pSetRfidReportCallback)
	{
		std::cerr << "Failed to get address of libJsonSDK.so API functions: " << dlerror() << std::endl;
		dlclose(handleJsonSDKLib);
		return 1;
	}

	
	PrinterInfo PrtInfo;
	std::string ptrIpRfid = "10.0.10.170";

	pRfidMonitorConnection((char*)ptrIpRfid.c_str());
	pPrinterMonitorConnection((char*)ptrIpRfid.c_str(), INFO_TYP::RFID_TYP);

	std::cout << "\n=======Rfid Printer Info: ";
	ShowPrinterInfo(&PrtInfo, INFO_TYP::RFID_TYP);


	if (PrtInfo.HasRfidOption == false)
	{
		std::cout << "WARNING: Missing RFID option on printer at: " << ptrIpRfid << std::endl;
		return 1;
	}


	// ============================
	// -------------  RFID Printer: setup for RFID report monitoring
	// ============================
	pSetRfidReportListening(true);
	pSetRfidReportCallback(RfidReportCallback);


	pSendPrintFile((char*)(ptrIpRfid.c_str()), (char*)"rfid.pgl");
	std::cout << "Sending RFID print job..." << std::endl;


	int userInput;

	// 循環等待用戶輸入，直到用戶輸入0為止
	do {

		std::cout << std::endl;
		std::cout << std::endl;
		std::cout << "Press 0 to exit";
		std::cout << std::endl;
		std::cin >> userInput;
	} while (userInput != 0);

	// 使用者輸入0後程式結束
	std::cout << std::endl;
	std::cout << "Program exit" << std::endl;
	std::cout << std::endl;

	pClose();
	pRfidMonitorDispose();
	pPrinterMonitorDispose(INFO_TYP::RFID_TYP);


	// 釋放 libCommSDK.so / libJsonSDK.so 庫
	dlclose(handleCommSDKLib);
	dlclose(handleJsonSDKLib);
    return 0;

}



