#pragma once
#include <cstdlib> // for malloc
#include <cstdint> // for uint64_t
#include <string>
#include <cstring>

#include <map>
#include <unordered_map>


enum COMM_TYP
{
    _USB_COMM = 0,
    _TCP_COMM,
    
};


struct stPrinterInfo {
    std::string Model;
    std::string SerialNumber;
    std::string FirmwarePartNumber;
    std::string FirmwareVersion;
    std::string PrintheadResolution;
    bool HasRfidOption;
    bool HasOdvOption;
};


enum INFO_TYP
{
    ODV_TYP = 0,
    RFID_TYP,
    PRINTER_TYP,
};


enum RfidDataType {
    EPC,
    USR,
    TID,
    ACS,
    UNKNOWN
};



class RfidReport
{
public:
    

    bool Failed() const;

    std::string Data() const;

    RfidDataType DataType() const;

    bool IsWriteOperation() const;

    const std::unordered_map<std::string, std::string>& RawReport() const {
        return _rawReport;
    }

    void SetRawReport(const std::unordered_map<std::string, std::string>& rawReport) {
        _rawReport = rawReport;
    }

private:
    std::unordered_map<std::string, std::string> _rawReport;
};


class OdvReport
{
public:
    bool Failed() const;

    std::string Data() const;

    std::string OverallGrade() const;

    std::string OverallGradeLetter() const;

    float OverallGradeAsFloat() const;

    std::string Symbology() const;

    const std::unordered_map<std::string, std::string>& RawReport() const {
        return rawReport;
    }

    void SetRawReport(const std::unordered_map<std::string, std::string>& report) {
        rawReport = report;
    }

    std::string base64Decode(const std::string& encoded_string) const;

private:
    std::unordered_map<std::string, std::string> rawReport;

    std::string DecodeBase64(const std::string& base64) const;
};



// Define the function prototypes
extern "C" {

    void MessengerGet(void* ptrComm, COMM_TYP commTyp, int maxInputMsgCapacity, bool usingDataPort);
    void MessengerRelease();
    void MessengerSendMsg(char* strCommand, char* strContent);
    void MessengerReadNextMsg(const char** strNextMsg);
    void MessengerSendMsgAndWaitForResponse(char* strCommand, char* strContent, int maxWaitTimeSecs, const char** strResponse);
    int MessengerUnreadMsgCount();
    void RfidMonitorConnection(char* lpAddress);
    void OdvMonitorConnection(char* lpAddress);
    void PrinterMonitorConnection(char* lpAddress, INFO_TYP InfoTyp);
    void GetPrinterInfo(stPrinterInfo* pPrtInfo, INFO_TYP InfoTyp);
    bool GetRfidReportListening();
    void SetRfidReportListening(bool value);
    bool GetOdvReportListening();
    void SetOdvReportListening(bool value);
    void RfidMonitorDispose();
    void OdvMonitorDispose();
    void PrinterMonitorDispose(INFO_TYP InfoTyp);
    void SetRfidReportCallback(void callback(const RfidReport& report));
    void SetOdvReportCallback(void callback(const OdvReport& report));
    void SetEngineStatusListening(bool value);
    void SetDisplayStatusListening(bool value);
    void SetAlertStatusListening(bool value);
    bool GetEngineStatusListening();
    bool GetDisplayStatusListening();
    bool GetAlertStatusListening();
    void SetEngineStatusCallback(void callback(const std::string& engineState));
    void SetDisplayStatusCallback(void callback(const std::vector<std::string>& newDisplayText));
    void SetAlertStatusCallback(void callback(const std::vector<std::string>& alert));
    char* GetPrinterAllValues(void* ptrComm, COMM_TYP commTyp, bool usingDataPort);
    char* GetPrinterValue(void* ptrComm, COMM_TYP commTyp, bool usingDataPort, const char* key);
    char* GetPrinterValues(void* ptrComm, COMM_TYP commTyp, bool usingDataPort, const std::vector<std::string>& keys);
    bool SetPrinterValue(void* ptrComm, COMM_TYP commTyp, bool usingDataPort, const char* key, const char* value);
    bool SetPrinterValues(void* ptrComm, COMM_TYP commTyp, bool usingDataPort, std::map<std::string, std::string>* keysToSet);
    char* GetPrinterProperties(void* ptrComm, COMM_TYP commTyp, bool usingDataPort, const char* key);
    char* GetPrinterPropertiesEx(void* ptrComm, COMM_TYP commTyp, bool usingDataPort, const char* keys[], size_t count);
    char* GetPrinterAllProperties(void* ptrComm, COMM_TYP commTyp, bool usingDataPort);
    void FreeString(char* str);
    char* GetPrinterConfig(void* ptrComm, COMM_TYP commTyp, bool usingDataPort, int CfgNumber);
    bool SetPrinterConfig(void* ptrComm, COMM_TYP commTyp, bool usingDataPort, char* pConfig);
    char* GetPrinterAllConfig(void* ptrComm, COMM_TYP commTyp, bool usingDataPort);






}

