import UniPRT

func PictureTest() -> Label {
    let lbl = Label(name: "PictureTest")
    
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: ScaleEnum.inch))
    
    let picture = TsplPicture(
        start: Points(x: 0.25, y: 0.25),
        imageName: "TMP"
    )
    lbl!.add(picture)
    
    return lbl!
}
 
func BcdPdf417() -> Label {
    let lbl = Label(name: "Pdf417Bcodes")
    
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: ScaleEnum.inch))
    
    let someText = "The happiness in your life depends on the quality of your thoughts. --Marcus Aurelius"
    let someShortText = "PI = 3.1415"
    
    let bcdDefault = Pdf417Barcode(start: Points(x: 0.25, y: 0.50), data: someText)
    bcdDefault!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    
    let bcdErrCorrectionLvl0 = Pdf417Barcode(start: Points(x: 0.25, y: 1.50), data: someShortText)
    bcdErrCorrectionLvl0!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdErrCorrectionLvl0!.errorCorrection = .level0
    
    let bcdErrCorrectionLvl5 = Pdf417Barcode(start: Points(x: 0.25, y: 2.00), data: someShortText)
    bcdErrCorrectionLvl5!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdErrCorrectionLvl5!.errorCorrection = .level5
    
    let bcdRowsLimited = Pdf417Barcode(start: Points(x: 0.25, y: 3.00), data: someShortText)
    bcdRowsLimited!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdRowsLimited!.rows = 15
    
    let bcdColsLimited = Pdf417Barcode(start: Points(x: 0.25, y: 4.00), data: someShortText)
    bcdColsLimited!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdColsLimited!.columns = 5
    
    lbl!.add(bcdDefault)
    lbl!.add(bcdErrCorrectionLvl0)
    lbl!.add(bcdErrCorrectionLvl5)
    lbl!.add(bcdRowsLimited)
    lbl!.add(bcdColsLimited)
    
    return lbl!
}

func SimpleTextLabel(name: String, address: String) -> Label {
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 600, unit: .inch))
    
    let lbl = Label(name: "SimpleLabel")
    
    let inchRuler = Ruler(scale: .inch)
    
    let line1 = Line(start: Points(x: 2.5, y: 1.0 / 16.0), end: Points(x: 2.5, y: 1.0), lineThickness: 1.0 / 32.0)
    line1!.ruler = inchRuler
    lbl!.add(line1)
    
    let line2 = Line(start: Points(x: 0.12, y: 1.0), end: Points(x: 3.88, y: 1.0), lineThickness: 1.0 / 32.0)
    line2!.ruler = inchRuler
    lbl!.add(line2)

    let line3 = Line(start: Points(x: 0.12, y: 3.5), end: Points(x: 3.88, y: 3.5), lineThickness: 1.0 / 32.0)
    line3!.ruler = inchRuler
    lbl!.add(line3)
    
    let line4 = Box(start: Points(x: 0.5, y: 1.25), end: Points(x: 3.5, y: 2.25), lineThickness: 1.0 / 16.0)
    line4!.ruler = inchRuler
    lbl?.add(line4)
    
    let barcodeItem128 = BarcodeItem(start: Points(x: 0.5, y: (1.0 + 1.5 + 1.0 / 4.0 + 1.2)), height: 1.2, data: "Code 128")
    
    let bcd128 = Barcode_1D(barcodeItem: barcodeItem128)
    bcd128!.barcodeType = BarcodeTypeEnum1D.dCode128
    bcd128!.printHumanReadable = true
    bcd128!.rotation = .none
    bcd128!.ruler = inchRuler
    bcd128!.barWidths = BarWidths(narrowBar: 0.015, wideBar: 0.015)
    bcd128!.barWidths.ruler = Ruler(scale: .inch)
    lbl!.add(bcd128)
    
    let barcodeItem93 = BarcodeItem(start: Points(x: 0.5, y: 3.5 - 1.0 / 8.0 - 0.6), height: 0.6, data: "CODE 93")
    
    let bcd93 = Barcode_1D(barcodeItem: barcodeItem93)
    bcd93!.barcodeType = BarcodeTypeEnum1D.dCode93
    bcd93!.printHumanReadable = true
    bcd93!.rotation = .none
    bcd93!.ruler = inchRuler
    bcd93!.barWidths = BarWidths(narrowBar: 0.025, wideBar: 0.025 * 3)
    bcd93!.barWidths.ruler = Ruler(scale: .inch)
    lbl!.add(bcd93)
    return lbl!
}

func RfidEncode() -> Label {
    let lbl = Label(name: "RfidLbl")

    let a32BitField: UInt32 = 0x11223344
    let a16BitField: UInt16 = 0xBEEF
    let a6CharAsciiString = "MyData"
    var epcHexData = RfidConvert.toHex(from: a32BitField)
    epcHexData! += RfidConvert.toHex(fromBytes: a6CharAsciiString.data(using: .ascii)!)
    epcHexData! += RfidConvert.toHex(fromUShort: a16BitField)

    let epc = Rfid_Write(memBlock: RfidMemBlockEnum.EPC, data: epcHexData)
    lbl!.add(epc)

    var userHexData = RfidConvert.toHex(fromASCIIString: "MyUserData")
    userHexData! += "0ABCDE0F"
    let userMem = Rfid_Write(memBlock: RfidMemBlockEnum.user, data: userHexData)
    userMem!.offsetFromStart = 2
    lbl!.add(userMem)

    return lbl!
}

func BcdMaxicodes() -> Label {
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 203, unit: .inch))

    let lbl = Label(name: "MaxiBcds")

    let maxiDataStructCarrier = MaxicodeMsgStructured(mode: .mode2, postalCode: "902557317", countryCode: "800", serviceClass: "200", remainingMsg: "Maxicode Carrier Standard")
    let maxicodeBarcodeSc = MaxicodeBarcode(start: Points(x: 0.5, y: 0.5), data: maxiDataStructCarrier)
    maxicodeBarcodeSc!.ruler = Ruler(scale: .inch)

    let maxiData = MaxicodeMsg(mode: .mode4, primaryMsg: "123456789", remainingMsg: "Maxicode unstructured")
    let maxicodeBarcode = MaxicodeBarcode(start: Points(x: 0.5, y: 3.5), data: maxiData)
    maxicodeBarcode!.ruler = Ruler(scale: .inch)

    lbl!.add(maxicodeBarcodeSc)
    lbl!.add(maxicodeBarcode)

    return lbl!
}

func BcdDataMatrix() -> Label {
    let lbl = Label(name: "DMatrixBcds")

    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 600, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))

    let dfltMatrix = DataMatrixBarcode(start: Points(x: 0.25, y: 0.25), data: "Default DataMatrix")

    let rectMatrix = DataMatrixBarcode(start: Points(x: 1.25, y: 0.25), data: "Rectangular DataMatrix")
    rectMatrix!.rotation = .counterClockwise
    rectMatrix!.rectangle = true
    rectMatrix!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let dfltMatrixMultiLine = DataMatrixBarcode(start: Points(x: 2.25, y: 0.25), data: "Line 1 DataMatrix")
    let Eol = dfltMatrixMultiLine!.ctrlChar(0x0D) + dfltMatrixMultiLine!.ctrlChar(0x0A)
    dfltMatrixMultiLine!.data += Eol + "Line 2 content" + Eol + "Line 3 content"

    let rectMatrixUserDefinedDimensions = DataMatrixBarcode(start: Points(x: 1.25, y: 1.75), data: "DataMatrix with user defined dimensions")
    rectMatrixUserDefinedDimensions!.rectangle = true
    rectMatrixUserDefinedDimensions!.rowsCols = NSValue(size: CGSize(width: 16, height: 36))
    rectMatrixUserDefinedDimensions!.cellSize = CellSquare(xDim: 0.030, ruler: Ruler(scale: .inch))

    lbl!.add(dfltMatrix)
    lbl!.add(rectMatrix)
    lbl!.add(dfltMatrixMultiLine)
    lbl!.add(rectMatrixUserDefinedDimensions)

    return lbl!
}

func BcdAztec() -> Label {
    let lbl = Label(name: "AztecBcodes")

    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))

    let someText = "Mr. AirTraveler, seat A, flight 200"

    let bcdDefault = AztecBarcode(start: Points(x: 0.25, y: 1.0), data: someText)
    bcdDefault!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let bcdFixedErrCorr = AztecBarcode(start: Points(x: 1.5, y: 1.0), data: someText)
    bcdFixedErrCorr!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdFixedErrCorr!.type = .fixedErrCorrection
    bcdFixedErrCorr!.fixedErrCorrection = 30

    let bcdCompact = AztecBarcode(start: Points(x: 0.25, y: 2.25), data: someText)
    bcdCompact!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdCompact!.type = .compact
    bcdCompact!.layers = 4

    let bcdFull = AztecBarcode(start: Points(x: 1.5, y: 2.25), data: someText)
    bcdFull!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdFull!.type = .full
    bcdFull!.layers = 5

    let bcdRuneA = AztecBarcode(start: Points(x: 0.25, y: 4.00), data: "0")
    bcdRuneA!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdRuneA!.type = .rune

    let bcdRuneB = AztecBarcode(start: Points(x: 0.75, y: 4.00), data: "255")
    bcdRuneB!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdRuneB!.type = .rune

    let bcdRuneC = AztecBarcode(start: Points(x: 1.25, y: 4.00), data: "123")
    bcdRuneC!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdRuneC!.type = .rune

    lbl!.add(bcdDefault)
    lbl!.add(bcdFixedErrCorr)
    lbl!.add(bcdFull)
    lbl!.add(bcdCompact)
    lbl!.add(bcdRuneA)
    lbl!.add(bcdRuneB)
    lbl!.add(bcdRuneC)

    return lbl!
}

func BcdQRCode() -> Label {
    let lbl = Label(name: "QRCodes")

    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: ScaleEnum.inch))

    let enText = "Tree in the forest"
    let jaText = "森の中の木"

    let english = QRBarcode(start: Points(x: 0.25, y: 1.0), data: enText)
    english!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let englishMasked = QRBarcode(start: Points(x: 1.5, y: 1.0), data: enText)
    englishMasked!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    englishMasked!.mask = .mask4

    let japanese = QRBarcode(start: Points(x: 0.25, y: 2.25), data: jaText)
    japanese!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    japanese!.mask = .mask1

    let japaneseMasked = QRBarcode(start: Points(x: 1.5, y: 2.25), data: jaText)
    japaneseMasked!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    japaneseMasked!.mask = .mask4

    let autoEncData = QRBarcode(start: Points(x: 0.25, y: 3.75), data: "12345678 TREE IN THE FOREST 森の中の木")
    autoEncData!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    autoEncData!.mask = .mask4

    let manualModeData: [[Any]] = [
        [NSNumber(value: QRCodeManualEncodingEnum.numeric.rawValue), "12345678"],
        [NSNumber(value: QRCodeManualEncodingEnum.alphaNumeric.rawValue), " TREE IN THE FOREST "],
        [NSNumber(value: QRCodeManualEncodingEnum.binary.rawValue), "森の中の木"]
    ]

    let manualEncData = QRBarcode(start: Points(x: 1.75, y: 3.75), manuallyEncodedData: manualModeData)
    manualEncData!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    manualEncData!.mask = .mask4

    lbl!.add(english)
    lbl!.add(englishMasked)
    lbl!.add(japanese)
    lbl!.add(japaneseMasked)
    lbl!.add(autoEncData)
    lbl!.add(manualEncData)

    return lbl!
}





//PGL part

func PglPictureTest() -> PglLabel {
    let lbl = PglLabel(name: "PictureTest")
    
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: ScaleEnum.inch))
    
    let picture = PglPicture(
        start: Points(x: 0.25, y: 0.25),
        imageName: "TMP"
    )
    lbl!.add(picture)
    
    return lbl!
}

func fileReplace(pathAndName: String, dataToWrite: String) {
    do {
        try dataToWrite.write(toFile: pathAndName, atomically: true, encoding: .utf8)
    } catch {
        print("Error writing file: \(error.localizedDescription)")
    }
}

func stringToHex(_ input: String) -> String {
    let utf8String = input.utf8
    var hexString = ""
    for byte in utf8String {
        hexString.append(String(format: "%02X", byte))
    }
    return hexString
}

func PglRfidEncode() -> PglLabel {
    let lbl = PglLabel(name: "RfidLbl")!

    let a32BitField: UInt32 = 0x11223344
    let a16BitField: UInt16 = 0xBEEF
    let a6CharAsciiString = "MyData"
    
    var epcHexData = String(format: "%08X%@", a32BitField, stringToHex(a6CharAsciiString))
    epcHexData += String(format: "%04X", a16BitField)

    let epc = PglRfid_Write(memBlock: RfidMemBlockEnum.EPC, data: epcHexData)!
    lbl.add(epc)
    
    var userDataHex = stringToHex("MyUserData")
    userDataHex += "0ABCDE0F"

    let userMem = PglRfid_Write(memBlock: RfidMemBlockEnum.user, data: userDataHex)!
    userMem.offsetFromStart = 2
    lbl.add(userMem)

    return lbl
}

func PglBcdPdf417() -> PglLabel {
    let lbl = PglLabel(name: "Pdf417Bcodes")!
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 600, unit: ScaleEnum.inch)!)
    Defaults.setRuler(Ruler(scale: ScaleEnum.inch)!)

    let someText = "The happiness in your life depends on the quality of your thoughts. --Marcus Aurelius"
    let someShortText = "PI = 3.1415"

    let bcdDefault = PglPdf417Barcode(start: Points(x: 0.25, y: 0.50)!, data: someText)!
    bcdDefault.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: ScaleEnum.inch)!)

    let bcdErrCorrectionLvl0 = PglPdf417Barcode(start: Points(x: 0.25, y: 1.50)!, data: someShortText)!
    bcdErrCorrectionLvl0.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdErrCorrectionLvl0.errorCorrection = Pdf417ErrCorrectionEnum.level0

    let bcdErrCorrectionLvl5 = PglPdf417Barcode(start: Points(x: 0.25, y: 2.00)!, data: someShortText)!
    bcdErrCorrectionLvl5.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdErrCorrectionLvl5.errorCorrection = Pdf417ErrCorrectionEnum.level5

    let bcdRowsLimited = PglPdf417Barcode(start: Points(x: 0.25, y: 3.00)!, data: someShortText)!
    bcdRowsLimited.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdRowsLimited.rows = 15

    let bcdColsLimited = PglPdf417Barcode(start: Points(x: 0.25, y: 4.00)!, data: someShortText)!
    bcdColsLimited.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdColsLimited.columns = 5

    lbl.add(bcdDefault)
    lbl.add(bcdErrCorrectionLvl0)
    lbl.add(bcdErrCorrectionLvl5)
    lbl.add(bcdRowsLimited)
    lbl.add(bcdColsLimited)

    return lbl
}

func PglBcdAztec() -> PglLabel {
    let lbl = PglLabel(name: "AztecBcodes")!
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: ScaleEnum.inch)!)
    Defaults.setRuler(Ruler(scale: ScaleEnum.inch)!)

    let someText = "Mr. AirTraveler, seat A, flight 200"

    let bcdDefault = PglAztecBarcode(start: Points(x: 0.25, y: 1.0)!, data: someText)!
    bcdDefault.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)

    let bcdFixedErrCorr = PglAztecBarcode(start: Points(x: 1.5, y: 1.0)!, data: someText)!
    bcdFixedErrCorr.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdFixedErrCorr.type = AztecCodeTypeEnum.fixedErrCorrection
    bcdFixedErrCorr.fixedErrCorrection = 30

    let bcdCompact = PglAztecBarcode(start: Points(x: 0.25, y: 2.25)!, data: someText)!
    bcdCompact.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdCompact.type = AztecCodeTypeEnum.compact
    bcdCompact.layers = 4

    let bcdFull = PglAztecBarcode(start: Points(x: 1.5, y: 2.25)!, data: someText)!
    bcdFull.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdFull.type = AztecCodeTypeEnum.full
    bcdFull.layers = 5

    let bcdRuneA = PglAztecBarcode(start: Points(x: 0.25, y: 4.00)!, data: "0")!
    bcdRuneA.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdRuneA.type = AztecCodeTypeEnum.rune

    let bcdRuneB = PglAztecBarcode(start: Points(x: 0.75, y: 4.00)!, data: "255")!
    bcdRuneB.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdRuneB.type = AztecCodeTypeEnum.rune

    let bcdRuneC = PglAztecBarcode(start: Points(x: 1.25, y: 4.00)!, data: "123")!
    bcdRuneC.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    bcdRuneC.type = AztecCodeTypeEnum.rune

    lbl.add(bcdDefault)
    lbl.add(bcdFixedErrCorr)
    lbl.add(bcdCompact)
    lbl.add(bcdRuneB)

    return lbl
}

func PglBcdQRCode() -> PglLabel {
    let lbl = PglLabel(name: "QRCodes")!
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: ScaleEnum.inch)!)
    Defaults.setRuler(Ruler(scale: ScaleEnum.inch)!)

    let enText = "Tree in the forest"
    let jaText = "森の中の木"

    let english = PglQRBarcode(start: Points(x: 0.25, y: 1.0)!, data: enText)!
    english.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)

    let englishMasked = PglQRBarcode(start: Points(x: 1.5, y: 1.0)!, data: enText)!
    englishMasked.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    englishMasked.mask = QRCodeMaskEnum.mask4

    let japanese = PglQRBarcode(start: Points(x: 0.25, y: 2.25)!, data: jaText)!
    japanese.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    japanese.mask = QRCodeMaskEnum.mask1

    let japaneseMasked = PglQRBarcode(start: Points(x: 1.5, y: 2.25)!, data: jaText)!
    japaneseMasked.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    japaneseMasked.mask = QRCodeMaskEnum.mask4

    let autoEncData = PglQRBarcode(start: Points(x: 0.25, y: 3.75)!, data: "12345678 TREE IN THE FOREST 森の中の木")!
    autoEncData.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    autoEncData.mask = QRCodeMaskEnum.mask4

    let manualModeData: [[Any]] = [
        [NSNumber(value: QRCodeManualEncodingEnum.numeric.rawValue), "12345678"],
        [NSNumber(value: QRCodeManualEncodingEnum.alphaNumeric.rawValue), " TREE IN THE FOREST "],
        [NSNumber(value: QRCodeManualEncodingEnum.binary.rawValue), "森の中の木"]
    ]

    let manualEncData = PglQRBarcode(start: Points(x: 1.75, y: 3.75)!, manuallyEncodedData: manualModeData)!
    manualEncData.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)
    manualEncData.mask = QRCodeMaskEnum.mask4

    lbl.add(english)
    lbl.add(englishMasked)
    lbl.add(japanese)
    lbl.add(japaneseMasked)
    lbl.add(autoEncData)
    lbl.add(manualEncData)

    return lbl
}

func PglBcdDataMatrix() -> PglLabel {
    let lbl = PglLabel(name: "DMatrixBcds")!
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: ScaleEnum.inch)!)
    Defaults.setRuler(Ruler(scale: ScaleEnum.inch)!)

    let dfltMatrix = PglDataMatrixBarcode(start: Points(x: 0.25, y: 0.25)!, data: "Default DataMatrix")!

    let rectMatrix = PglDataMatrixBarcode(start: Points(x: 1.25, y: 0.25)!, data: "Rectangular DataMatrix")!
    rectMatrix.rotation = RotateEnum.counterClockwise
    rectMatrix.rectangle = true
    rectMatrix.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: ScaleEnum.inch)!)

    let dfltMatrixMultiLine = PglDataMatrixBarcode(start: Points(x: 2.25, y: 0.25)!, data: "Line 1 DataMatrix")!

    let Eol = dfltMatrixMultiLine.ctrlChar(0x0D)! + dfltMatrixMultiLine.ctrlChar(0x0A)! // CR + LF
    dfltMatrixMultiLine.data = dfltMatrixMultiLine.data + Eol + "Line 2 content" + Eol + "Line 3 content"

    let rectMatrixUserDefinedDimensions = PglDataMatrixBarcode(start: Points(x: 1.25, y: 1.75)!, data: "DataMatrix with user defined dimensions")!
    rectMatrixUserDefinedDimensions.rectangle = true
    rectMatrixUserDefinedDimensions.rowsCols = NSValue(size: CGSize(width: 16, height: 36))
    rectMatrixUserDefinedDimensions.cellSize = CellSquare(xDim: 0.030, ruler: Ruler(scale: ScaleEnum.inch)!)

    lbl.add(dfltMatrix)
    lbl.add(rectMatrix)
    lbl.add(dfltMatrixMultiLine)
    lbl.add(rectMatrixUserDefinedDimensions)

    return lbl
}

func PglBcdMaxicodes() -> PglLabel {
    let lbl = PglLabel(name: "MaxiBcds")!

    let maxiDataStructCarrier = PglMaxicodeMsgStructured(mode: MaxicodeModeEnum.mode2, postalCode: "90255", countryCode: "800", serviceClass: "200", remainingMsg: "Maxicode Carrier Standard")!
    let maxicodeBarcodeSc = PglMaxicodeBarcode(start: Points(x: 0.5, y: 0.5)!, data: maxiDataStructCarrier)!
    maxicodeBarcodeSc.ruler = Ruler(scale: ScaleEnum.inch)!

    let maxiDataOss = PglMaxicodeMsgStructuredOpenSystemStandard(mode: MaxicodeModeEnum.mode3, year: "24", postalCode: "OHA123", countryCode: "123", serviceClass: "400", remainingMsg: "Maxicode Open Standard Format")!
    let maxicodeBarcodeOss = PglMaxicodeBarcode(start: Points(x: 0.5, y: 2.0)!, data: maxiDataOss)!
    maxicodeBarcodeOss.ruler = Ruler(scale: ScaleEnum.inch)!

    let maxiData = PglMaxicodeMsg(mode: MaxicodeModeEnum.mode4, primaryMsg: "123456789", remainingMsg: "Maxicode unstructured")!
    let maxicodeBarcode = PglMaxicodeBarcode(start: Points(x: 0.5, y: 3.5)!, data: maxiData)!
    maxicodeBarcode.ruler = Ruler(scale: ScaleEnum.inch)!

    lbl.add(maxicodeBarcodeSc)
    lbl.add(maxicodeBarcodeOss)
    lbl.add(maxicodeBarcode)

    return lbl
}

func PglSimpleLabel(name: String, address: String) -> PglLabel {
    let lbl = PglLabel(name: "SimpleLabel")!

    let inchRuler = Ruler(scale: ScaleEnum.inch)!
    let mmRuler = Ruler(scale: ScaleEnum.MM)!

    let line1 = PglLine(start: Points(x: 2.5, y: 1.0 / 16.0)!, end: Points(x: 2.5, y: 1.0)!, lineThickness: 1.0 / 32.0)!
    line1.ruler = inchRuler
    lbl.add(line1)

    let line2 = PglLine(start: Points(x: 0.12, y: 1.0)!, end: Points(x: 3.88, y: 1.0)!, lineThickness: 1.0 / 32.0)!
    line2.ruler = inchRuler
    lbl.add(line2)

    let line3 = PglLine(start: Points(x: 0.12, y: 3.5)!, end: Points(x: 3.88, y: 3.5)!, lineThickness: 1.0 / 32.0)!
    line3.ruler = inchRuler
    lbl.add(line3)

    let box1 = PglBox(start: Points(x: 0.5, y: 1.25)!, end: Points(x: 3.5, y: 2.25)!, lineThickness: 1.0 / 16.0)!
    box1.ruler = inchRuler
    lbl.add(box1)

    let productText = PglText()!
    productText.fontSizeUnits = FontSizeUnitsEnum.ruler
    productText.ruler = inchRuler
    productText.alignment = AlignEnum.center
    productText.fontName = "93952.sf"

    let textItem1 = TextItem(start: Points(x: 2.0, y: 1.25 + 7.0 / 16.0)!, data: "MY MAGIC")!
    textItem1.fontSize = FontSize(x: 3.0 / 16.0, y: 7.0 / 16.0)
    productText.text.add(textItem1)

    let textItem2 = TextItem(start: Points(x: 2.0, y: 1.25 + 1.0 - 3.0 / 16.0)!, data: "PRODUCT")!
    textItem2.fontSize = FontSize(x: 3.0 / 16.0, y: 7.0 / 16.0)
    productText.text.add(textItem2)

    lbl.add(productText)

    let boldToFrom = PglText()!
    boldToFrom.fontSizeUnits = FontSizeUnitsEnum.ruler
    boldToFrom.ruler = mmRuler
    boldToFrom.fontStyle = FontStyleEnum.bold
    boldToFrom.fontName = "92248.sf"

    let bold_textItem1 = TextItem(start: Points(x: 5.0, y: 5.0)!, data: "TO:")!
    bold_textItem1.fontSize = FontSize(x: 2.5, y: 5.0)
    boldToFrom.text.add(bold_textItem1)

    let bold_textItem2 = TextItem(start: Points(x: (2.5 + 1 / 16.0) * 25.4, y: 5.0)!, data: "FROM:")!
    boldToFrom.text.add(bold_textItem2)

    lbl.add(boldToFrom)

    let companyName = PglText()!
    companyName.fontSizeUnits = FontSizeUnitsEnum.percent
    companyName.ruler = mmRuler
    companyName.fontStyle = FontStyleEnum.italic
    companyName.fontName = "92500.sf"

    let companyNameTextItem = TextItem(start: Points(x: (2.5 + 1 / 16.0 + 1 / 8.0) * 25.4, y: 17.0)!, data: "Happy Inc.")!
    companyNameTextItem.fontSize = FontSize(x: 2.0, y: 3.0)
    companyName.text.add(companyNameTextItem)

    lbl.add(companyName)

    let nameTxt = PglText()!
    nameTxt.fontSizeUnits = FontSizeUnitsEnum.ruler
    nameTxt.ruler = mmRuler
    nameTxt.fontStyle = FontStyleEnum.italic

    let nameTextItem = TextItem(start: Points(x: 8.0, y: 10.0)!, data: name)!
    nameTextItem.fontSize = FontSize(x: 2.5, y: 5.0)
    nameTxt.text.add(nameTextItem)

    lbl.add(nameTxt)

    let addressTxt = PglText()!
    addressTxt.ruler = mmRuler

    let addressTextItem = TextItem(start: Points(x: 8.0, y: 17.0)!, data: address)!
    addressTxt.text.add(addressTextItem)

    lbl.add(addressTxt)

    let bcd128 = PglBarcode_1D(barcodeItem: BarcodeItem(start: Points(x: 0.5, y: (1.5 + 1 / 4.0 + 1.2))!, height: 1.2, data: "Code 128")!)!
    bcd128.barcodeType = BarcodeTypeEnum1D.dCode128
    bcd128.printHumanReadable = true
    bcd128.rotation = RotateEnum.none
    bcd128.ruler = inchRuler
    bcd128.barWidths = PglBarWidths(narrowBar: 0.015, wideBar: 0.015 * 4.1)!
    bcd128.barWidths!.ruler = inchRuler
    lbl.add(bcd128)

    let bcd93 = PglBarcode_1D(barcodeItem: BarcodeItem(start: Points(x: 0.5, y: 3.5 - 1 / 8.0 - 0.6)!, height: 0.6, data: "CODE 93")!)!
    bcd93.barcodeType = BarcodeTypeEnum1D.dCode93
    bcd93.printHumanReadable = true
    bcd93.rotation = RotateEnum.none
    bcd93.ruler = inchRuler
    bcd93.barWidths = PglBarWidths(narrowBar: 0.025, wideBar: 0.025 * 4.1)!
    bcd93.barWidths!.ruler = inchRuler
    lbl.add(bcd93)

    let dmCustomer = PglDataMatrixBarcode(start: Points(x: 2.7, y: 4.0)!, data: name)!
    dmCustomer.cellSize = CellSquare(xDim: 0.040, ruler: inchRuler)
    dmCustomer.ruler = inchRuler
    let Eol = dmCustomer.ctrlChar(0x0D)! + dmCustomer.ctrlChar(0x0A)!
    dmCustomer.data = "\(dmCustomer.data!)\(Eol)\(address)"
    lbl.add(dmCustomer)

    return lbl
}

func RulerLines(length: Float, vertical: Bool, inchUnits: Bool, margin: Float) -> [PglLine] {
    var rulerLines: [PglLine] = []
    let tickRuler = inchUnits ? Ruler(scale: .inch)! : Ruler(scale: .MM)!

    var rulerLength = length
    var tickThickness: Float = 0.010
    var tickLength: Float = 1 / 16.0
    let ticksPerUnit: Float = inchUnits ? 16.0 : 1.0

    var margin = margin

    if !inchUnits {
        tickThickness *= MM_PER_INCH
        tickLength *= MM_PER_INCH
        margin *= MM_PER_INCH
    }

    rulerLength -= tickThickness

    for i in stride(from: 1, through: (rulerLength * ticksPerUnit), by: 1) {
        var tick = tickLength
        if inchUnits {
            if Int(i) % 16 == 0 {
                tick *= 3.5
            } else if Int(i) % 8 == 0 {
                tick *= 2.5
            } else if Int(i) % 4 == 0 {
                tick *= 2.0
            } else if Int(i) % 2 == 0 {
                tick *= 1.5
            }
        } else {
            if Int(i) % 10 == 0 {
                tick *= 3.0
            } else if Int(i) % 5 == 0 {
                tick *= 1.5
            }
        }

        let tickLine: PglLine
        if vertical {
            tickLine = PglLine(xStart: margin, yStart: i / ticksPerUnit, xEnd: margin + tick, yEnd: i / ticksPerUnit, lineThickness: tickThickness)!
        } else {
            tickLine = PglLine(xStart: i / ticksPerUnit, yStart: margin, xEnd: i / ticksPerUnit, yEnd: margin + tick, lineThickness: tickThickness)!
        }


        tickLine.ruler = tickRuler
        rulerLines.append(tickLine)
    }

    return rulerLines
}

func RuleredLabel(width: Float, length: Float, inchUnits: Bool, rulerMargin: Float) -> PglLabel {
    let verRulerTicks = RulerLines(length: length, vertical: true, inchUnits: inchUnits, margin: rulerMargin)
    let horRulerTicks = RulerLines(length: width, vertical: false, inchUnits: inchUnits, margin: rulerMargin)

    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: ScaleEnum.inch)!)

    let rulerLbl = PglLabel(name: "Ruler")!
    for tickLine in verRulerTicks {
        rulerLbl.add(tickLine)
    }
    for tickLine in horRulerTicks {
        rulerLbl.add(tickLine)
    }

    return rulerLbl
}


