import SwiftUI
import CoreBluetooth
import UniPRT


enum CommType: Int, CaseIterable, Identifiable {
    case bluetooth, tcp, ble
    var id: Int { rawValue }
    var title: String {
        switch self {
        case .bluetooth: return "Bluetooth"
        case .tcp:       return "TCP/IP"
        case .ble:       return "BLE"
        }
    }
}

struct ContentView: View {
    @State private var ipText = "34:81:F4:CB:F4:99"
    @State private var commType: CommType = .bluetooth
    @StateObject private var viewModel = CommunicationViewModel()

    var body: some View {
        NavigationView {
            Form {
                Section(header: Text("Communication Type")) {
                    Picker("Comm Type", selection: $commType) {
                        ForEach(CommType.allCases) { type in
                            Text(type.title).tag(type)
                        }
                    }
                    .pickerStyle(SegmentedPickerStyle())
                }

                Section(header: Text("Address / Name")) {
                    TextField("Enter MAC / IP / BLE Name", text: $ipText)
                        .textFieldStyle(RoundedBorderTextFieldStyle())
                        .keyboardType(.asciiCapable)
                }

                Section(header: Text("Actions")) {
                    switch commType {
                    case .bluetooth:
                        Button(viewModel.btComm == nil ? "BT Connect" : "BT Close") {
                            viewModel.connectCloseBT(ip: ipText)
                        }
                        Button("BT Open") {
                            viewModel.openBT()
                        }

                    case .tcp:
                        Button(viewModel.tcpComm == nil ? "TCP Connect" : "TCP Close") {
                            viewModel.connectCloseTCP(ip: ipText)
                        }
                        Button("TCP Open") {
                            viewModel.openTCP()
                        }

                    case .ble:
                        Button("Scan BLE Devices") {
                            viewModel.scanBLE()
                        }
                        if !viewModel.bleDevices.isEmpty {
                            Picker("Select Device", selection: $viewModel.selectedBleDevice) {
                                ForEach(viewModel.bleDevices, id: \.self) { dev in
                                    Text(dev).tag(dev as String?)
                                }
                            }
                            .pickerStyle(WheelPickerStyle())

                            Button(viewModel.bleComm == nil ? "BLE Close" : "BLE Connect") {
                                viewModel.toggleBLEConnect()
                            }
                        }
                    }
                }



                Section(header: Text("Results")) {
                    ScrollView {
                        Text(viewModel.resultText)
                            .frame(maxWidth: .infinity, alignment: .leading)
                            .font(.system(size: 12, weight: .regular, design: .monospaced))
                            .padding(.vertical, 4)
                    }
                    .frame(height: 200)
                    .border(Color.gray)
                }
                
                Section(header: Text("Printer Monitoring")) {
                    Button("Start RFID Monitor") {
                        viewModel.runRfidMonitor()
                    }
                    Button("Start Printer Monitor") {
                        viewModel.runPrinterMonitor()
                    }
                    Button("Start ODV Monitor") {
                        viewModel.runOdvMonitor()
                    }
                }

                //Label Maker
                Button("Generate Labels (TSPL & PGL)") {
                    viewModel.generateAllLabels()
                }
                Button("Mgmt BLE") { viewModel.ble_mgmt() }
                Button("Mgmt TCP") { viewModel.tcp_mgmt() }
                Button("Mgmt BT") { viewModel.bt_mgmt() }

                //monitor

            }
            .navigationTitle("Communication")
        }
    }
}

class CommunicationViewModel: ObservableObject {
    @Published var resultText = ""
    // BT
    var btComm: BtComm?
    // TCP
    var tcpComm: TcpComm?
    // BLE
    var bleComm: BleComm?
    
    var comm: IComm?
    @Published var bleDevices: [String] = []
    @Published var selectedBleDevice: String = ""

    // MARK: - BT Actions
    func connectCloseBT(ip: String) {
        if let comm = btComm {
            comm.close()
            btComm = nil
            appendResult("BT Connection closed")
        } else {
            btComm = BtComm(deviceAddress: ip)
            appendResult("BT Connection initialized")
        }
    }
    func openBT() {
        btComm?.open()
        self.comm = btComm
        appendResult("BT Connection opened")
    }

    // MARK: - TCP Actions
    func connectCloseTCP(ip: String) {
        if let comm = tcpComm {
            comm.close()
            tcpComm = nil
            appendResult("TCP Connection closed")
        } else {
            tcpComm = TcpComm(ipAddress: ip, port: 3007)
            appendResult("TCP Connection initialized")
        }
    }
    
    func openTCP() {
        tcpComm?.open()
        self.comm = tcpComm
        appendResult("TCP Connection opened")
    }

    // MARK: - BLE Actions
    func scanBLE() {
        bleComm = BleComm()
        // trigger scan after delay to allow state
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) {
            self.bleComm?.availableDevices()
            DispatchQueue.main.asyncAfter(deadline: .now() + 3) {
                self.bleDevices = self.bleComm?.discoveredDevices as? [String] ?? []
                self.appendResult("Found BLE devices: \(self.bleDevices)")
            }
        }
    }
    func toggleBLEConnect() {
        guard let ble = bleComm else { return }
        if ble.isConnected() {
            ble.close()
            appendResult("BLE Connection closed")
        } else if !selectedBleDevice.isEmpty {
            ble.open(selectedBleDevice)
            self.comm = bleComm
            appendResult("BLE Connected to \(selectedBleDevice)")
        }
    }



    
    //Label Maker
    func generateAllLabels() {
        var output = "LabelMaker Results:\n\n"

        // TSPL
        output += "▶ RfidEncode:\n\(String(describing: RfidEncode()))\n\n"
        output += "▶ SimpleTextLabel:\n\(String(describing: SimpleTextLabel(name: "Mr. Milky Cheese", address: "123 No Way Road")))\n\n"
        output += "▶ BcdPdf417:\n\(String(describing: BcdPdf417()))\n\n"
        output += "▶ BcdMaxicodes:\n\(String(describing: BcdMaxicodes()))\n\n"
        output += "▶ BcdDataMatrix:\n\(String(describing: BcdDataMatrix()))\n\n"
        output += "▶ BcdAztec:\n\(String(describing: BcdAztec()))\n\n"
        output += "▶ BcdQRCode:\n\(String(describing: BcdQRCode()))\n\n"
        output += "▶ TsplPictureTest:\n\(String(describing: PictureTest()))\n\n"
        output += "▶ PictureTest:\n\(String(describing: PictureTest()))\n\n"

        // PGL
        output += "▶ PglRfidEncode:\n\(String(describing: PglRfidEncode()))\n\n"
        output += "▶ PglBcdPdf417:\n\(String(describing: PglBcdPdf417()))\n\n"
        output += "▶ PglBcdAztec:\n\(String(describing: PglBcdAztec()))\n\n"
        output += "▶ PglBcdQRCode:\n\(String(describing: PglBcdQRCode()))\n\n"
        output += "▶ PglBcdDataMatrix:\n\(String(describing: PglBcdDataMatrix()))\n\n"
        output += "▶ PglBcdMaxicodes:\n\(String(describing: PglBcdMaxicodes()))\n\n"
        output += "▶ PglSimpleLabel:\n\(String(describing: PglSimpleLabel(name: "Mr. Einstein", address: "123 Relativity Road")))\n\n"
        output += "▶ PglRuleredLabel (inch):\n\(String(describing: RuleredLabel(width: 4.0, length: 6.0, inchUnits: true, rulerMargin: 1.0/8.0)))\n\n"
        output += "▶ PglRuleredLabel (mm):\n\(String(describing: RuleredLabel(width: 4.0 * 25.4, length: 6.0 * 25.4, inchUnits: false, rulerMargin: 1.0/8.0)))\n\n"
        output += "▶ PglPictureTest:\n\(String(describing: PglPictureTest()))\n\n"

        appendResult(output)
    }
    
    func ble_mgmt() {
        guard let ble = bleComm else {
            appendResult("No BLE connection.")
            return
        }

        DispatchQueue.global().async {
            let settingsRW = SettingsReadWrite(bleComm: ble, usingDataPort: true)
            var output = "📡 BLE Mgmt Results:\n\n"
            
            // Get specific values
            let keys = ["BT.PairMethod", "BT.ConnectName"]
            if let partialValues = settingsRW?.getValuesForKeys(keys, timeout: 10_000) {
                output += "🔸 Partial Values:\n\(partialValues)\n\n"
            }
            
            // Get all properties
            if let allProperties = settingsRW?.getAllProperties(withTimeout: 45_000) {
                output += "🔸 All Properties:\n\(allProperties)\n\n"
            }
            
            // Get all values
            if let allValues = settingsRW?.getAllValues(withTimeout: 25_000) {
                output += "🔸 All Values:\n\(allValues)\n\n"
            }
            
            // Get properties for keys
            if let propKeys = settingsRW?.getPropertiesForKeys(keys, timeout: 10_000) {
                output += "🔸 Properties for Keys:\n\(propKeys)\n\n"
            }
            
            // Get a single property
            if let singleProp = settingsRW?.getPropertiesForKey("BT.PairMethod", timeout: 10_000) {
                output += "🔸 Single Property (BT.PairMethod):\n\(singleProp)\n\n"
            }
            DispatchQueue.main.async {
                self.appendResult(output)
            }
        }

//        appendResult(output)
    }
    
    func tcp_mgmt() {
       
        let settingsRW = SettingsReadWrite(tcpComm: (self.comm as! TcpComm), usingDataPort: false)
        var output = "TCP Mgmt Results:\n\n"

        if let result = settingsRW?.setValue("eJustWork", forKey: "BT.PairMethod") {
            output += "Set single value: \(result)\n"
        }

        if let result = settingsRW?.setValues(["BT.PairMethod":"eNumericComp", "BT.ConnectName":"BT-PTX3"], timeout: 10000) {
            output += "Set multiple values: \(result)\n"
        }

        let keys = ["BT.PairMethod", "BT.ConnectName"]

        if let val = settingsRW?.getPropertiesForKey("BT.PairMethod", timeout: 10000) {
            output += "getPropertiesForKey:\n\(val)\n\n"
        }

        if let val = settingsRW?.getValuesForKeys(keys, timeout: 10000) {
            output += "getValuesForKeys:\n\(val)\n\n"
        }

        if let val = settingsRW?.getAllValues(withTimeout: 30000) {
            output += "getAllValues:\n\(val)\n\n"
        }

        if let val = settingsRW?.getPropertiesForKeys(keys, timeout: 10000) {
            output += "getPropertiesForKeys:\n\(val)\n\n"
        }

        if let val = settingsRW?.getAllProperties(withTimeout: 30000) {
            output += "getAllProperties:\n\(val)\n\n"
        }

        appendResult(output)
    }

    func bt_mgmt() {
        
        let settingsRW = SettingsReadWrite(btComm: btComm, usingDataPort: true)

        var output = "BT Mgmt Results:\n\n"

        let keys = [
            "BT.PairMethod",
            "BT.ConnectName",
            "System.KeyboardCC",
            "System.Wireless.MenuJob-d"
        ]

        if let val = settingsRW?.getValuesForKeys(keys, timeout: 10000) {
            output += "getValuesForKeys:\n\(val)\n\n"
        }

        if let val = settingsRW?.getAllProperties(withTimeout: 30000) {
            output += "getAllProperties:\n\(val)\n\n"
        }

        if let val = settingsRW?.getAllValues(withTimeout: 20000) {
            output += "getAllValues:\n\(val)\n\n"
        }

        if let val = settingsRW?.getPropertiesForKeys(keys, timeout: 10000) {
            output += "getPropertiesForKeys:\n\(val)\n\n"
        }

        if let val = settingsRW?.getPropertiesForKey("BT.PairMethod", timeout: 5000) {
            output += "getPropertiesForKey:\n\(val)\n\n"
        }

        appendResult(output)
    }
    
    func runRfidMonitor() {

       

        let rfidMonitor = RfidMonitor(tcpComm: self.tcpComm)
        let printerMonitor = PrinterMonitor(tcpComm: self.tcpComm)

        if let info = printerMonitor?.getPrinterInfo() {
            appendResult("RFID Printer Info:\nModel: \(info.model() ?? "")\nSN: \(info.serialNumber() ?? "")")
        }

        DispatchQueue.global().async {
            rfidMonitor?.rfidReportCallback = { report in
                var result = "\n[RFID Report]\n"
                result += (report?.failed() == nil) ? "RFID Failed." : "RFID Passed."
                result += "\nData: \(report?.data() ?? "")"
                self.appendResult(result)
            }
            rfidMonitor?.setRfidReportListening(true)
        }
    }

    func runPrinterMonitor() {

        
        let printerMonitor = PrinterMonitor(tcpComm: self.tcpComm)

        if let info = printerMonitor?.getPrinterInfo() {
            appendResult("Printer Info:\nModel: \(info.model() ?? "")\nSN: \(info.serialNumber() ?? "")")
        }

        DispatchQueue.global().async {
            printerMonitor?.engineStatusCallback = {
                self.appendResult("Engine Status: \($0 ?? "")")
            }
            printerMonitor?.setEngineStatusListening(true)
        }
    }

    func runOdvMonitor() {
       
        let odvMonitor = OdvMonitor(tcpComm: self.tcpComm)
        let printerMonitor = PrinterMonitor(tcpComm: self.tcpComm)

        if let info = printerMonitor?.getPrinterInfo() {
            appendResult("ODV Printer Info:\nModel: \(info.model() ?? "")\nSN: \(info.serialNumber() ?? "")")
        }

        DispatchQueue.global().async {
            odvMonitor?.odvReportCallback = { report in
                var result = "\n[ODV Report]\n"
                result += (report?.failed() == nil) ? "Barcode Failed." : "Barcode Passed."
                result += "\nGrade: \(String(format: "%.2f", report?.overallGradeAsFloat() ?? 0))"
                result += "\nData: \(report?.data() ?? "")"
                self.appendResult(result)
            }
            odvMonitor?.setOdvReportListening(true)
        }
    }





    // MARK: - Helpers
    private func appendResult(_ text: String) {
        DispatchQueue.main.async {
            if self.resultText.isEmpty {
                self.resultText = text
            } else {
                self.resultText += "\n" + text
            }
        }
    }
}

