//
//  Box.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/16.
//

#ifndef Box_h
#define Box_h

#import <Foundation/Foundation.h>
#import <UniPRT/IBox.h>
#import <UniPRT/IPoint.h>
#import <UniPRT/PrintPlane.h>
#import <UniPRT/Defaults.h>
#import <UniPRT/Point.h>
#import <UniPRT/_Box.h>

/**
 * \~English
 * @class _Box
 * @brief A class representing a box shape for TSPL.
 *
 * This class implements the IBox protocol and provides functionality for initializing and describing a box in TSPL.
 *
 * @ingroup TSPL
 *
 * \~Chinese
 * @class _Box
 * @brief 表示 TSPL 中的方形对象的类。
 *
 * 该类实现了 IBox 协议，提供了在 TSPL 中初始化和描述方形对象的功能。
 *
 * @ingroup TSPL
 *
 * \~Chinese-Traditional
 * @class _Box
 * @brief 表示 TSPL 中的方形對象的類。
 *
 * 該類實現了 IBox 協議，提供了在 TSPL 中初始化和描述方形對象的功能。
 *
 * @ingroup TSPL
 */
@interface Box : NSObject <IBox>

/**
 * \~English
 * @brief The ruler used to measure the box.
 *
 * \~Chinese
 * @brief 用于测量方形对象的标尺。
 *
 * \~Chinese-Traditional
 * @brief 用於測量方形對象的標尺。
 */
@property (nonatomic, strong) id<IRuler> ruler;

/**
 * \~English
 * @brief The starting point of the box.
 *
 * \~Chinese
 * @brief 方形对象的起始点。
 *
 * \~Chinese-Traditional
 * @brief 方形對象的起始點。
 */
@property (nonatomic, strong) id<IPoint> start;

/**
 * \~English
 * @brief The ending point of the box.
 *
 * \~Chinese
 * @brief 方形对象的终点。
 *
 * \~Chinese-Traditional
 * @brief 方形對象的終點。
 */
@property (nonatomic, strong) id<IPoint> end;

/**
 * \~English
 * @brief The thickness of the box's lines.
 *
 * \~Chinese
 * @brief 方形对象线条的粗细。
 *
 * \~Chinese-Traditional
 * @brief 方形對象線條的粗細。
 */
@property (nonatomic) float lineThickness;

/**
 * \~English
 * @brief The corner rounding radius of the box.
 *
 * \~Chinese
 * @brief 方形对象的圆角半径。
 *
 * \~Chinese-Traditional
 * @brief 方形對象的圓角半徑。
 */
@property (nonatomic) float cornerRounding;

/**
 * \~English
 * @brief Initializes a new instance of the _Box class with specified start and end points and line thickness.
 *
 * @param start The starting point of the box.
 * @param end The ending point of the box.
 * @param lineThickness The thickness of the box's lines.
 * @return A new instance of _Box.
 *
 * \~Chinese
 * @brief 使用指定的起点和终点以及线的粗细初始化 _Box 类的新实例。
 *
 * @param start 方形对象的起点。
 * @param end 方形对象的终点。
 * @param lineThickness 方形对象线条的粗细。
 * @return _Box 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的起點和終點以及線的粗細初始化 _Box 類的新實例。
 *
 * @param start 方形對象的起點。
 * @param end 方形對象的終點。
 * @param lineThickness 方形對象線條的粗細。
 * @return _Box 的新實例。
 */
- (instancetype)initWithStart:(id<IPoint>)start end:(id<IPoint>)end lineThickness:(float)lineThickness;

/**
 * \~English
 * @brief Initializes a new instance of the _Box class with specified start and end coordinates and line thickness.
 *
 * @param xStart The x-coordinate of the starting point.
 * @param yStart The y-coordinate of the starting point.
 * @param xEnd The x-coordinate of the ending point.
 * @param yEnd The y-coordinate of the ending point.
 * @param lineThickness The thickness of the box's lines.
 * @return A new instance of _Box.
 *
 * \~Chinese
 * @brief 使用指定的起点和终点坐标以及线的粗细初始化 _Box 类的新实例。
 *
 * @param xStart 起点的 x 坐标。
 * @param yStart 起点的 y 坐标。
 * @param xEnd 终点的 x 坐标。
 * @param yEnd 终点的 y 坐标。
 * @param lineThickness 方形对象线条的粗细。
 * @return _Box 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的起點和終點坐標以及線的粗細初始化 _Box 類的新實例。
 *
 * @param xStart 起點的 x 坐標。
 * @param yStart 起點的 y 坐標。
 * @param xEnd 終點的 x 坐標。
 * @param yEnd 終點的 y 坐標。
 * @param lineThickness 方形對象線條的粗細。
 * @return _Box 的新實例。
 */
- (instancetype)initWithXStart:(float)xStart yStart:(float)yStart xEnd:(float)xEnd yEnd:(float)yEnd lineThickness:(float)lineThickness;

/**
 * \~English
 * @brief Returns a string that can be sent to the printer.
 *
 * This method generates a string representation of the _Box instance that
 * is formatted and ready to be sent to a printer. The string includes all the
 * necessary commands and data required for printing in TSPL.
 *
 * @return A string description of the _Box instance.
 *
 * \~Chinese
 * @brief 返回可以发送到打印机的字符串。
 *
 * 该方法生成 _Box 实例的字符串表示形式，
 * 格式化并准备好发送到打印机。字符串包括在 TSPL 中打印所需的所有命令和数据。
 *
 * @return _Box 实例的字符串描述。
 *
 * \~Chinese-Traditional
 * @brief 返回可以發送到打印機的字符串。
 *
 * 該方法生成 _Box 實例的字符串表示形式，
 * 格式化並準備好發送到打印機。字符串包括在 TSPL 中打印所需的所有命令和數據。
 *
 * @return _Box 實例的字符串描述。
 */
- (NSString *)description;

@end

#endif /* _Box_h */

