//
//  JsonMessenger.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/8.
//

#ifndef JsonMessenger_h
#define JsonMessenger_h


#endif /* JsonMessenger_h */

#import <Foundation/Foundation.h>
#import <UniPRT/JsonComm.h>
#import <UniPRT/JsonMng.h>
#import <UniPRT/TcpComm.h>
#import <UniPRT/JsonCommController.h>
#import <UniPRT/BtComm.h>
#import <UniPRT/BleComm.h>

@class TcpComm;
@class BtComm;
@class UsbComm;
@class JsonMng;
@class JsonComm;
@class BleComm;

/**
 * \~English
 * @class JsonMessenger
 * @brief A class for managing JSON-based messaging.
 * @ingroup Mgmt
 *
 * \~Chinese
 * @class JsonMessenger
 * @brief 用于管理基于 JSON 的消息传递的类。
 * @ingroup Mgmt
 *
 * \~Chinese-Traditional
 * @class JsonMessenger
 * @brief 用於管理基於 JSON 的消息傳遞的類。
 * @ingroup Mgmt
 */
@interface JsonMessenger : NSObject {
    
    /** @cond */
    
    BOOL _usingDataPort;              /**< A flag indicating whether the data port is used. */
    BOOL _disposed;                   /**< A flag indicating whether the object is disposed. */
    TcpComm *_tcpComm;                /**< TCP communication instance. */
    JsonComm *_MgmtComm;              /**< Management JSON communication instance. */
    JsonComm *_MgmtCommPtr;           /**< Pointer to the management JSON communication instance. */
    ImpChannel *_MsgsFromPtr;         /**< Channel for messages from the pointer. */
    
    NSString *_myMgmtId;              /**< Management ID. */
    long _TrackCount;                 /**< Track count. */
    BOOL _sdkManagedComm;             /**< A flag indicating whether the SDK manages communication. */
    NSString *_managedDescriptor;     /**< Descriptor managed by the SDK. */
    
    /** @endcond */
}


/**
 * \~English
 * @brief Initializes a new instance with the given communication pointer, communication type, maximum input message capacity, and data port usage flag.
 *
 * The oldest message is dropped if messages not read before more messages received from printer.
 *
 * Set usingDataPort to true if the communication to printer commToPtr is through the same port that print data is sent through.
 *
 * This is important in order to properly format management messages so that printer differentiates the messages from print data.
 *
 * @param commToPtr The pointer to the communication instance.
 * @param iCommTyp The type of communication.
 * @param maxInputMsgCapacity The maximum input message capacity.
 * @param usingDataPort A flag indicating whether to use the data port.
 * @return An initialized instance of JsonMessenger, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用给定的通信指针、通信类型、最大输入消息容量和数据端口使用标志初始化一个新实例。
 *
 * 如果在从打印机收到更多消息之前没有读取消息，则最早的消息将被丢弃。
 *
 * 如果与打印机的通信通过发送打印数据的相同端口进行，请将 usingDataPort 设置为 true。
 *
 * 这对于正确格式化管理消息非常重要，以便打印机区分来自打印数据的消息。
 *
 * @param commToPtr 通信实例的指针。
 * @param iCommTyp 通信类型。
 * @param maxInputMsgCapacity 最大输入消息容量。
 * @param usingDataPort 指示是否使用数据端口的标志。
 * @return 一个初始化的 JsonMessenger 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用給定的通信指針、通信類型、最大輸入消息容量和數據端口使用標誌初始化一個新實例。
 *
 * 如果在從打印機收到更多消息之前沒有讀取消息，則最早的消息將被丟棄。
 *
 * 如果與打印機的通信通過發送打印數據的相同端口進行，請將 usingDataPort 設置為 true。
 *
 * 這對於正確格式化管理消息非常重要，以便打印機區分來自打印數據的消息。
 *
 * @param commToPtr 通信實例的指針。
 * @param iCommTyp 通信類型。
 * @param maxInputMsgCapacity 最大輸入消息容量。
 * @param usingDataPort 指示是否使用數據端口的標誌。
 * @return 一個初始化的 JsonMessenger 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithCommToPtr:(AComm *)commToPtr iCommTyp:(CommType)iCommTyp maxInputMsgCapacity:(int)maxInputMsgCapacity usingDataPort:(BOOL)usingDataPort;

/**
 * \~English
 * @brief Send a management command and data/content to printer.
 *
 * The optional TrackNo is the field used in the JSON command structure sent to printer and can be used by caller to check for response from printer. If no TrackNo is provided, one is generated.
 *
 * The return string is the TrackNo field used in the JSON message sent to the printer to allow caller to differentiate any response associated with the message sent by comparing the TrackNo field of any response received from printer.
 *
 * @param command The command of the message.
 * @param content The content of the message.
 * @param trackNo The track number of the message.
 * @return A string representing the result of the send operation.
 *
 * \~Chinese
 * @brief 发送管理命令和数据/内容到打印机。
 *
 * 可选的 TrackNo 是发送到打印机的 JSON 命令结构中使用的字段，调用者可以使用它来检查打印机的响应。如果未提供 TrackNo，则会生成一个。
 *
 * 返回的字符串是发送到打印机的 JSON 消息中使用的 TrackNo 字段，以允许调用者通过比较从打印机接收到的任何响应的 TrackNo 字段来区分与发送的消息相关的任何响应。
 *
 * @param command 消息的命令。
 * @param content 消息的内容。
 * @param trackNo 消息的跟踪号。
 * @return 表示发送操作结果的字符串。
 *
 * \~Chinese-Traditional
 * @brief 發送管理命令和數據/內容到打印機。
 *
 * 可選的 TrackNo 是發送到打印機的 JSON 命令結構中使用的字段，調用者可以使用它來檢查打印機的響應。如果未提供 TrackNo，則會生成一個。
 *
 * 返回的字符串是發送到打印機的 JSON 消息中使用的 TrackNo 字段，以允許調用者通過比較從打印機接收到的任何響應的 TrackNo 字段來區分與發送的消息相關的任何響應。
 *
 * @param command 消息的命令。
 * @param content 消息的內容。
 * @param trackNo 消息的跟踪號。
 * @return 表示發送操作結果的字符串。
 */
- (NSString *)sendMsgWithCommand:(NSString *)command content:(NSString *)content trackNo:(NSString *)trackNo;

/**
 * \~English
 * @brief Send a management command and data/content to printer and wait for response maxWaitTimeSecs time.
 *
 * Null object returned if no response received before timeout expired.
 *
 * All messages are discarded until the response to the specific message sent is received. The discarded messages may include any unread/old messages waiting to be read prior to the call to this method.
 *
 * @param command The command of the message.
 * @param content The content of the message.
 * @param maxWaitTimeSecs The maximum wait time in seconds.
 * @return A JsonMng instance representing the response.
 *
 * \~Chinese
 * @brief 发送管理命令和数据/内容到打印机并等待最大等待时间秒数的响应。
 *
 * 如果在超时之前未收到响应，则返回空对象。
 *
 * 在接收到发送的特定消息的响应之前，所有消息都会被丢弃。被丢弃的消息可能包括在调用此方法之前等待读取的任何未读/旧消息。
 *
 * @param command 消息的命令。
 * @param content 消息的内容。
 * @param maxWaitTimeSecs 最大等待时间（秒）。
 * @return 表示响应的 JsonMng 实例。
 *
 * \~Chinese-Traditional
 * @brief 發送管理命令和數據/內容到打印機並等待最大等待時間秒數的響應。
 *
 * 如果在超時之前未收到響應，則返回空對象。
 *
 * 在接收到發送的特定消息的響應之前，所有消息都會被丟棄。被丟棄的消息可能包括在調用此方法之前等待讀取的任何未讀/舊消息。
 *
 * @param command 消息的命令。
 * @param content 消息的內容。
 * @param maxWaitTimeSecs 最大等待時間（秒）。
 * @return 表示響應的 JsonMng 實例。
 */
- (JsonMng *)sendMsgAndWaitForResponseWithCommand:(NSString *)command content:(NSString *)content maxWaitTimeSecs:(int)maxWaitTimeSecs;

/**
 * \~English
 * @brief Read next message that may have been received asynchronously from printer.
 *
 * @return A JsonMng instance representing the next message.
 *
 * \~Chinese
 * @brief 读取可能从打印机异步接收到的下一条消息。
 *
 * @return 表示下一条消息的 JsonMng 实例。
 *
 * \~Chinese-Traditional
 * @brief 讀取可能從打印機異步接收到的下一條消息。
 *
 * @return 表示下一條消息的 JsonMng 實例。
 */
- (JsonMng *)readNextMsg;

/**
 * \~English
 * @brief Sends contents without any message formatting directly to printer.
 *
 * This is for cases where data needs to be sent without any added management message formatting.
 *
 * If there is a special management message or data need to be sent in a port that is shared for both data and management, this would be used to send data directly to printer.
 *
 * @param dataToSend The data to send.
 *
 * \~Chinese
 * @brief 将内容直接发送到打印机，不进行任何消息格式化。
 *
 * 适用于需要在没有任何附加管理消息格式的情况下发送数据的情况。
 *
 * 如果有特殊的管理消息或需要在同时用于数据和管理的端口发送的数据，这将用于直接将数据发送到打印机。
 *
 * @param dataToSend 要发送的数据。
 *
 * \~Chinese-Traditional
 * @brief 將內容直接發送到打印機，不進行任何消息格式化。
 *
 * 適用於需要在沒有任何附加管理消息格式的情況下發送數據的情況。
 *
 * 如果有特殊的管理消息或需要在同時用於數據和管理的端口發送的數據，這將用於直接將數據發送到打印機。
 *
 * @param dataToSend 要發送的數據。
 */
- (void)sendMsgRawWithDataToSend:(NSString *)dataToSend;

/**
 * \~English
 * @brief Disposes the object, releasing any resources.
 *
 * \~Chinese
 * @brief 释放对象，释放所有资源。
 *
 * \~Chinese-Traditional
 * @brief 釋放對象，釋放所有資源。
 */
- (void)dispose;

@end
