//
//  JsonConfig.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/4/19.
//

#ifndef JsonConfig_h
#define JsonConfig_h


#endif /* JsonConfig_h */

#import <Foundation/Foundation.h>
#import <UniPRT/Config.h>
#import <UniPRT/JsonComm.h>
#import <UniPRT/BleComm.h>
#import <UniPRT/BtComm.h>

// Assuming JsonComm and Config are already translated Objective-C classes

@class JsonComm;
@class Config;

/**
 * \~English
 * @class JsonConfig
 * @brief A class for managing configuration through JSON communication.
 * @ingroup Settings
 *
 * \~Chinese
 * @class JsonConfig
 * @brief 用于通过 JSON 通信管理配置的类。
 * @ingroup Settings
 *
 * \~Chinese-Traditional
 * @class JsonConfig
 * @brief 用於通過 JSON 通信管理配置的類。
 * @ingroup Settings
 */
@interface JsonConfig : NSObject

/**
 * \~English
 * @brief Must pass an opened TcpComm connection that can remain opened if listening for unsolicited messages from printer.
 *
 * @param connection The TCP communication object.
 * @param usingDataPort Whether to use data port for communication.
 * @return An initialized instance of JsonConfig, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 必须传递一个已打开的 TcpComm 连接，如果需要监听来自打印机的非请求消息，该连接必须保持打开。
 *
 * @param connection TCP 通信对象。
 * @param usingDataPort 是否使用数据端口通信。
 * @return 一个初始化的 JsonConfig 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 必須傳遞一個已打開的 TcpComm 連接，如果需要監聽來自印表機的非請求消息，該連接必須保持打開。
 *
 * @param connection TCP 通訊對象。
 * @param usingDataPort 是否使用資料端口通訊。
 * @return 一個初始化的 JsonConfig 實例，如果無法建立對象，則回傳 nil。
 */
- (instancetype)initWithTcpComm:(TcpComm *)connection usingDataPort:(BOOL)usingDataPort;

/**
 * \~English
 * @brief Initializes with an opened BtComm connection.
 *
 * @param connection The Bluetooth communication object.
 * @param usingDataPort Whether to use data port for communication.
 * @return An initialized instance of JsonConfig.
 *
 * \~Chinese
 * @brief 使用已打开的 BtComm 蓝牙连接初始化。
 *
 * @param connection 蓝牙通信对象。
 * @param usingDataPort 是否使用数据端口通信。
 * @return 一个初始化的 JsonConfig 实例。
 *
 * \~Chinese-Traditional
 * @brief 使用已開啟的 BtComm 藍牙連線初始化。
 *
 * @param connection 藍牙通訊對象。
 * @param usingDataPort 是否使用資料端口通訊。
 * @return 一個初始化的 JsonConfig 實例。
 */
- (instancetype)initWithBtComm:(BtComm *)connection usingDataPort:(BOOL)usingDataPort;

/**
 * \~English
 * @brief Initializes with an opened BleComm connection.
 *
 * @param connection The BLE communication object.
 * @param usingDataPort Whether to use data port for communication.
 * @return An initialized instance of JsonConfig.
 *
 * \~Chinese
 * @brief 使用已打开的 BleComm 藍牙低功耗連接初始化。
 *
 * @param connection BLE 通信对象。
 * @param usingDataPort 是否使用数据端口通信。
 * @return 一个初始化的 JsonConfig 实例。
 *
 * \~Chinese-Traditional
 * @brief 使用已開啟的 BleComm 藍牙低功耗連線初始化。
 *
 * @param connection BLE 通訊對象。
 * @param usingDataPort 是否使用資料端口通訊。
 * @return 一個初始化的 JsonConfig 實例。
 */
- (instancetype)initWithBleComm:(BleComm *)connection usingDataPort:(BOOL)usingDataPort;

/**
 * \~English
 * @brief Initializes a new instance with the given JSON communication object.
 *
 * @param mgmtComm The JSON communication object.
 * @return An initialized instance of JsonConfig, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用给定的 JSON 通信对象初始化一个新实例。
 *
 * @param mgmtComm JSON 通信对象。
 * @return 一个初始化的 JsonConfig 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用給定的 JSON 通信對象初始化一個新實例。
 *
 * @param mgmtComm JSON 通信對象。
 * @return 一個初始化的 JsonConfig 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithJsonComm:(JsonComm *)mgmtComm;

/**
 * \~English
 * @brief Constructor called when creating a connection managed by the SDK.
 *
 * @param commDescriptor Descriptor for the communication used to connect with printer, TCP descriptor format: "ip" or "ip:port" e.g. "127.0.0.1" or "127.0.0.1:3007". If port is not provided, default value is used.
 * @return An initialized instance of JsonConfig, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 在创建由 SDK 管理的连接时调用的构造函数。
 *
 * @param commDescriptor 用于连接打印机的通信描述符，TCP 描述符格式："ip" 或 "ip:port"，例如 "127.0.0.1" 或 "127.0.0.1:3007"。如果未提供端口，则使用默认值。
 * @return 一个初始化的 JsonConfig 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 在創建由 SDK 管理的連接時調用的構造函數。
 *
 * @param commDescriptor 用於連接打印機的通信描述符，TCP 描述符格式："ip" 或 "ip:port"，例如 "127.0.0.1" 或 "127.0.0.1:3007"。如果未提供端口，則使用默認值。
 * @return 一個初始化的 JsonConfig 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithCommDescriptor:(NSString *)commDescriptor;

// Destructor
/**
 * \~English
 * @brief Destructor for the JsonConfig object.
 *
 * \~Chinese
 * @brief JsonConfig 对象的析构函数。
 *
 * \~Chinese-Traditional
 * @brief JsonConfig 對象的析構函數。
 */
//- (void)dealloc;

// Methods

/**
 * \~English
 * @brief Gets all configurations.
 *
 * @return An array of Config objects representing all configurations.
 *
 * \~Chinese
 * @brief 获取所有配置。
 *
 * @return 一个表示所有配置的 Config 对象数组。
 *
 * \~Chinese-Traditional
 * @brief 獲取所有配置。
 *
 * @return 一個表示所有配置的 Config 對象數組。
 */
- (NSArray<Config *> *)getAllConfigWithTimeout:(int) timeout;

/**
 * \~English
 * @brief Gets the configuration with the specified number.
 *
 * @param cfgNumber The number of the configuration to retrieve.
 * @return The configuration object with the specified number.
 *
 * \~Chinese
 * @brief 获取具有指定编号的配置。
 *
 * @param cfgNumber 要检索的配置编号。
 * @return 具有指定编号的配置对象。
 *
 * \~Chinese-Traditional
 * @brief 獲取具有指定編號的配置。
 *
 * @param cfgNumber 要檢索的配置編號。
 * @return 具有指定編號的配置對象。
 */
- (Config *)getConfigWithNumber:(int)cfgNumber timeout:(int)timeout;

/**
 * \~English
 * @brief Sets the specified configuration.
 *
 * @param pCfg The Config object representing the configuration to set.
 * @return YES if the configuration was successfully set, NO otherwise.
 *
 * \~Chinese
 * @brief 设置指定的配置。
 *
 * @param pCfg 表示要设置的配置的 Config 对象。
 * @return 如果配置成功设置，则返回 YES；否则返回 NO。
 *
 * \~Chinese-Traditional
 * @brief 設置指定的配置。
 *
 * @param pCfg 表示要設置的配置的 Config 對象。
 * @return 如果配置成功設置，則返回 YES；否則返回 NO。
 */
- (BOOL)setConfig:(Config *)pCfg :(int)timeout;

@end
