//
//  RfidConvert.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/8.
//

#ifndef RfidConvert_h
#define RfidConvert_h

// RfidConvert.h
#import <Foundation/Foundation.h>

/**
 * @~english
 * @class RfidConvert
 * @brief Support methods to help convert non-byte data to order stored in RFID tags.
 *
 * Data is stored in the RFID tag in MSB to LSB. The PC running the SDK may be little or big endian and these support
 * methods help with conversions that may be affected by the endian of the platform running the SDK.
 *
 * @ingroup Interface
 *
 * @~chinese
 * @class RfidConvert
 * @brief 支持将非字节数据转换为RFID标签中存储的顺序的方法。
 *
 * 数据在RFID标签中以MSB到LSB存储。运行SDK的PC可能是小端或大端，这些支持方法有助于进行可能受运行SDK的平台字节序影响的转换。
 *
 * @ingroup Interface
 *
 * @~Chinese-Traditional
 * @class RfidConvert
 * @brief 支持將非字節數據轉換為RFID標籤中存儲的順序的方法。
 *
 * 數據在RFID標籤中以MSB到LSB存儲。運行SDK的PC可能是小端或大端，這些支持方法有助於進行可能受運行SDK的平台字節序影響的轉換。
 *
 * @ingroup Interface
 */
@interface RfidConvert : NSObject

/**
 * @~english
 * @brief Convert 4 bytes read from RFID tag into 32 bit unsigned integer.
 *
 * @~chinese
 * @brief 将从RFID标签读取的4个字节转换为32位无符号整数。
 *
 * @~Chinese-Traditional
 * @brief 將從RFID標籤讀取的4個字節轉換為32位無符號整數。
 */
+ (uint32_t)toUInt:(NSData *)bytesFromRfidTag;

/**
 * @~english
 * @brief Convert 2 bytes read from RFID tag into 16 bit unsigned integer.
 *
 * @~chinese
 * @brief 将从RFID标签读取的2个字节转换为16位无符号整数。
 *
 * @~Chinese-Traditional
 * @brief 將從RFID標籤讀取的2個字節轉換為16位無符號整數。
 */
+ (uint16_t)toUShort:(NSData *)bytesFromRfidTag;

/**
 * @~english
 * @brief Convert a 4 byte unsigned integer into hex string that can be written on RFID tag.
 *
 * @~chinese
 * @brief 将4字节无符号整数转换为可写入RFID标签的十六进制字符串。
 *
 * @~Chinese-Traditional
 * @brief 將4字節無符號整數轉換為可寫入RFID標籤的十六進制字符串。
 */
+ (NSString *)toHexFromUInt:(uint32_t)uintData;

/**
 * @~english
 * @brief Convert a 2 byte unsigned integer into hex string that can be written on RFID tag.
 *
 * @~chinese
 * @brief 将2字节无符号整数转换为可写入RFID标签的十六进制字符串。
 *
 * @~Chinese-Traditional
 * @brief 將2字節無符號整數轉換為可寫入RFID標籤的十六進制字符串。
 */
+ (NSString *)toHexFromUShort:(uint16_t)ushortData;

/**
 * @~english
 * @brief Convert a 32 bit unsigned integer into a byte array that can be written to RFID tag. Lower index has MSB.
 *
 * @~chinese
 * @brief 将32位无符号整数转换为可写入RFID标签的字节数组。低索引为MSB。
 *
 * @~Chinese-Traditional
 * @brief 將32位無符號整數轉換為可寫入RFID標籤的字節數組。低索引為MSB。
 */
+ (NSData *)toBytesFromUInt:(uint32_t)uintData;

/**
 * @~english
 * @brief Convert a 16 bit unsigned short integer into a byte array that can be written to RFID tag. Lower index has MSB.
 *
 * @~chinese
 * @brief 将16位无符号短整型转换为可写入RFID标签的字节数组。低索引为MSB。
 *
 * @~Chinese-Traditional
 * @brief 將16位無符號短整型轉換為可寫入RFID標籤的字節數組。低索引為MSB。
 */
+ (NSData *)toBytesFromUShort:(uint16_t)ushortData;

/**
 * @~english
 * @brief Convenience routine that takes and array of bytes and converts to hex string that can be written to RFID tag.
 *
 * @~chinese
 * @brief 便利方法，将字节数组转换为可写入RFID标签的十六进制字符串。
 *
 * @~Chinese-Traditional
 * @brief 便利方法，將字節數組轉換為可寫入RFID標籤的十六進制字符串。
 */
+ (NSString *)toHexFromBytes:(NSData *)bytes;

/**
 * @~english
 * @brief Convenience routine that takes an ASCII string and converts to hex.
 *
 * @~chinese
 * @brief 便利方法，将ASCII字符串转换为十六进制。
 *
 * @~Chinese-Traditional
 * @brief 便利方法，將ASCII字符串轉換為十六進制。
 */
+ (NSString *)toHexFromASCIIString:(NSString *)asciiString;

@end


#endif /* RfidConvert_h */
