﻿#include <cstdio>
#include <iostream>
#include <fstream>
#include <functional>
#include <vector>
#include <utility>
#include <string>
#include <limits>
#include <dlfcn.h> // 包含動態加載庫的頭文件
#include <unistd.h>
#include <termios.h>
#include <sys/select.h>


enum COMM_TYP
{
	TCP_COMM = 1
};


// TCP
enum DescriptorPortType {

	// Data
	DATA,
	// Management
	MGMT,
	// Status
	STATUS
};


struct PrinterInfo {
	std::string Model;
	std::string SerialNumber;
	std::string FirmwarePartNumber;
	std::string FirmwareVersion;
	std::string PrintheadResolution;
	bool HasRfidOption;
	bool HasOdvOption;
};


enum INFO_TYP
{
	ODV_TYP = 0,
	RFID_TYP,
	PRINTER_TYP,

};

using byte = unsigned char;

// CommSDK. dll export
typedef int (*BytesAvailableFunc)();
typedef bool (*ConnectedFunc)();
typedef void (*ReadFunc)(byte* buffer, int size);
typedef void (*WriteFunc)(byte* dataOut, int iDataSize);
typedef byte* (*WriteAndWaitForResponseFunc)(byte* dataOut, int iDataSize, int responseStartTimeOut, int responseEndTimeOut, char* completetionToken);
typedef void (*CloseFunc)();
typedef void (*OpenFunc)();
typedef void (*TcpConnectFunc)(char* ipAddress, int port);
typedef void (*GetCommFunc)(void*& ptrComm);


// Tool APIs
typedef void (*SendPrintFileFunc)(char* ipAddress, char* fileName);


// JsonSDK. dll export
typedef void (*MessengerGetFunc)(void* ptrComm, COMM_TYP commTyp, int maxInputMsgCapacity, bool usingDataPort);
typedef void (*MessengerReleaseFunc)();
typedef void (*MessengerSendMsgFunc)(char* strCommand, char* strContent);
typedef void (*MessengerReadNextMsgFunc)(const char** strNextMsg);
typedef void (*MessengerSendMsgAndWaitForResponseFunc)(char* strCommand, char* strContent, int maxWaitTimeSecs, const char** strResponse);
typedef int (*MessengerUnreadMsgCountFunc)();
typedef void (*PrinterMonitorConnectionFunc)(char* lpAddress, INFO_TYP InfoTyp);
typedef void (*GetPrinterInfoFunc)(PrinterInfo* pPrtInfo, INFO_TYP InfoTyp);
typedef void (*PrinterMonitorDisposeFunc)(INFO_TYP InfoTyp);
typedef bool (*GetEngineStatusListeningFunc)();
typedef void (*SetEngineStatusListeningFunc)(bool value);
typedef bool (*GetDisplayStatusListeningFunc)();
typedef void (*SetDisplayStatusListeningFunc)(bool value);
typedef bool (*GetAlertStatusListeningFunc)();
typedef void (*SetAlertStatusListeningFunc)(bool value);
typedef void (*SetEngineStatusCallbackFunc)(void callback(const std::string& engineState));
typedef void (*SetDisplayStatusCallbackFunc)(void callback(const std::vector<std::string>& newDisplayText));
typedef void (*SetAlertStatusCallbackFunc)(void callback(const std::vector<std::string>& alert));


BytesAvailableFunc pBytesAvailable = NULL;
ConnectedFunc pConnected = NULL;
ReadFunc pRead = NULL;
WriteFunc pWrite = NULL;
WriteAndWaitForResponseFunc pWriteAndWaitForResponse = NULL;
CloseFunc pClose = NULL;
OpenFunc pOpen = NULL;
TcpConnectFunc pTcpConnect = NULL;
GetCommFunc pGetComm = NULL;
SendPrintFileFunc pSendPrintFile = NULL;


MessengerGetFunc pMessengerGet = NULL;
MessengerReleaseFunc pMessengerRelease = NULL;
MessengerSendMsgFunc pMessengerSendMsg = NULL;
MessengerReadNextMsgFunc pMessengerReadNextMsg = NULL;
MessengerSendMsgAndWaitForResponseFunc pMessengerSendMsgAndWaitForResponse = NULL;
MessengerUnreadMsgCountFunc pMessengerUnreadMsgCount = NULL;


PrinterMonitorConnectionFunc pPrinterMonitorConnection = NULL;
GetPrinterInfoFunc pGetPrinterInfo = NULL;
PrinterMonitorDisposeFunc pPrinterMonitorDispose = NULL;
GetEngineStatusListeningFunc pGetEngineStatusListening = NULL;
SetEngineStatusListeningFunc pSetEngineStatusListening = NULL;
GetDisplayStatusListeningFunc pGetDisplayStatusListening = NULL;
SetDisplayStatusListeningFunc pSetDisplayStatusListening = NULL;
GetAlertStatusListeningFunc pGetAlertStatusListening = NULL;
SetAlertStatusListeningFunc pSetAlertStatusListening = NULL;
SetEngineStatusCallbackFunc pSetEngineStatusCallback = NULL;
SetDisplayStatusCallbackFunc pSetDisplayStatusCallback = NULL;
SetAlertStatusCallbackFunc pSetAlertStatusCallback = NULL;


PrinterInfo PrtInfo;
void ShowPrinterInfo(std::string strIP)
{
	pPrinterMonitorConnection((char*)strIP.c_str(), INFO_TYP::PRINTER_TYP);
	pGetPrinterInfo(&PrtInfo, INFO_TYP::PRINTER_TYP);

	std::cout << std::endl;
	std::cout << "Printer Model: " << PrtInfo.Model << std::endl;
	std::cout << "Printer SN: " << PrtInfo.SerialNumber << std::endl;
	std::cout << "Printer FW PN: " << PrtInfo.FirmwarePartNumber << std::endl;
	std::cout << "Printer FW Ver: " << PrtInfo.FirmwareVersion << std::endl;
	std::cout << "Printhead Resolution (Dots/Inch): " << PrtInfo.PrintheadResolution << std::endl;
	std::cout << std::endl;
	std::cout << "Has RFID: " << (PrtInfo.HasRfidOption ? "yes" : "no") << std::endl;
	std::cout << "Has ODV: " << (PrtInfo.HasOdvOption ? "yes" : "no") << std::endl;

}


void PtrAlertNoticeListener(const std::vector<std::string>& alert)
{
	// Print out alerts: e.g. "2418" ("Print Head Open" fault/alert)
	// "0000" = no alerts
	std::cout << "Printer Alert #: \n  " << alert[0] << " - " << alert[1] << "\n\n";
}

void PtrEngineStatusNoticeListener(const std::string& engineState)
{
	// Print out engine status: e.g. "idle", "offline", "online", "printing"...
	std::cout << "Engine Status: \n  " << engineState << "\n\n";
}

void PtrDisplayStatusNoticeListener(const std::vector<std::string>& newDisplayText)
{
	// Print display msgs: e.g. "ONLINE" "ETHERNET/PGL/LP+" or "PRINT HEAD UP" //"Close Print Head"
	std::cout << "Printer Display: \n";
	for (const std::string& txtLine : newDisplayText) {
		std::cout << "  " << txtLine << "\n";
	}
}


void WaitForSomethingToHappen_PrinterMonitor(const std::string& strIP) 
{
	std::string usage = "\nUsage: \r\n '1' print test file \r\n '0' quit\r\n Enter the number and then press <Enter>. Or, press 0 to exit. ";
	std::cout << usage;

	while (true) {
		fd_set fds;
		FD_ZERO(&fds);
		FD_SET(STDIN_FILENO, &fds); // Add stdin to the set

		struct timeval tv;
		tv.tv_sec = 0;
		tv.tv_usec = 100000; // Timeout of 100 milliseconds

		int ready = select(STDIN_FILENO + 1, &fds, NULL, NULL, &tv);
		if (ready == -1) {
			// Error occurred
			perror("select");
			return;
		}
		else if (ready > 0) {
			// Check if stdin is ready for reading
			if (FD_ISSET(STDIN_FILENO, &fds)) {
				char choice;
				std::cin >> choice;

				// Call the function associated with the selected test
				switch (choice) {
				case '0': // quit/exit
					return;

				case '1': // send test print job
					pSendPrintFile(const_cast<char*>(strIP.c_str()), const_cast<char*>("Hello_1.pgl"));
					break;

				default:
					std::cout << usage;
					break;
				}
			}
		}
	}
}


void PrinterMonitor(std::string strIP)
{
	try {
		pPrinterMonitorConnection((char*)strIP.c_str(), INFO_TYP::PRINTER_TYP);

		pSetAlertStatusListening(true);
		pSetAlertStatusCallback(PtrAlertNoticeListener);

		pSetEngineStatusListening(true);
		pSetEngineStatusCallback(PtrEngineStatusNoticeListener);

		pSetDisplayStatusListening(true);
		pSetDisplayStatusCallback(PtrDisplayStatusNoticeListener);


		WaitForSomethingToHappen_PrinterMonitor(strIP);
		

	}
	catch (const std::exception& e) {
		std::cerr << "Exception Msg: " << e.what() << std::endl;
	}


	pPrinterMonitorDispose(INFO_TYP::PRINTER_TYP);

}





void* handleCommSDKLib;
void* handleJsonSDKLib;

int main()
{
	// 加載 libCommSDK.so 庫
	handleCommSDKLib = dlopen("libCommSDK.so", RTLD_LAZY);
	if (!handleCommSDKLib) {
		std::cerr << "Failed to load libCommSDK.so: " << dlerror() << std::endl;
		return 1;
	}

	// 加載 libJsonSDK.so 庫
	handleJsonSDKLib = dlopen("libJsonSDK.so", RTLD_LAZY);
	if (!handleJsonSDKLib) {
		std::cerr << "Failed to load libCommSDK.so: " << dlerror() << std::endl;
		return 1;
	}


	pBytesAvailable = reinterpret_cast<BytesAvailableFunc>(dlsym(handleCommSDKLib, "BytesAvailable"));
	pConnected = reinterpret_cast<ConnectedFunc>(dlsym(handleCommSDKLib, "Connected"));
	pRead = reinterpret_cast<ReadFunc>(dlsym(handleCommSDKLib, "Read"));
	pWrite = reinterpret_cast<WriteFunc>(dlsym(handleCommSDKLib, "Write"));
	pWriteAndWaitForResponse = reinterpret_cast<WriteAndWaitForResponseFunc>(dlsym(handleCommSDKLib, "WriteAndWaitForResponse"));
	pClose = reinterpret_cast<CloseFunc>(dlsym(handleCommSDKLib, "Close"));
	pOpen = reinterpret_cast<OpenFunc>(dlsym(handleCommSDKLib, "Open"));
	pTcpConnect = reinterpret_cast<TcpConnectFunc>(dlsym(handleCommSDKLib, "TcpConnect"));
	pGetComm = reinterpret_cast<GetCommFunc>(dlsym(handleCommSDKLib, "GetComm"));
	pSendPrintFile = reinterpret_cast<SendPrintFileFunc>(dlsym(handleCommSDKLib, "SendPrintFile"));

	if (!pBytesAvailable || !pConnected || !pRead || !pWrite || 
		!pWriteAndWaitForResponse || !pOpen || !pTcpConnect ||
		!pGetComm || !pSendPrintFile)
	{
		std::cerr << "Failed to get address of libCommSDK.so API functions: " << dlerror() << std::endl;
		dlclose(handleCommSDKLib);
		return 1;
	}


	pMessengerGet = reinterpret_cast<MessengerGetFunc>(dlsym(handleJsonSDKLib, "MessengerGet"));
	pMessengerRelease = reinterpret_cast<MessengerReleaseFunc>(dlsym(handleJsonSDKLib, "MessengerRelease"));
	pMessengerSendMsg = reinterpret_cast<MessengerSendMsgFunc>(dlsym(handleJsonSDKLib, "MessengerSendMsg"));
	pMessengerReadNextMsg = reinterpret_cast<MessengerReadNextMsgFunc>(dlsym(handleJsonSDKLib, "MessengerReadNextMsg"));
	pMessengerSendMsgAndWaitForResponse = reinterpret_cast<MessengerSendMsgAndWaitForResponseFunc>(dlsym(handleJsonSDKLib, "MessengerSendMsgAndWaitForResponse"));
	pMessengerUnreadMsgCount = reinterpret_cast<MessengerUnreadMsgCountFunc>(dlsym(handleJsonSDKLib, "MessengerUnreadMsgCount"));

	
	pPrinterMonitorConnection = reinterpret_cast<PrinterMonitorConnectionFunc>(dlsym(handleJsonSDKLib, "PrinterMonitorConnection"));
	pGetPrinterInfo = reinterpret_cast<GetPrinterInfoFunc>(dlsym(handleJsonSDKLib, "GetPrinterInfo"));
	pPrinterMonitorDispose = reinterpret_cast<PrinterMonitorDisposeFunc>(dlsym(handleJsonSDKLib, "PrinterMonitorDispose"));
	pGetEngineStatusListening = reinterpret_cast<GetEngineStatusListeningFunc>(dlsym(handleJsonSDKLib, "GetEngineStatusListening"));
	pSetEngineStatusListening = reinterpret_cast<SetEngineStatusListeningFunc>(dlsym(handleJsonSDKLib, "SetEngineStatusListening"));
	pGetDisplayStatusListening = reinterpret_cast<GetDisplayStatusListeningFunc>(dlsym(handleJsonSDKLib, "GetDisplayStatusListening"));
	pSetDisplayStatusListening = reinterpret_cast<SetDisplayStatusListeningFunc>(dlsym(handleJsonSDKLib, "SetDisplayStatusListening"));
	pGetAlertStatusListening = reinterpret_cast<GetAlertStatusListeningFunc>(dlsym(handleJsonSDKLib, "GetAlertStatusListening"));
	pSetAlertStatusListening = reinterpret_cast<SetAlertStatusListeningFunc>(dlsym(handleJsonSDKLib, "SetAlertStatusListening"));
	pSetEngineStatusCallback = reinterpret_cast<SetEngineStatusCallbackFunc>(dlsym(handleJsonSDKLib, "SetEngineStatusCallback"));
	pSetDisplayStatusCallback = reinterpret_cast<SetDisplayStatusCallbackFunc>(dlsym(handleJsonSDKLib, "SetDisplayStatusCallback"));
	pSetAlertStatusCallback = reinterpret_cast<SetAlertStatusCallbackFunc>(dlsym(handleJsonSDKLib, "SetAlertStatusCallback"));
	

	if (!pMessengerGet || !pMessengerRelease || !pMessengerSendMsg || !pMessengerReadNextMsg || !pMessengerSendMsgAndWaitForResponse || !pMessengerUnreadMsgCount ||
		!pPrinterMonitorConnection || !pGetPrinterInfo || !pPrinterMonitorDispose || !pGetEngineStatusListening || !pSetEngineStatusListening || 
		!pGetDisplayStatusListening || !pSetDisplayStatusListening || !pGetAlertStatusListening || !pSetAlertStatusListening ||
		!pSetEngineStatusCallback || !pSetDisplayStatusCallback || !pSetAlertStatusCallback)
	{
		std::cerr << "Failed to get address of libJsonSDK.so API functions: " << dlerror() << std::endl;
		dlclose(handleJsonSDKLib);
		return 1;
	}

	std::string ptrIp = "10.0.10.180";
	ShowPrinterInfo(ptrIp);

	PrinterMonitor(ptrIp);

	

	pClose();
	

	// 釋放 libCommSDK.so / libJsonSDK.so 庫
	dlclose(handleCommSDKLib);
	dlclose(handleJsonSDKLib);
    return 0;

	
}