#ifndef MaxicodeModeEnum_h
#define MaxicodeModeEnum_h

#import <Foundation/Foundation.h>

/**
 * @~english
 * @class MaxicodeModeEnum
 * @brief Maxicode modes.
 *
 * Maxicode barcodes have two main message structures referred to as Primary Message and Secondary Message. The mode determines the content of the message structures and the Error Correction (EEC = Enhanced Error Correction, SEC = Standard Error Correction).
 *
 * Since each message section, Primary Message or Secondary Message, can have different levels of error correction, the data to be encoded into the Maxicode barcode is divided into message sections depending on the mode.
 *
 * Modes 2 & 3:
 *
 * Specific format for the Primary Message that requires:
 *
 * - postal code (6-9 chars)
 * - country code (3 char numeric)
 * - service class (3 char numeric)
 *
 * Secondary Message can be Free Format and is dependent on the application. i.e. UPS Shipping requires specific fields and field separators.
 *
 * @ingroup Interface
 *
 * @~chinese
 * @class MaxicodeModeEnum
 * @brief Maxicode 模式。
 *
 * Maxicode 条码有两个主要的信息结构，分别称为主要信息和次要信息。模式决定了信息结构的内容以及纠错级别（EEC = 增强纠错，SEC = 标准纠错）。
 *
 * 由于主要信息或次要信息的每个部分可以有不同的纠错级别，因此要编码到 Maxicode 条码中的数据会根据模式分为信息部分。
 *
 * 模式 2 和 3:
 *
 * 主要信息的特定格式要求：
 *
 * - 邮政编码（6-9 字符）
 * - 国家代码（3 位数字）
 * - 服务类别（3 位数字）
 *
 * 次要信息可以是自由格式，取决于应用程序。例如 UPS 运输需要特定的字段和字段分隔符。
 *
 * @ingroup Interface
 *
 * @~chinese_traditional
 * @class MaxicodeModeEnum
 * @brief Maxicode 模式。
 *
 * Maxicode 條碼有兩個主要的信息結構，分別稱為主要信息和次要信息。模式決定了信息結構的內容以及糾錯級別（EEC = 增強糾錯，SEC = 標準糾錯）。
 *
 * 由於主要信息或次要信息的每個部分可以有不同的糾錯級別，因此要編碼到 Maxicode 條碼中的數據會根據模式分為信息部分。
 *
 * 模式 2 和 3:
 *
 * 主要信息的特定格式要求：
 *
 * - 郵政編碼（6-9 字符）
 * - 國家代碼（3 位數字）
 * - 服務類別（3 位數字）
 *
 * 次要信息可以是自由格式，取決於應用程序。例如 UPS 運輸需要特定的字段和字段分隔符。
 *
 * @ingroup Interface
 *
 * @~english
 * | Value                           | Description                                              |
 * |---------------------------------|----------------------------------------------------------|
 * | MaxicodeModeEnumMode2            | Structured Carrier Message: 6-9 chars postal code (numeric). |
 * | MaxicodeModeEnumMode3          | Structured Carrier Message: 6 chars postal code (alphanumeric). |
 * | MaxicodeModeEnumMode4        | Standard Symbol. Unstructured data, general-use barcode. EEC Primary Message, SEC Secondary Message. |
 *
 * @~chinese
 * | 值                           | 描述                                                |
 * |---------------------------------|----------------------------------------------------------|
 * | MaxicodeModeEnumMode2            | 结构化承运人信息: 6-9 字符的邮政编码（数字）。 |
 * | MaxicodeModeEnumMode3          | 结构化承运人信息: 6 字符的邮政编码（字母数字）。 |
 * | MaxicodeModeEnumMode4        | 标准符号。非结构化数据，通用条码。EEC 主要信息，SEC 次要信息。 |
 *
 * @~chinese_traditional
 * | 值                           | 描述                                                |
 * |---------------------------------|----------------------------------------------------------|
 * | MaxicodeModeEnumMode2            | 結構化承運人信息: 6-9 字符的郵政編碼（數字）。 |
 * | MaxicodeModeEnumMode3          | 結構化承運人信息: 6 字符的郵政編碼（字母數字）。 |
 * | MaxicodeModeEnumMode4        | 標準符號。非結構化數據，通用條碼。EEC 主要信息，SEC 次要信息。 |
 */
typedef NS_ENUM(NSInteger, MaxicodeModeEnum) {
    MaxicodeModeEnumMode2 = 2,
    MaxicodeModeEnumMode3 = 3,
    MaxicodeModeEnumMode4 = 4
};

#endif /* MaxicodeModeEnum_h */
