//
//  PrinterMonitor.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/11.
//

#ifndef PrinterMonitor_h
#define PrinterMonitor_h


#endif /* PrinterMonitor_h */

#import <Foundation/Foundation.h>
#import <UniPRT/TcpComm.h>
#import <UniPRT/JsonComm.h>
#import <UniPRT/PrinterInfo.h>
#import <UniPRT/SynchronizationContext.h>
#import <UniPRT/JsonCommController.h>
#import <UniPRT/JsonStringBuilder.h>

/**
 * \~English
 * @class PrinterMonitor
 * @brief Module to facilitate listening/retrieval of printer unsolicited messaging or retrieval of printer status/information.
 *
 * Unsolicited messages, if enabled on printer, can be sent at any time. This module requires a constant connection to the printer's management port in order to listen for any messaging sent out by printer.
 * @ingroup Monitor
 *
 * \~Chinese
 * @class PrinterMonitor
 * @brief 模块用于方便监听/检索打印机的非请求消息或检索打印机状态/信息。
 *
 * 如果在打印机上启用了非请求消息，则可以随时发送。该模块需要与打印机的管理端口保持恒定的连接，以便监听打印机发送的任何消息。
 * @ingroup Monitor
 *
 * \~Chinese-Traditional
 * @class PrinterMonitor
 * @brief 模塊用於方便監聽/檢索打印機的非請求消息或檢索打印機狀態/信息。
 *
 * 如果在打印機上啟用了非請求消息，則可以隨時發送。該模塊需要與打印機的管理端口保持恆定的連接，以便監聽打印機發送的任何消息。
 * @ingroup Monitor
 */
@interface PrinterMonitor : NSObject

/** @cond */

/**
 * \~English
 * @brief The current engine status of the printer.
 *
 * \~Chinese
 * @brief 打印机的当前引擎状态。
 *
 * \~Chinese-Traditional
 * @brief 打印機的當前引擎狀態。
 */
@property (nonatomic, strong) NSString *engineStatus;

/**
 * \~English
 * @brief The current alert statuses of the printer.
 *
 * \~Chinese
 * @brief 打印机的当前警报状态。
 *
 * \~Chinese-Traditional
 * @brief 打印機的當前警報狀態。
 */
@property (nonatomic, strong) NSArray<NSString *> *alertStatus;

/**
 * \~English
 * @brief The current display statuses of the printer.
 *
 * \~Chinese
 * @brief 打印机的当前显示状态。
 *
 * \~Chinese-Traditional
 * @brief 打印機的當前顯示狀態。
 */
@property (nonatomic, strong) NSArray<NSString *> *displayStatus;

/**
 * \~English
 * @brief A constant ID that the class does not care about.
 *
 * \~Chinese
 * @brief 类不关心的常量 ID。
 *
 * \~Chinese-Traditional
 * @brief 類不關心的常量 ID。
 */
@property (nonatomic, strong) NSString *DONT_CARE_ID;

/** @endcond */

/**
 * \~English
 * @brief Enable/disable listening/parsing unsolicited engine status from printer.
 *
 * Note that this must be enabled in order to receive any notifications, EngineStatusCallback, from printer.
 *
 * \~Chinese
 * @brief 启用/禁用监听/解析打印机发送的非请求引擎状态。
 *
 * 请注意，必须启用此功能才能接收来自打印机的任何通知（EngineStatusCallback）。
 *
 * \~Chinese-Traditional
 * @brief 啟用/禁用監聽/解析打印機發送的非請求引擎狀態。
 *
 * 請注意，必須啟用此功能才能接收來自打印機的任何通知（EngineStatusCallback）。
 */
@property (nonatomic) BOOL engineStatusListening;

/**
 * \~English
 * @brief Enable/disable listening/parsing unsolicited front panel display text from printer.
 *
 * Note that this must be enabled in order to receive any notifications, DisplayStatusCallback, from printer.
 *
 * \~Chinese
 * @brief 启用/禁用监听/解析打印机发送的非请求前面板显示文本。
 *
 * 请注意，必须启用此功能才能接收来自打印机的任何通知（DisplayStatusCallback）。
 *
 * \~Chinese-Traditional
 * @brief 啟用/禁用監聽/解析打印機發送的非請求前面板顯示文本。
 *
 * 請注意，必須啟用此功能才能接收來自打印機的任何通知（DisplayStatusCallback）。
 */
@property (nonatomic) BOOL displayStatusListening;

/**
 * \~English
 * @brief Enable/disable listening/parsing unsolicited fault status alerts from printer.
 *
 * Note that this must be enabled in order to receive any notifications, AlertStatusCallback, from printer.
 *
 * \~Chinese
 * @brief 启用/禁用监听/解析打印机发送的非请求故障状态警报。
 *
 * 请注意，必须启用此功能才能接收来自打印机的任何通知（AlertStatusCallback）。
 *
 * \~Chinese-Traditional
 * @brief 啟用/禁用監聽/解析打印機發送的非請求故障狀態警報。
 *
 * 請注意，必須啟用此功能才能接收來自打印機的任何通知（AlertStatusCallback）。
 */
@property (nonatomic) BOOL alertStatusListening;

/**
 * \~English
 * @brief Initializes a new instance with the given TCP communication object.
 *
 * @param connection The TCP communication object.
 * @return An initialized instance of PrinterMonitor, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用给定的 TCP 通信对象初始化一个新实例。
 *
 * @param connection TCP 通信对象。
 * @return 一个初始化的 PrinterMonitor 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用給定的 TCP 通信對象初始化一個新實例。
 *
 * @param connection TCP 通信對象。
 * @return 一個初始化的 PrinterMonitor 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithTcpComm:(TcpComm *)connection;

/**
 * \~English
 * @brief Initializes a new instance with the given JSON communication object.
 *
 * @param mgmtComm The JSON communication object.
 * @return An initialized instance of PrinterMonitor, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用给定的 JSON 通信对象初始化一个新实例。
 *
 * @param mgmtComm JSON 通信对象。
 * @return 一个初始化的 PrinterMonitor 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用給定的 JSON 通信對象初始化一個新實例。
 *
 * @param mgmtComm JSON 通信對象。
 * @return 一個初始化的 PrinterMonitor 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithJsonComm:(JsonComm *)mgmtComm;

/**
 * \~English
 * @brief Constructor called when creating a connection managed by the SDK.
 *
 * commDescriptor is descriptor for the communication used to connect with printer.
 *
 * TCP descriptor format: "ip" or "ip:port"
 *
 * e.g. "127.0.0.1" or "127.0.0.1:3007"
 *
 * If port is not provided, default value is used.
 *
 * @param commDescriptor The communication descriptor.
 * @return An initialized instance of PrinterMonitor, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 创建由 SDK 管理的连接时调用的构造函数。
 *
 * commDescriptor 是用于连接打印机的通信描述符。
 *
 * TCP 描述符格式："ip" 或 "ip:port"
 *
 * 例如 "127.0.0.1" 或 "127.0.0.1:3007"
 *
 * 如果未提供端口，则使用默认值。
 *
 * @param commDescriptor 通信描述符。
 * @return 一个初始化的 PrinterMonitor 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 創建由 SDK 管理的連接時調用的構造函數。
 *
 * commDescriptor 是用於連接打印機的通信描述符。
 *
 * TCP 描述符格式："ip" 或 "ip:port"
 *
 * 例如 "127.0.0.1" 或 "127.0.0.1:3007"
 *
 * 如果未提供端口，則使用默認值。
 *
 * @param commDescriptor 通信描述符。
 * @return 一個初始化的 PrinterMonitor 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithCommDescriptor:(NSString *)commDescriptor;

/**
 * \~English
 * @brief Get printer information.
 *
 * Useful for cases where want to check model, firmware version, options installed and other hardware specific details.
 *
 * Implemented in PrinterMonitor.
 *
 * @return A PrinterInfo object containing the printer information.
 *
 * \~Chinese
 * @brief 获取打印机信息。
 *
 * 适用于需要检查型号、固件版本、安装选项和其他硬件特定详细信息的情况。
 *
 * 实现在 PrinterMonitor 中。
 *
 * @return 包含打印机信息的 PrinterInfo 对象。
 *
 * \~Chinese-Traditional
 * @brief 獲取打印機信息。
 *
 * 適用於需要檢查型號、固件版本、安裝選項和其他硬件特定詳細信息的情況。
 *
 * 實現在 PrinterMonitor 中。
 *
 * @return 包含打印機信息的 PrinterInfo 對象。
 */
- (PrinterInfo *)getPrinterInfo;

/**
 * \~English
 * @brief Query the printer for the engine status.
 *
 * Engine status is useful to check if printer is in a state that can print before sending a print job.
 *
 * The string returned is any of the "fault", "idle", "offline", "pause", "printing", "present" (Label Present).
 *
 * Implemented in PrinterMonitor.
 *
 * @return A string representing the engine status.
 *
 * \~Chinese
 * @brief 查询打印机的引擎状态。
 *
 * 引擎状态用于检查打印机在发送打印作业之前是否处于可打印状态。
 *
 * 返回的字符串可以是 "fault", "idle", "offline", "pause", "printing", "present"（标签存在）中的任何一个。
 *
 * 实现在 PrinterMonitor 中。
 *
 * @return 表示引擎状态的字符串。
 *
 * \~Chinese-Traditional
 * @brief 查詢打印機的引擎狀態。
 *
 * 引擎狀態用於檢查打印機在發送打印作業之前是否處於可打印狀態。
 *
 * 返回的字符串可以是 "fault", "idle", "offline", "pause", "printing", "present"（標籤存在）中的任何一個。
 *
 * 實現在 PrinterMonitor 中。
 *
 * @return 表示引擎狀態的字符串。
 */
- (NSString *)getEngineStatus;

/**
 * \~English
 * @brief Query the printer for fault status.
 *
 * Fault status is used to check if printer is in a fault state that needs attention or may prevent processing of any print job.
 *
 * The alert string array returned represents the alert number sent by printer at alert[0] and brief description at alert[1].
 *
 * e.g. "2418" represents a "Print Head Open" alert. alert[0] = "2418"; alert[1] = "Print Head Open";
 *
 * Implemented in PrinterMonitor.
 *
 * @return An array of strings representing the fault status.
 *
 * \~Chinese
 * @brief 查询打印机的故障状态。
 *
 * 故障状态用于检查打印机是否处于需要注意的故障状态，或者可能会阻止处理任何打印作业。
 *
 * 返回的警报字符串数组表示打印机在 alert[0] 处发送的警报编号以及 alert[1] 处的简短描述。
 *
 * 例如，"2418" 表示一个 "Print Head Open" 警报。alert[0] = "2418"; alert[1] = "Print Head Open";
 *
 * 实现在 PrinterMonitor 中。
 *
 * @return 表示故障状态的字符串数组。
 *
 * \~Chinese-Traditional
 * @brief 查詢打印機的故障狀態。
 *
 * 故障狀態用於檢查打印機是否處於需要注意的故障狀態，或者可能會阻止處理任何打印作業。
 *
 * 返回的警報字符串數組表示打印機在 alert[0] 處發送的警報編號以及 alert[1] 處的簡短描述。
 *
 * 例如，"2418" 表示一個 "Print Head Open" 警報。alert[0] = "2418"; alert[1] = "Print Head Open";
 *
 * 實現在 PrinterMonitor 中。
 *
 * @return 表示故障狀態的字符串數組。
 */
- (NSArray<NSString *> *)getFaultStatus;

/**
 * \~English
 * @brief Sets whether engine status listening is enabled.
 *
 * @param value YES to enable listening, NO to disable.
 *
 * \~Chinese
 * @brief 设置是否启用引擎状态监听。
 *
 * @param value YES 表示启用监听，NO 表示禁用。
 *
 * \~Chinese-Traditional
 * @brief 設置是否啟用引擎狀態監聽。
 *
 * @param value YES 表示啟用監聽，NO 表示禁用。
 */
- (void)SetEngineStatusListening:(BOOL)value;

/**
 * \~English
 * @brief Sets whether display status listening is enabled.
 *
 * @param value YES to enable listening, NO to disable.
 *
 * \~Chinese
 * @brief 设置是否启用显示状态监听。
 *
 * @param value YES 表示启用监听，NO 表示禁用。
 *
 * \~Chinese-Traditional
 * @brief 設置是否啟用顯示狀態監聽。
 *
 * @param value YES 表示啟用監聽，NO 表示禁用。
 */
- (void)SetDisplayStatusListening:(BOOL)value;

/**
 * \~English
 * @brief Sets whether alert status listening is enabled.
 *
 * @param value YES to enable listening, NO to disable.
 *
 * \~Chinese
 * @brief 设置是否启用警报状态监听。
 *
 * @param value YES 表示启用监听，NO 表示禁用。
 *
 * \~Chinese-Traditional
 * @brief 設置是否啟用警報狀態監聽。
 *
 * @param value YES 表示啟用監聽，NO 表示禁用。
 */
- (void)SetAlertStatusListening:(BOOL)value;

/**
 * \~English
 * @typedef EngineStatusCallback
 * @brief A callback type for engine status changes.
 *
 * @param newEngineState The new engine state.
 *
 * \~Chinese
 * @typedef EngineStatusCallback
 * @brief 引擎状态更改的回调类型。
 *
 * @param newEngineState 新的引擎状态。
 *
 * \~Chinese-Traditional
 * @typedef EngineStatusCallback
 * @brief 引擎狀態更改的回調類型。
 *
 * @param newEngineState 新的引擎狀態。
 */
typedef void (^EngineStatusCallback)(NSString *newEngineState);

/**
 * \~English
 * @typedef DisplayStatusCallback
 * @brief A callback type for display status changes.
 *
 * @param panelText The new panel text.
 *
 * \~Chinese
 * @typedef DisplayStatusCallback
 * @brief 显示状态更改的回调类型。
 *
 * @param panelText 新的面板文本。
 *
 * \~Chinese-Traditional
 * @typedef DisplayStatusCallback
 * @brief 顯示狀態更改的回調類型。
 *
 * @param panelText 新的面板文本。
 */
typedef void (^DisplayStatusCallback)(NSArray<NSString *> *panelText);

/**
 * \~English
 * @typedef AlertStatusCallback
 * @brief A callback type for alert status changes.
 *
 * @param alertText The new alert text.
 *
 * \~Chinese
 * @typedef AlertStatusCallback
 * @brief 警报状态更改的回调类型。
 *
 * @param alertText 新的警报文本。
 *
 * \~Chinese-Traditional
 * @typedef AlertStatusCallback
 * @brief 警報狀態更改的回調類型。
 *
 * @param alertText 新的警報文本。
 */
typedef void (^AlertStatusCallback)(NSArray<NSString *> *alertText);

/**
 * \~English
 * @brief Holds the function to call when engine status received from printer. Function must match signature of EngineStatusNotice.
 *
 * \~Chinese
 * @brief 当从打印机接收到引擎状态时调用的函数。函数必须符合 EngineStatusNotice 的签名。
 *
 * \~Chinese-Traditional
 * @brief 當從打印機接收到引擎狀態時調用的函數。函數必須符合 EngineStatusNotice 的簽名。
 */
@property (nonatomic, copy) EngineStatusCallback engineStatusCallback;

/**
 * \~English
 * @brief Holds the function to call when display text messages are received from printer. Function must match signature of DisplayStatusNotice.
 *
 * \~Chinese
 * @brief 当从打印机接收到显示文本消息时调用的函数。函数必须符合 DisplayStatusNotice 的签名。
 *
 * \~Chinese-Traditional
 * @brief 當從打印機接收到顯示文本消息時調用的函數。函數必須符合 DisplayStatusNotice 的簽名。
 */
@property (nonatomic, copy) DisplayStatusCallback displayStatusCallback;

/**
 * \~English
 * @brief Holds the function to call when fault/alert messages are received from printer. Function must match signature of AlertStatusNotice.
 *
 * \~Chinese
 * @brief 当从打印机接收到故障/警报消息时调用的函数。函数必须符合 AlertStatusNotice 的签名。
 *
 * \~Chinese-Traditional
 * @brief 當從打印機接收到故障/警報消息時調用的函數。函數必須符合 AlertStatusNotice 的簽名。
 */
@property (nonatomic, copy) AlertStatusCallback alertStatusCallback;

@end
