//
//  IBarWidths.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/7.
//

#ifndef IBarWidths_h
#define IBarWidths_h

#import <Foundation/Foundation.h>
#import <UniPRT/IRuler.h>

/**
 * @~english
 * @protocol IBarWidths
 * @brief Sets the Narrow/Wide bar widths.
 *
 * It should be noted that not all bar ratios are readable by barcode scanners. Typical Wide:Narrow ratios are 2:1, 2.5:1, 3:1, 4:1. A few test labels should be printed and read by the intended barcode scanner before committing to printing long batches of labels.
 *
 * @ingroup Interface
 *
 * @~chinese
 * @protocol IBarWidths
 * @brief 设置窄条和宽条的宽度。
 *
 * 需要注意的是，并非所有的条形比率都可以被条码扫描仪读取。典型的宽条:窄条比率为 2:1, 2.5:1, 3:1, 4:1。在打印大量标签之前，应该打印一些测试标签并用条码扫描仪读取。
 *
 * @ingroup Interface
 *
 * @~Chinese-Traditional
 * @protocol IBarWidths
 * @brief 設置窄條和寬條的寬度。
 *
 * 需要注意的是，並非所有的條形比率都可以被條碼掃描儀讀取。典型的寬條:窄條比率為 2:1, 2.5:1, 3:1, 4:1。在打印大量標籤之前，應該打印一些測試標籤並用條碼掃描儀讀取。
 *
 * @ingroup Interface
 */
@protocol IBarWidths <NSObject>

/**
 * @~english
 * @brief Ruler to use for bar widths.
 *
 * In some cases, it may be desirable to specify the bar widths in native dots to avoid rounding errors since the smallest print unit on a thermal printer is a dot and any bar widths that result in fractional dot printing will result in rounding error.
 *
 * Note that rounding errors can more drastically affect barcode readability on smaller X-dimension/(NarrowBar) barcodes due to rounding errors resulting in larger percentage inaccuracy as X-dimension gets smaller.
 *
 * @~chinese
 * @brief 用于条宽的标尺。
 *
 * 在某些情况下，最好以本机点数指定条的宽度，以避免舍入误差，因为热敏打印机上最小的打印单位是一个点，任何导致部分点打印的条宽都会导致舍入误差。
 *
 * 请注意，由于舍入误差会导致X尺寸（窄条）条码的百分比不准确性增大，因此舍入误差对较小的X尺寸（窄条）条码的可读性影响更大。
 *
 * @~Chinese-Traditional
 * @brief 用於條寬的標尺。
 *
 * 在某些情況下，最好以本機點數指定條的寬度，以避免捨入誤差，因為熱敏打印機上最小的打印單位是一個點，任何導致部分點打印的條寬都會導致捨入誤差。
 *
 * 請注意，由於捨入誤差會導致X尺寸（窄條）條碼的百分比不準確性增大，因此捨入誤差對較小的X尺寸（窄條）條碼的可讀性影響更大。
 */
@property (nonatomic, strong) id<IRuler> ruler;

/**
 * @~english
 * @brief Width of smallest bar. This is typically referred to as the X-dimension of the barcode.
 *
 * @~chinese
 * @brief 最小条的宽度。通常被称为条码的X尺寸。
 *
 * @~Chinese-Traditional
 * @brief 最小條的寬度。通常被稱為條碼的X尺寸。
 */
@property (nonatomic, assign) float narrowBar;

/**
 * @~english
 * @brief Width of widest bar. This is typically a multiple of the NarrowBar in order for barcode scanners to differentiate bar widths.
 *
 * @~chinese
 * @brief 最宽条的宽度。通常是窄条的倍数，以便条码扫描仪区分条宽。
 *
 * @~Chinese-Traditional
 * @brief 最寬條的寬度。通常是窄條的倍數，以便條碼掃描儀區分條寬。
 */
@property (nonatomic, assign) float wideBar;

@end

#endif /* IBarWidths_h */
