//
//  AAztecBarcode.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/7.
//

#ifndef AAztecBarcode_h
#define AAztecBarcode_h

#import <Foundation/Foundation.h>

#import <UniPRT/ABarcode2D.h>
#import <UniPRT/ISquareCell.h>
#import <UniPRT/AztecCodeTypeEnum.h>

/**
 * \~English
 * @class AAztecBarcode
 * @brief Base class for Aztec 2D barcode.
 * @ingroup Interface
 *
 * \~Chinese
 * @class AAztecBarcode
 * @brief Aztec 2D 条形码的基类。
 * @ingroup Interface
 *
 * \~Chinese-Traditional
 * @class AAztecBarcode
 * @brief Aztec 2D 條形碼的基類。
 * @ingroup Interface
 */
@interface AAztecBarcode : ABarcode2D <ISquareCell>

/**
 * \~English
 * @brief The cell size for the barcode.
 *
 * \~Chinese
 * @brief 条形码的单元格大小。
 *
 * \~Chinese-Traditional
 * @brief 條形碼的單元格大小。
 */
@property (nonatomic, strong) id<ICellSquare> cellSize;

/**
 * \~English
 * @brief The type of the Aztec barcode.
 *
 * \~Chinese
 * @brief Aztec 条形码的类型。
 *
 * \~Chinese-Traditional
 * @brief Aztec 條形碼的類型。
 */
@property (nonatomic, assign) AztecCodeTypeEnum type;

/**
 * \~English
 * @brief The number of layers to use for AztecCodeTypeEnum.Compact or AztecCodeTypeEnum.Full barcode types.
 *
 * Valid range: 1 - 4 for AztecCodeTypeEnum.Compact barcode type.
 *
 * Valid range: 1 - 32 for AztecCodeTypeEnum.Full barcode type.
 *
 * Any values outside of range will be ignored and limited to be within range.
 *
 * \~Chinese
 * @brief AztecCodeTypeEnum.Compact 或 AztecCodeTypeEnum.Full 条形码类型使用的层数。
 *
 * 有效范围：AztecCodeTypeEnum.Compact 条形码类型为 1 - 4。
 *
 * 有效范围：AztecCodeTypeEnum.Full 条形码类型为 1 - 32。
 *
 * 任何超出范围的值将被忽略并限制在范围内。
 *
 * \~Chinese-Traditional
 * @brief AztecCodeTypeEnum.Compact 或 AztecCodeTypeEnum.Full 條形碼類型使用的層數。
 *
 * 有效範圍：AztecCodeTypeEnum.Compact 條形碼類型為 1 - 4。
 *
 * 有效範圍：AztecCodeTypeEnum.Full 條形碼類型為 1 - 32。
 *
 * 任何超出範圍的值將被忽略並限制在範圍內。
 */
@property (nonatomic, assign) int layers;

/**
 * \~English
 * @brief This percent error correction value only applies to AztecCodeTypeEnum.FixedErrCorrection.
 *
 * Valid range: 1 - 99. Any values outside of range will be ignored and limited to be within range.
 *
 * \~Chinese
 * @brief 此百分比错误更正值仅适用于 AztecCodeTypeEnum.FixedErrCorrection。
 *
 * 有效范围：1 - 99。任何超出范围的值将被忽略并限制在范围内。
 *
 * \~Chinese-Traditional
 * @brief 此百分比錯誤更正值僅適用於 AztecCodeTypeEnum.FixedErrCorrection。
 *
 * 有效範圍：1 - 99。任何超出範圍的值將被忽略並限制在範圍內。
 */
@property (nonatomic, assign) int fixedErrCorrection;

/**
 * \~English
 * @brief Initializes a new instance of the AAztecBarcode class.
 *
 * \~Chinese
 * @brief 初始化 AAztecBarcode 类的新实例。
 *
 * \~Chinese-Traditional
 * @brief 初始化 AAztecBarcode 類的新實例。
 */
- (instancetype)init;

/**
 * \~English
 * @brief Initializes a new instance of the AAztecBarcode class with the specified start point and data.
 *
 * @param start The starting point of the Aztec barcode.
 * @param data The data to be encoded in the Aztec barcode.
 * @return A new instance of AAztecBarcode.
 *
 * \~Chinese
 * @brief 使用指定的起点和数据初始化 AAztecBarcode 类的新实例。
 *
 * @param start Aztec 条形码的起点。
 * @param data 要编码到 Aztec 条形码中的数据。
 * @return AAztecBarcode 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的起點和數據初始化 AAztecBarcode 類的新實例。
 *
 * @param start Aztec 條形碼的起點。
 * @param data 要編碼到 Aztec 條形碼中的數據。
 * @return AAztecBarcode 的新實例。
 */
- (instancetype)initWithStart:(id<IPoint>)start data:(NSString *)data NS_DESIGNATED_INITIALIZER;

/**
 * \~English
 * @brief Gets the number of layers within the valid range.
 *
 * @return The number of layers.
 *
 * \~Chinese
 * @brief 获取有效范围内的层数。
 *
 * @return 层数。
 *
 * \~Chinese-Traditional
 * @brief 獲取有效範圍內的層數。
 *
 * @return 層數。
 */
- (int)getLayersWithinRange;

/**
 * \~English
 * @brief Checks if the barcode has layers.
 *
 * @return YES if the barcode has layers, NO otherwise.
 *
 * \~Chinese
 * @brief 检查条形码是否有层。
 *
 * @return 如果条形码有层，则返回 YES，否则返回 NO。
 *
 * \~Chinese-Traditional
 * @brief 檢查條形碼是否有層。
 *
 * @return 如果條形碼有層，則返回 YES，否則返回 NO。
 */
- (BOOL)hasLayers;

/**
 * \~English
 * @brief Gets the error correction percentage.
 *
 * @return The error correction percentage.
 *
 * \~Chinese
 * @brief 获取错误更正百分比。
 *
 * @return 错误更正百分比。
 *
 * \~Chinese-Traditional
 * @brief 獲取錯誤更正百分比。
 *
 * @return 錯誤更正百分比。
 */
- (int)getErrCorrectionPercent;

@end

#endif /* AAztecBarcode_h */
