//
//  TcpComm.h
//  sdk_json_Objective-C_IOS
//
//  Created by Realbuber on 2024/4/23.
//

#ifndef TcpComm_h
#define TcpComm_h


#endif /* TcpComm_h */
#import <Foundation/Foundation.h>
#import <UniPRT/AComm.h>

/**
 * \~English
 * @brief Port for sending print data.
 *
 * \~Chinese
 * @brief 用于发送打印数据的端口。
 *
 * \~Chinese-Traditional
 * @brief 用於發送打印數據的端口。
 */
#define DEFAULT_DATA_PORT 9100

/**
 * \~English
 * @brief Port used to control and manage printers.
 *
 * \~Chinese
 * @brief 用于控制和管理打印机的端口。
 *
 * \~Chinese-Traditional
 * @brief 用於控制和管理打印機的端口。
 */
#define DEFAULT_MGMT_PORT 3007

/**
 * \~English
 * @brief Printer may be configured to send status messages to different ports based on the "Ret. Status Port" setting on the printer.
 *
 * If the "Ret. Status Port" is set to "E-NET Stat Port", the printer may send status messages to the "Status Port Number" setting which has a default of 3002.
 *
 * e.g. Emulation (printer language such as PGL) diagnostic messages from printer.
 *
 * e.g. Capturing ODV barcode data and grade with the following printer menu settings:
 *
 * - **System/Printer Mgmt/Ret. Status Port** set to **E-NET Stat Port**
 * - **System/Printer Mgmt/Status Port Number** set to **3002**
 * - **ODV2D/Control/Telemetry Data** set to **Full Report**
 * - **ODV2D/Control/Return Data** set to **Data+Grade** for listening to "barcode data + grade"
 *
 * \~Chinese
 * @brief 打印机可以配置为根据打印机上的 "Ret. Status Port" 设置将状态消息发送到不同的端口。
 *
 * 如果 "Ret. Status Port" 设置为 "E-NET Stat Port"，则打印机可能会将状态消息发送到 "Status Port Number" 设置，默认值为 3002。
 *
 * 例如：从打印机获取仿真（如 PGL）诊断消息。
 *
 * 例如：通过以下打印机菜单设置捕获 ODV 条码数据和等级：
 *
 * - **System/Printer Mgmt/Ret. Status Port** 设置为 **E-NET Stat Port**
 * - **System/Printer Mgmt/Status Port Number** 设置为 **3002**
 * - **ODV2D/Control/Telemetry Data** 设置为 **Full Report**
 * - **ODV2D/Control/Return Data** 设置为 **Data+Grade**，用于监听 "条码数据 + 等级"
 *
 * \~Chinese-Traditional
 * @brief 打印機可以配置為根據打印機上的 "Ret. Status Port" 設置將狀態消息發送到不同的端口。
 *
 * 如果 "Ret. Status Port" 設置為 "E-NET Stat Port"，則打印機可能會將狀態消息發送到 "Status Port Number" 設置，默認值為 3002。
 *
 * 例如：從打印機獲取仿真（如 PGL）診斷消息。
 *
 * 例如：通過以下打印機菜單設置捕獲 ODV 條碼數據和等級：
 *
 * - **System/Printer Mgmt/Ret. Status Port** 設置為 **E-NET Stat Port**
 * - **System/Printer Mgmt/Status Port Number** 設置為 **3002**
 * - **ODV2D/Control/Telemetry Data** 設置為 **Full Report**
 * - **ODV2D/Control/Return Data** 設置為 **Data+Grade**，用於監聽 "條碼數據 + 等級"
 */
#define DEFAULT_STATUS_PORT 3002

/**
 * \~English
 * @class TcpComm
 * @brief A class for TCP communication, inheriting from AComm and conforming to the NSStreamDelegate protocol.
 * @ingroup Comm
 *
 * \~Chinese
 * @class TcpComm
 * @brief 一个用于 TCP 通信的类，继承自 AComm 并符合 NSStreamDelegate 协议。
 * @ingroup Comm
 *
 * \~Chinese-Traditional
 * @class TcpComm
 * @brief 一個用於 TCP 通信的類，繼承自 AComm 並符合 NSStreamDelegate 協議。
 * @ingroup Comm
 */
@interface TcpComm : AComm <NSStreamDelegate>

/** @cond */
/**
 * \~English
 * @brief The input stream for reading data.
 *
 * \~Chinese
 * @brief 用于读取数据的输入流。
 *
 * \~Chinese-Traditional
 * @brief 用於讀取數據的輸入流。
 */
@property (nonatomic, readonly) NSInputStream *inputStream;

/**
 * \~English
 * @brief The output stream for writing data.
 *
 * \~Chinese
 * @brief 用于写入数据的输出流。
 *
 * \~Chinese-Traditional
 * @brief 用於寫入數據的輸出流。
 */
@property (nonatomic, readonly) NSOutputStream *outputStream;

/** @endcond */

/**
 * \~English
 * @brief A flag indicating whether the connection is established.
 *
 * \~Chinese
 * @brief 指示连接是否已建立的标志。
 *
 * \~Chinese-Traditional
 * @brief 指示連接是否已建立的標誌。
 */
@property (nonatomic, readwrite) BOOL connected;

/**
 * \~English
 * @brief Initializes a new instance with the given IP address and port.
 *
 * @param ipAddress The IP address of the TCP server.
 * @param port The port number of the TCP server.
 * @return An initialized instance of TcpComm, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用给定的 IP 地址和端口初始化一个新实例。
 *
 * @param ipAddress TCP 服务器的 IP 地址。
 * @param port TCP 服务器的端口号。
 * @return 一个初始化的 TcpComm 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用給定的 IP 地址和端口初始化一個新實例。
 *
 * @param ipAddress TCP 服務器的 IP 地址。
 * @param port TCP 服務器的端口號。
 * @return 一個初始化的 TcpComm 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithIPAddress:(NSString *)ipAddress port:(NSUInteger)port;

/**
 * \~English
 * @brief Opens the connection to the TCP server.
 *
 * \~Chinese
 * @brief 打开与 TCP 服务器的连接。
 *
 * \~Chinese-Traditional
 * @brief 打開與 TCP 服務器的連接。
 */
- (void)open;

/**
 * \~English
 * @brief Closes the connection to the TCP server.
 *
 * \~Chinese
 * @brief 关闭与 TCP 服务器的连接。
 *
 * \~Chinese-Traditional
 * @brief 關閉與 TCP 服務器的連接。
 */
- (void)close;

/**
 * \~English
 * @brief Writes data to the TCP server.
 *
 * @param text The data to write to the server.
 *
 * \~Chinese
 * @brief 将数据写入 TCP 服务器。
 *
 * @param text 要写入服务器的数据。
 *
 * \~Chinese-Traditional
 * @brief 將數據寫入 TCP 服務器。
 *
 * @param text 要寫入服務器的數據。
 */
- (void)write:(NSData *)text;

/**
 * \~English
 * @brief Reads data from the TCP server.
 *
 * @return The data read from the server.
 *
 * \~Chinese
 * @brief 从 TCP 服务器读取数据。
 *
 * @return 从服务器读取的数据。
 *
 * \~Chinese-Traditional
 * @brief 從 TCP 服務器讀取數據。
 *
 * @return 從服務器讀取的數據。
 */
- (NSData *)read;

/**
 * \~English
 * @brief Returns a descriptor string using the specified IP and port.
 *
 * @param ip The IP address to use in the descriptor.
 * @param port The port to use in the descriptor.
 * @return A descriptor string.
 *
 * \~Chinese
 * @brief 使用指定的 IP 和端口返回描述符字符串。
 *
 * @param ip 用于描述符的 IP 地址。
 * @param port 用于描述符的端口。
 * @return 一个描述符字符串。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的 IP 和端口返回描述符字符串。
 *
 * @param ip 用於描述符的 IP 地址。
 * @param port 用於描述符的端口。
 * @return 一個描述符字符串。
 */
- (NSString *)descriptorUseIp:(NSString *)ip descriptorUsePort:(NSString *)port;

/**
 * \~English
 * @brief Writes data to the TCP server and waits for a response.
 *
 * @param text The data to write to the server.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the server.
 *
 * \~Chinese
 * @brief 将数据写入 TCP 服务器并等待响应。
 *
 * @param text 要写入服务器的数据。
 * @param responseStartTimeOut 响应开始的超时间隔。
 * @param responseEndTimeOut 响应结束的超时间隔。
 * @param completionToken 表示响应完成的令牌。
 * @return 从服务器接收到的响应数据。
 *
 * \~Chinese-Traditional
 * @brief 將數據寫入 TCP 服務器並等待響應。
 *
 * @param text 要寫入服務器的數據。
 * @param responseStartTimeOut 響應開始的超時間隔。
 * @param responseEndTimeOut 響應結束的超時間隔。
 * @param completionToken 表示響應完成的令牌。
 * @return 從服務器接收到的響應數據。
 */
- (NSData *)writeAndWaitForResponse:(NSData *)text
               responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                 responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                    completionToken:(NSString *)completionToken;

/**
 * \~English
 * @brief Writes JSON data to the TCP server and waits for a response.
 *
 * @param text The JSON data to write to the server.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the server.
 *
 * \~Chinese
 * @brief 将 JSON 数据写入 TCP 服务器并等待响应。
 *
 * @param text 要写入服务器的 JSON 数据。
 * @param responseStartTimeOut 响应开始的超时间隔。
 * @param responseEndTimeOut 响应结束的超时间隔。
 * @param completionToken 表示响应完成的令牌。
 * @return 从服务器接收到的响应数据。
 *
 * \~Chinese-Traditional
 * @brief 將 JSON 數據寫入 TCP 服務器並等待響應。
 *
 * @param text 要寫入服務器的 JSON 數據。
 * @param responseStartTimeOut 響應開始的超時間隔。
 * @param responseEndTimeOut 響應結束的超時間隔。
 * @param completionToken 表示響應完成的令牌。
 * @return 從服務器接收到的響應數據。
 */
- (NSData *)writeAndWaitForResponseJson:(NSData *)text
                   responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                     responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                        completionToken:(NSString *)completionToken;

/**
 * \~English
 * @brief Types of descriptor ports.
 *
 * This enum defines the types of descriptor ports used in the system.
 *
 * Example usage:
 * @code
 * DescriptorPortType portType = DescriptorPortTypeData;
 * switch (portType) {
 *     case DescriptorPortTypeData:
 *         // Handle data port
 *         break;
 *     case DescriptorPortTypeMGMT:
 *         // Handle management port
 *         break;
 *     case DescriptorPortTypeStatus:
 *         // Handle status port
 *         break;
 * }
 * @endcode
 *
 * \~Chinese
 * @brief 描述符端口的类型。
 *
 * 此枚举定义了系统中使用的描述符端口类型。
 *
 * 示例用法：
 * @code
 * DescriptorPortType portType = DescriptorPortTypeData;
 * switch (portType) {
 *     case DescriptorPortTypeData:
 *         // 处理数据端口
 *         break;
 *     case DescriptorPortTypeMGMT:
 *         // 处理管理端口
 *         break;
 *     case DescriptorPortTypeStatus:
 *         // 处理状态端口
 *         break;
 * }
 * @endcode
 *
 * \~Chinese-Traditional
 * @brief 描述符端口的類型。
 *
 * 此枚舉定義了系統中使用的描述符端口類型。
 *
 * 示例用法：
 * @code
 * DescriptorPortType portType = DescriptorPortTypeData;
 * switch (portType) {
 *     case DescriptorPortTypeData:
 *         // 處理數據端口
 *         break;
 *     case DescriptorPortTypeMGMT:
 *         // 處理管理端口
 *         break;
 *     case DescriptorPortTypeStatus:
 *         // 處理狀態端口
 *         break;
 * }
 * @endcode
 */
typedef NS_ENUM(NSInteger, DescriptorPortType) {
    DescriptorPortTypeData,
    DescriptorPortTypeMGMT,
    DescriptorPortTypeStatus,
};

/** @cond */
/**
 * \~English
 * @brief Configures connection settings with the specified IP address and port.
 *
 * @param ip The IP address to use.
 * @param port The port to use.
 *
 * \~Chinese
 * @brief 使用指定的 IP 地址和端口配置连接设置。
 *
 * @param ip 要使用的 IP 地址。
 * @param port 要使用的端口。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的 IP 地址和端口配置連接設置。
 *
 * @param ip 要使用的 IP 地址。
 * @param port 要使用的端口。
 */
- (void)connectionSettingsWithIP:(NSString *)ip port:(NSUInteger)port;
/** @endcond */

/**
 * \~English
 * @brief Validates the descriptor with the specified hint and port type.
 *
 * @param descriptorHint A hint for the descriptor.
 * @param portTypeHint The type of port to validate.
 * @return A validated descriptor string.
 *
 * \~Chinese
 * @brief 使用指定的提示和端口类型验证描述符。
 *
 * @param descriptorHint 描述符的提示。
 * @param portTypeHint 要验证的端口类型。
 * @return 验证后的描述符字符串。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的提示和端口類型驗證描述符。
 *
 * @param descriptorHint 描述符的提示。
 * @param portTypeHint 要驗證的端口類型。
 * @return 驗證後的描述符字符串。
 */
- (NSString *)validateDescriptor:(NSString *)descriptorHint withPortType:(DescriptorPortType)portTypeHint;

/**
 * \~English
 * @brief Sends a print file to the specified IP address.
 *
 * @param ipAddress The IP address of the printer.
 * @param fileName The name of the file to print.
 *
 * \~Chinese
 * @brief 将打印文件发送到指定的 IP 地址。
 *
 * @param ipAddress 打印机的 IP 地址。
 * @param fileName 要打印的文件名。
 *
 * \~Chinese-Traditional
 * @brief 將打印文件發送到指定的 IP 地址。
 *
 * @param ipAddress 打印機的 IP 地址。
 * @param fileName 要打印的文件名。
 */
- (void)sendPrintFile:(NSString *)ipAddress fileName:(NSString *)fileName;

/**
 * \~English
 * @brief Sends a print string using **TSPL** or **PGL** to the specified IP address.
 *
 * @param ipAddress The IP address of the printer.
 * @param data The data to print.
 *
 * \~Chinese
 * @brief 使用 **TSPL** 或 **PGL** 将打印字符串发送到指定的 IP 地址。
 *
 * @param ipAddress 打印机的 IP 地址。
 * @param data 要打印的数据。
 *
 * \~Chinese-Traditional
 * @brief 使用 **TSPL** 或 **PGL** 將打印字符串發送到指定的 IP 地址。
 *
 * @param ipAddress 打印機的 IP 地址。
 * @param data 要打印的數據。
 */
- (void)sendPrintString:(NSString *)ipAddress data:(NSData *)data;

/**
 * \~English
 * @brief Writes a file to the printer.
 *
 * @param fileName The name of the file to write to the printer.
 *
 * \~Chinese
 * @brief 将文件写入打印机。
 *
 * @param fileName 要写入打印机的文件名。
 *
 * \~Chinese-Traditional
 * @brief 將文件寫入打印機。
 *
 * @param fileName 要寫入打印機的文件名。
 */
- (void)writePrinterFile:(NSString *)fileName;

@end
