//
//  AComm.h
//  sdk_json_Objective-C_IOS
//
//  Created by Realbuber on 2024/4/23.
//

#ifndef AComm_h
#define AComm_h


#endif /* AComm_h */
#import <Foundation/Foundation.h>
#import <UniPRT/IComm.h>

/**
 * @class AComm
 * @ingroup Comm
 *
 * \~English
 * @brief A base class for communication interfaces, conforming to the IComm protocol.
 *
 * \~Chinese
 * @brief 一个用于通信接口的基类，符合 IComm 协议。
 *
 * \~Chinese-Traditional
 * @brief 一個用於通信接口的基類，符合 IComm 協議。
 */
@interface AComm : NSObject <IComm>

/**
 * \~English
 * @brief The maximum packet size for communication.
 *
 * \~Chinese
 * @brief 通信的最大数据包大小。
 *
 * \~Chinese-Traditional
 * @brief 通信的最大數據包大小。
 */
@property (nonatomic, readonly) int maxPacketSize;

/**
 * \~English
 * @brief The descriptor for the communication interface.
 *
 * \~Chinese
 * @brief 通信接口的描述符。
 *
 * \~Chinese-Traditional
 * @brief 通信接口的描述符。
 */
@property (nonatomic, readonly) NSString *descriptor;

/**
 * \~English
 * @brief The number of bytes available to read.
 *
 * \~Chinese
 * @brief 可读取的字节数。
 *
 * \~Chinese-Traditional
 * @brief 可讀取的字節數。
 */
@property (nonatomic, readonly) int bytesAvailable;

/**
 * \~English
 * @brief A flag indicating whether the connection is established.
 *
 * \~Chinese
 * @brief 指示连接是否建立的标志。
 *
 * \~Chinese-Traditional
 * @brief 指示連接是否建立的標誌。
 */
@property (nonatomic) BOOL isConnected;

/**
 * \~English
 * @brief Closes the communication interface.
 *
 * \~Chinese
 * @brief 关闭通信接口。
 *
 * \~Chinese-Traditional
 * @brief 關閉通信接口。
 */
- (void)close;

/**
 * \~English
 * @brief Opens the communication interface.
 *
 * \~Chinese
 * @brief 打开通信接口。
 *
 * \~Chinese-Traditional
 * @brief 打開通信接口。
 */
- (void)open;

/**
 * \~English
 * @brief Reads data from the communication interface.
 *
 * @return The data read from the interface.
 *
 * \~Chinese
 * @brief 从通信接口读取数据。
 *
 * @return 从接口读取的数据。
 *
 * \~Chinese-Traditional
 * @brief 從通信接口讀取數據。
 *
 * @return 從接口讀取的數據。
 */
- (NSData *)read;

/**
 * \~English
 * @brief Writes data to the communication interface.
 *
 * @param dataOut The data to write to the interface.
 *
 * \~Chinese
 * @brief 将数据写入通信接口。
 *
 * @param dataOut 要写入接口的数据。
 *
 * \~Chinese-Traditional
 * @brief 將數據寫入通信接口。
 *
 * @param dataOut 要寫入接口的數據。
 */
- (void)write:(NSData *)dataOut;

/**
 * \~English
 * @brief Reads data into a binary writer (output stream).
 *
 * @param binDataIn The output stream to write the data into.
 *
 * \~Chinese
 * @brief 将数据读入二进制写入器（输出流）。
 *
 * @param binDataIn 要写入数据的输出流。
 *
 * \~Chinese-Traditional
 * @brief 將數據讀入二進制寫入器（輸出流）。
 *
 * @param binDataIn 要寫入數據的輸出流。
 */
- (void)readToBinaryWriter:(NSOutputStream *)binDataIn;

/**
 * \~English
 * @brief Waits for data to be available within a specified timeout.
 *
 * @param msTimeOut The timeout interval in milliseconds.
 *
 * \~Chinese
 * @brief 等待数据在指定的超时内可用。
 *
 * @param msTimeOut 超时间隔（毫秒）。
 *
 * \~Chinese-Traditional
 * @brief 等待數據在指定的超時內可用。
 *
 * @param msTimeOut 超時間隔（毫秒）。
 */
- (void)waitForDataWithTimeout:(NSTimeInterval)msTimeOut;

/**
 * \~English
 * @brief Writes data from a binary reader (input stream) to the communication interface.
 *
 * @param binReader The input stream to read the data from.
 *
 * \~Chinese
 * @brief 将数据从二进制读取器（输入流）写入通信接口。
 *
 * @param binReader 要从中读取数据的输入流。
 *
 * \~Chinese-Traditional
 * @brief 將數據從二進制讀取器（輸入流）寫入通信接口。
 *
 * @param binReader 要從中讀取數據的輸入流。
 */
- (void)writeFromBinaryReader:(NSInputStream *)binReader;

/**
 * \~English
 * @brief Writes data to the communication interface and waits for a response, using binary streams.
 *
 * @param binDataIn The output stream to write the data into.
 * @param binDataOut The input stream to read the response from.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 *
 * \~Chinese
 * @brief 将数据写入通信接口并等待响应，使用二进制流。
 *
 * @param binDataIn 要写入数据的输出流。
 * @param binDataOut 要从中读取响应的输入流。
 * @param responseStartTimeOut 响应开始的超时间隔。
 * @param responseEndTimeOut 响应结束的超时间隔。
 * @param completionToken 表示响应完成的令牌。
 *
 * \~Chinese-Traditional
 * @brief 將數據寫入通信接口並等待響應，使用二進制流。
 *
 * @param binDataIn 要寫入數據的輸出流。
 * @param binDataOut 要從中讀取響應的輸入流。
 * @param responseStartTimeOut 響應開始的超時間隔。
 * @param responseEndTimeOut 響應結束的超時間隔。
 * @param completionToken 表示響應完成的令牌。
 */
- (void)writeAndWaitForResponseToBinaryWriter:(NSOutputStream *)binDataIn
                              fromBinaryReader:(NSInputStream *)binDataOut
                          responseStartTimeout:(int)responseStartTimeOut
                            responseEndTimeout:(int)responseEndTimeOut
                               completionToken:(NSString *)completionToken;

/**
 * \~English
 * @brief Writes data to the communication interface and waits for a response.
 *
 * @param text The data to write to the interface.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the interface.
 *
 * \~Chinese
 * @brief 将数据写入通信接口并等待响应。
 *
 * @param text 要写入接口的数据。
 * @param responseStartTimeOut 响应开始的超时间隔。
 * @param responseEndTimeOut 响应结束的超时间隔。
 * @param completionToken 表示响应完成的令牌。
 * @return 从接口接收到的响应数据。
 *
 * \~Chinese-Traditional
 * @brief 將數據寫入通信接口並等待響應。
 *
 * @param text 要寫入接口的數據。
 * @param responseStartTimeOut 響應開始的超時間隔。
 * @param responseEndTimeOut 響應結束的超時間隔。
 * @param completionToken 表示響應完成的令牌。
 * @return 從接口接收到的響應數據。
 */
- (NSData *)writeAndWaitForResponse:(NSData *)text
               responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                 responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                    completionToken:(NSString *)completionToken;

@end
