//
//  AQRBarcode.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/8.
//

#ifndef AQRBarcode_h
#define AQRBarcode_h

#import <Foundation/Foundation.h>

#import <UniPRT/ABarcode2D.h>
#import <UniPRT/ISquareCell.h>
#import <UniPRT/QRCodeMaskEnum.h>
#import <UniPRT/QRCodeModelEnum.h>
#import <UniPRT/QRCodeErrorCorrectionEnum.h>

/**
 * \~English
 * @class AQRBarcode
 * @brief Base class for QR Code 2D barcode.
 *
 * Data encoded in a QR barcode can be encoded automatically or manually. Unless there is a need to optimize the amount of data stored in the barcode, automatic encoding should be used.
 *
 * \~Chinese
 * @class AQRBarcode
 * @brief 二维二维码的基类。
 *
 * 在二维码中编码的数据可以是自动编码的或手动编码的。除非需要优化存储在条形码中的数据量，否则应使用自动编码。
 *
 * \~Chinese-Traditional
 * @class AQRBarcode
 * @brief 二維二維碼的基類。
 *
 * 在二維碼中編碼的數據可以是自動編碼的或手動編碼的。除非需要優化存儲在條形碼中的數據量，否則應使用自動編碼。
 *
 * @ingroup Interface
 */
@interface AQRBarcode : ABarcode2D <ISquareCell>

/**
 * \~English
 * @brief The starting point for the QR code.
 *
 * \~Chinese
 * @brief 二维码的起始点。
 *
 * \~Chinese-Traditional
 * @brief 二維碼的起始點。
 */
@property (nonatomic, strong) id<IPoint> start;

/**
 * \~English
 * @brief Automatically encoded data. Manually, DataManuallyEncoded, encoded has priority over automatic encoding.
 *
 * Data can be encoded automatically or manually. If there is manually encoded data, this automatically encoded data will be ignored.
 *
 * \~Chinese
 * @brief 自动编码的数据。手动编码的 DataManuallyEncoded 优先于自动编码。
 *
 * 数据可以是自动编码的或手动编码的。如果有手动编码的数据，则此自动编码数据将被忽略。
 *
 * \~Chinese-Traditional
 * @brief 自動編碼的數據。手動編碼的 DataManuallyEncoded 優先於自動編碼。
 *
 * 數據可以是自動編碼的或手動編碼的。如果有手動編碼的數據，則此自動編碼數據將被忽略。
 */
@property (nonatomic, strong) NSString *data;

/**
 * \~English
 * @brief Manually encoded data. Manually encoded has priority over automatic encoding.
 *
 * Data can be encoded automatically or manually. If there is automatically encoded data, it is ignored and this manually encoded data will be used.
 *
 * For manually encoding data, a list of pairs of strings and their related encoding method can be provided using this property.
 *
 * \~Chinese
 * @brief 手动编码的数据。手动编码优先于自动编码。
 *
 * 数据可以是自动编码的或手动编码的。如果有自动编码的数据，则将忽略该数据，使用此手动编码的数据。
 *
 * 对于手动编码数据，可以使用此属性提供字符串对及其相关编码方法的列表。
 *
 * \~Chinese-Traditional
 * @brief 手動編碼的數據。手動編碼優先於自動編碼。
 *
 * 數據可以是自動編碼的或手動編碼的。如果有自動編碼的數據，則將忽略該數據，使用此手動編碼的數據。
 *
 * 對於手動編碼數據，可以使用此屬性提供字符串對及其相關編碼方法的列表。
 */
@property (nonatomic, strong) NSArray<NSArray *> *dataManuallyEncoded; // Array of tuples replaced by NSArray of NSArrays

/**
 * \~English
 * @brief Mask used in the QR code generation.
 *
 * \~Chinese
 * @brief 用于生成二维码的掩码。
 *
 * \~Chinese-Traditional
 * @brief 用於生成二維碼的掩碼。
 */
@property (nonatomic, assign) QRCodeMaskEnum mask;

/**
 * \~English
 * @brief Model of the QR code.
 *
 * \~Chinese
 * @brief 二维码的模型。
 *
 * \~Chinese-Traditional
 * @brief 二維碼的模型。
 */
@property (nonatomic, assign) QRCodeModelEnum model;

/**
 * \~English
 * @brief Error correction level of the QR code.
 *
 * \~Chinese
 * @brief 二维码的错误更正级别。
 *
 * \~Chinese-Traditional
 * @brief 二維碼的錯誤更正級別。
 */
@property (nonatomic, assign) QRCodeErrorCorrectionEnum errorCorrection;

/**
 * \~English
 * @brief Size of the cells that make up the QR code.
 *
 * \~Chinese
 * @brief 构成二维码的单元格的大小。
 *
 * \~Chinese-Traditional
 * @brief 構成二維碼的單元格的大小。
 */
@property (nonatomic, strong) id<ICellSquare> cellSize;

/**
 * \~English
 * @brief Used for most cases where data will be encoded automatically into barcode.
 *
 * \~Chinese
 * @brief 用于大多数数据将被自动编码到条形码的情况。
 *
 * \~Chinese-Traditional
 * @brief 用於大多數數據將被自動編碼到條形碼的情況。
 */
- (instancetype)initWithStart:(id<IPoint>)start data:(NSString *)data;

/**
 * \~English
 * @brief This is used for cases where manual encoding of barcode data is desired.
 *
 * \~Chinese
 * @brief 这用于需要手动编码条形码数据的情况。
 *
 * \~Chinese-Traditional
 * @brief 這用於需要手動編碼條形碼數據的情況。
 */
- (instancetype)initWithStart:(id<IPoint>)start manuallyEncodedData:(NSArray<NSArray *> *)dataManuallyEncoded;

@end

#endif /* AQRBarcode_h */
