//
//  TcpComm.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/3/25.
//

#import <Foundation/Foundation.h>
#import <AutoIdSDK/AComm.h>

#define DEFAULT_DATA_PORT 9100
#define DEFAULT_STATUS_PORT 3002
#define DEFAULT_MGMT_PORT 3007

/**
 * @class TcpComm
 * @brief A class for TCP communication, inheriting from AComm and conforming to the NSStreamDelegate protocol.
 * @ingroup Comm
 */
@interface TcpComm : AComm <NSStreamDelegate>

/**
 * @brief The input stream for reading data.
 */
@property (nonatomic, readonly) NSInputStream *inputStream;

/**
 * @brief The output stream for writing data.
 */
@property (nonatomic, readonly) NSOutputStream *outputStream;

/**
 * @brief A flag indicating whether the connection is established.
 */
@property (nonatomic, readwrite) BOOL connected;

/**
 * @brief Initializes a new instance with the given IP address and port.
 *
 * @param ipAddress The IP address of the TCP server.
 * @param port The port number of the TCP server.
 * @return An initialized instance of TcpComm, or nil if the object could not be created.
 */
- (instancetype)initWithIPAddress:(NSString *)ipAddress port:(NSUInteger)port;

/**
 * @brief Opens the connection to the TCP server.
 */
- (void)open;

/**
 * @brief Closes the connection to the TCP server.
 */
- (void)close;

/**
 * @brief Writes data to the TCP server.
 *
 * @param text The data to write to the server.
 */
- (void)write:(NSData *)text;

/**
 * @brief Reads data from the TCP server.
 *
 * @return The data read from the server.
 */
- (NSData *)read;

/**
 * @brief Returns a descriptor string using the specified IP and port.
 *
 * @param ip The IP address to use in the descriptor.
 * @param port The port to use in the descriptor.
 * @return A descriptor string.
 */
- (NSString *)descriptorUseIp:(NSString *)ip descriptorUsePort:(NSString *)port;

/**
 * @brief Writes data to the TCP server and waits for a response.
 *
 * @param text The data to write to the server.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the server.
 */
- (NSData *)writeAndWaitForResponse:(NSData *)text
               responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                 responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                    completionToken:(NSString *)completionToken;

/**
 * @brief Writes JSON data to the TCP server and waits for a response.
 *
 * @param text The JSON data to write to the server.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the server.
 */
- (NSData *)writeAndWaitForResponseJson:(NSData *)text
                   responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                     responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                        completionToken:(NSString *)completionToken;

/**
 * @brief Types of descriptor ports.
 */
typedef NS_ENUM(NSInteger, DescriptorPortType) {
    DescriptorPortTypeData,
    DescriptorPortTypeMGMT,
    DescriptorPortTypeStatus,
};

/**
 * @brief Configures connection settings with the specified IP address and port.
 *
 * @param ip The IP address to use.
 * @param port The port to use.
 */
- (void)connectionSettingsWithIP:(NSString *)ip port:(NSUInteger)port;

/**
 * @brief Validates the descriptor with the specified hint and port type.
 *
 * @param descriptorHint A hint for the descriptor.
 * @param portTypeHint The type of port to validate.
 * @return A validated descriptor string.
 */
- (NSString *)validateDescriptor:(NSString *)descriptorHint withPortType:(DescriptorPortType)portTypeHint;

/**
 * @brief Sends a print file to the specified IP address.
 *
 * @param ipAddress The IP address of the printer.
 * @param fileName The name of the file to print.
 */
- (void)sendPrintFile:(NSString *)ipAddress fileName:(NSString *)fileName;

/**
 * @brief Sends a print string to the specified IP address.
 *
 * @param ipAddress The IP address of the printer.
 * @param data The data to print.
 */
- (void)sendPrintString:(NSString *)ipAddress data:(NSData *)data;

/**
 * @brief Writes a file to the printer.
 *
 * @param fileName The name of the file to write to the printer.
 */
- (void)writePrinterFile:(NSString *)fileName;

@end


