import SwiftUI
import UniPRT

struct ContentView: View {
    @State private var ipText = "10.0.10.178"
    @State private var resultText = ""
    @State private var communicationType = 0
    @State private var inputText = ""
    
    @StateObject private var viewModel = CommunicationViewModel()

    var body: some View {

        NavigationSplitView {
            // Sidebar section
            VStack {
                Text("Communication Type")
                    .font(.title)
                    .padding()

                Picker("Select Type", selection: $communicationType) {
                    Text("TCP").tag(0)
                }
                .pickerStyle(SegmentedPickerStyle())
                .padding()

                TextField("Enter IP Address", text: $ipText)
                    .textFieldStyle(RoundedBorderTextFieldStyle())
                    .padding()
                
            }
            .font(.title)
            .padding()
            .background(Color.blue)
            .cornerRadius(10)
        } detail: {
            // Detail section
            ScrollView {
                VStack(alignment: .leading) {
                    TextEditor(text: $inputText)
                        .frame(height: 100)
                        .border(Color.secondary, width: 1)
                    
                    ScrollView {
                        Text(resultText)
                            .frame(maxWidth: .infinity, alignment: .leading)
                    }
                    .frame(height: 100)
                    .border(Color.secondary, width: 1)

                    HStack {
                        Button("Connect/Close") {
                            resultText = viewModel.connectCloseAction(ip: ipText)
                        }
                        Button("Open") {
                            resultText = viewModel.openAction()
                        }
                        Button("Write") {
                            resultText = viewModel.writeAction(stringData: inputText)
                        }
                        Button("Write and Wait for Response") {
                            resultText = viewModel.writeAndWaitForResponseAction(stringData: inputText)
                        }
                        Button("Read") {
                            resultText = viewModel.readAction()
                        }
                    }
                    .buttonStyle(BorderlessButtonStyle())
                }
                .padding()
            }
        }
        .frame(minWidth: 800, minHeight: 600)
    }
}

struct ContentView_Previews: PreviewProvider {
    static var previews: some View {
        ContentView()
    }
}

class CommunicationViewModel: ObservableObject {
    @Published var resultText = ""

    var tcpComm: TcpComm?

    func openAction() -> String {
        self.resultText = "open error"
        tcpComm?.open()
        self.resultText = "TCP Connection opened"
        return self.resultText
    }

    func writeAction(stringData: String) -> String {
        self.resultText = "write Error"
        tcpComm?.write(stringData.data(using: .utf8))
        self.resultText = "TCP Data written"
        return self.resultText
    }

    func readAction() -> String{
        if let data = tcpComm?.read() {
            self.resultText = String(data: data, encoding: .utf8) ?? "Failed to decode data"
            return self.resultText
        }
        return "Error Occured."
    }

    func connectCloseAction(ip: String) -> String {
        self.resultText = "connect error"
        if tcpComm != nil {
            tcpComm?.close()
            tcpComm = nil
            self.resultText = "TCP Connection closed"
        } else {
            tcpComm = TcpComm(ipAddress: ip , port: 9100)
            self.resultText = "TCP Reconnected"
        }
        return self.resultText
    }

    func writeAndWaitForResponseAction(stringData: String) -> String {
        if let response = tcpComm?.writeAndWait(forResponse: stringData.data(using: .utf8), responseStartTimeOut: 3000, responseEndTimeOut: 3000, completionToken: "\r\n") {
            self.resultText = String(data: response, encoding: .utf8) ?? "Failed to decode response"
            return  self.resultText
        }
        return "Error Occured."
    }
}

#Preview {
    ContentView()
}
