//
//  SettingsReadWrite.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/14.
//

#ifndef SettingsReadWrite_h
#define SettingsReadWrite_h


#endif /* SettingsReadWrite_h */
#import <Foundation/Foundation.h>
#import <AutoIdSDK/TcpComm.h>
#import <AutoIdSDK/JsonComm.h>
#import <AutoIdSDK/Setting.h>
#import <AutoIdSDK/JsonCommController.h>
#import <AutoIdSDK/Debug.h>
#import <AutoIdSDK/JsonStringBuilder.h>
#import <AutoIdSDK/JsonStringTokenizer.h>




/**
 * @class SettingsReadWrite
 * @brief A class for reading and writing settings over TCP or JSON communication.
 * @ingroup JSONMng
 */

/**
 * @brief Command item constant.
 */
extern NSString *const CMD_ITEM;

/**
 * @brief Command property constant.
 */
extern NSString *const CMD_PROP;

@interface SettingsReadWrite : NSObject

/**
 * @brief Initializes a new instance with the given TCP communication object.
 *
 * @param connection The TCP communication object.
 * @return An initialized instance of SettingsReadWrite, or nil if the object could not be created.
 */
- (instancetype)initWithTcpComm:(TcpComm *)connection;

/**
 * @brief Initializes a new instance with the given JSON communication object.
 *
 * @param mgmtComm The JSON communication object.
 * @return An initialized instance of SettingsReadWrite, or nil if the object could not be created.
 */
- (instancetype)initWithJsonComm:(JsonComm *)mgmtComm;

/**
 * @brief Initializes a new instance with the given communication descriptor.
 *
 * @param commDescriptor The communication descriptor.
 * @return An initialized instance of SettingsReadWrite, or nil if the object could not be created.
 */
- (instancetype)initWithCommDescriptor:(NSString *)commDescriptor;

/**
 * @brief Gets all settings values.
 *
 * @return A dictionary containing all settings keys and their corresponding values.
 */
- (NSDictionary<NSString *, NSString *> *)getAllValues;

/**
 * @brief Gets the value for a specific key.
 *
 * @param key The key for which to get the value.
 * @return The value associated with the specified key.
 */
- (NSString *)getValueForKey:(NSString *)key;

/**
 * @brief Gets the values for a list of keys.
 *
 * @param keys An array of keys for which to get the values.
 * @return A dictionary containing the specified keys and their corresponding values.
 */
- (NSDictionary<NSString *, NSString *> *)getValuesForKeys:(NSArray<NSString *> *)keys;

/**
 * @brief Sets the value for a specific key.
 *
 * @param value The value to set.
 * @param key The key for which to set the value.
 * @return YES if the value was successfully set, NO otherwise.
 */
- (BOOL)setValue:(NSString *)value forKey:(NSString *)key;

/**
 * @brief Sets the values for a list of keys.
 *
 * @param keysToSet A dictionary containing the keys and values to set.
 * @return YES if the values were successfully set, NO otherwise.
 */
- (BOOL)setValues:(NSDictionary<NSString *, NSString *> *)keysToSet;

/**
 * @brief Gets the properties for a specific key.
 *
 * @param key The key for which to get the properties.
 * @return A Setting object containing the properties for the specified key.
 */
- (Setting *)getPropertiesForKey:(NSString *)key;

/**
 * @brief Gets the properties for a list of keys.
 *
 * @param keys An array of keys for which to get the properties.
 * @return A dictionary containing the specified keys and their corresponding Setting objects.
 */
- (NSDictionary<NSString *, Setting *> *)getPropertiesForKeys:(NSArray<NSString *> *)keys;

/**
 * @brief Gets all properties for all keys.
 *
 * @return A dictionary containing all keys and their corresponding Setting objects.
 */
- (NSDictionary<NSString *, Setting *> *)getAllProperties;

@end

