from abc import ABC, abstractmethod
from typing import List, Optional, Protocol
from .ISettings import AlignEnum, IRotation, RotateEnum
from .ICoordinates import IPoint, IRuler, Points
from .IAlignment import IAlignment
from .IFont import IFont, IFontSize, IFontSizeUnits, IFontStyle, FontSize, FontSizeUnitsEnum, FontStyleEnum


class ITextItem(Protocol):
    """
    @protocol ITextItem
    @brief Specify text and position on plane where text starts.
    @ingroup Interface
    """
    start: IPoint
    data: str
    font_size: IFontSize


class ITextSettings(IFont, IFontStyle, IFontSizeUnits, IAlignment, IRotation, Protocol):
    """
    @protocol ITextSettings
    @brief A protocol defining the interface for TextSettings classes.
    @ingroup Interface
    """
    ruler: IRuler


class IText(ITextSettings, Protocol):
    """
    @protocol IText
    @brief A protocol defining the interface for Text classes.
    @ingroup Interface
    """
    text: List[ITextItem]

    @abstractmethod
    def __str__(self) -> str:
        pass


class ITextTwoByte(Protocol):
    """
    @protocol ITextTwoByte
    @brief A protocol for handling two-byte characters in text.
    """
    pass


class ATextItem(ITextItem):
    """
    @class ATextItem
    @brief Base class for ATextItem.
    @ingroup Interface
    """
    def __init__(self, start: IPoint, data: str):
        self.start = start
        self.data = data
        self._font_size = FontSize(0.0, 0.0)  # default font size

    @property
    def font_size(self) -> IFontSize:
        if not self._font_size:
            self._font_size = FontSize(0.0, 0.0)  # default no scaling
        return self._font_size

    @font_size.setter
    def font_size(self, font_size: IFontSize):
        self._font_size = font_size


class AText(IText):
    """
    @class AText
    @brief Base class for AText.
    @ingroup Interface
    """
    def __init__(self, text_item: Optional[ITextItem] = None):
        self.text: List[ITextItem] = []
        self.ruler: Optional[IRuler] = None
        self.font_name: Optional[str] = None
        self.font_style: FontStyleEnum = FontStyleEnum.NORMAL
        self.font_size_units: FontSizeUnitsEnum = FontSizeUnitsEnum.POINTS
        self.alignment: AlignEnum = AlignEnum.DEFAULT
        self.rotation: RotateEnum = RotateEnum.NONE

        if text_item:
            self.text.append(text_item)

    def __str__(self) -> str:
        return self.to_string_with_print_plane(None)

    def to_string_with_print_plane(self, print_plane: Optional[object]) -> str:
        return ""


class TextItem(ATextItem):
    """
    @class TextItem
    @brief A class for TextItem, inheriting from ATextItem.
    @ingroup Interface
    """
    def __init__(self, start: IPoint, data: str):
        super().__init__(start, data)

    @classmethod
    def with_font_size(cls, start: IPoint, font_size: IFontSize, data: str) -> 'TextItem':
        instance = cls(start, data)
        instance.font_size = font_size
        return instance

    @classmethod
    def with_coordinates(cls, x_start: float, y_start: float, data: str) -> 'TextItem':
        start_point = Points(x_start, y_start)
        return cls(start_point, data)

    @classmethod
    def with_coordinates_and_size(cls, x_start: float, y_start: float, size_x: float, size_y: float, data: str) -> 'TextItem':
        start_point = Points(x_start, y_start)
        font_size = FontSize(size_x, size_y)
        return cls.with_font_size(start_point, font_size, data)
