//
//  RfidMonitor.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/04/11.
//

#ifndef RfidMonitor_h
#define RfidMonitor_h

#import <Foundation/Foundation.h>
#import <AutoIdSDK/TcpComm.h>
#import <AutoIdSDK/JsonComm.h>
#import <AutoIdSDK/SynchronizationContext.h>
#import <AutoIdSDK/JsonCommController.h>
#import <AutoIdSDK/RfidReport.h>


/**
 * @class RfidMonitor
 * @brief A class for monitoring RFID reports and statuses.
 * @ingroup Monitor
 */
@interface RfidMonitor : NSObject

/**
 * @brief A constant ID that the class does not care about.
 */
@property (nonatomic, strong) NSString *DONT_CARE_ID;

/**
 * @brief The current engine status of the RFID system.
 */
@property (nonatomic, strong) NSString *engineStatus;

/**
 * @brief The current alert statuses of the RFID system.
 */
@property (nonatomic, strong) NSArray<NSString *> *alertStatus;

/**
 * @brief The current display statuses of the RFID system.
 */
@property (nonatomic, strong) NSArray<NSString *> *displayStatus;

/**
 * @brief The descriptor managed by the monitor.
 */
@property (nonatomic, strong) NSString *managedDescriptor;

/**
 * @brief A flag indicating whether RFID report listening is enabled.
 */
@property (nonatomic) BOOL rfidReportListening;

/**
 * @brief The thread for listening to RFID reports.
 */
@property (nonatomic, strong) NSThread *listenerThread;

/**
 * @brief The synchronization context.
 */
@property (nonatomic, strong) SynchronizationContext *syncContext;

/**
 * @brief Initializes a new instance with the given TCP communication object.
 *
 * @param connection The TCP communication object.
 * @return An initialized instance of RfidMonitor, or nil if the object could not be created.
 */
- (instancetype)initWithTcpComm:(TcpComm *)connection;

/**
 * @brief Initializes a new instance with the given JSON communication object.
 *
 * @param mgmtComm The JSON communication object.
 * @return An initialized instance of RfidMonitor, or nil if the object could not be created.
 */
- (instancetype)initWithJsonComm:(JsonComm *)mgmtComm;

/**
 * @brief Initializes a new instance with the given communication descriptor.
 *
 * @param commDescriptor The communication descriptor.
 * @return An initialized instance of RfidMonitor, or nil if the object could not be created.
 */
- (instancetype)initWithCommDescriptor:(NSString *)commDescriptor;

/**
 * @brief Sets whether RFID report listening is enabled.
 *
 * @param value YES to enable listening, NO to disable.
 */
- (void)SetRfidReportListening:(BOOL)value;

/**
 * @typedef RfidReportCallback
 * @brief A callback type for RFID report changes.
 *
 * @param report The new RFID report.
 */
typedef void (^RfidReportCallback)(RfidReport *report);

/**
 * @brief The callback for RFID report changes.
 */
@property (nonatomic, copy) RfidReportCallback rfidReportCallback;

@end

#endif /* RfidMonitor_h */


