//
//  APdf417.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/8.
//

#ifndef APdf417_h
#define APdf417_h

#import <Foundation/Foundation.h>
#import <UniPRT/ABarcode2D.h>
#import <UniPRT/IRectangleCell.h>
#import <UniPRT/Pdf417ErrCorrectionEnum.h>

/**
 * \~English
 * @class APdf417
 * @brief Base class for PDF417 barcode.
 * @ingroup Interface
 *
 * \~Chinese
 * @class APdf417
 * @brief PDF417条形码的基类。
 * @ingroup Interface
 *
 * \~Chinese-Traditional
 * @class APdf417
 * @brief PDF417條碼的基類。
 * @ingroup Interface
 */
@interface APdf417 : ABarcode2D <IRectangleCell>

/**
 * \~English
 * @brief The starting point of the PDF417 barcode.
 *
 * \~Chinese
 * @brief PDF417条形码的起始点。
 *
 * \~Chinese-Traditional
 * @brief PDF417條碼的起始點。
 */
@property (nonatomic, strong) id<IPoint> start;

/**
 * \~English
 * @brief The data to be encoded in the PDF417 barcode.
 *
 * \~Chinese
 * @brief 要在PDF417条形码中编码的数据。
 *
 * \~Chinese-Traditional
 * @brief 要在PDF417條碼中編碼的數據。
 */
@property (nonatomic, strong) NSString *data;

/**
 * \~English
 * @brief The cell size can be thought of as the width and height of the narrowest bar element within a PDF417 row.
 *
 * Changing the dimensions of the smallest element affects the overall width and height of the barcode.
 *
 * \~Chinese
 * @brief 单元格大小可以看作是PDF417行中最窄条元素的宽度和高度。
 *
 * 更改最小元素的尺寸会影响条形码的整体宽度和高度。
 *
 * \~Chinese-Traditional
 * @brief 單元格大小可以看作是PDF417行中最窄條元素的寬度和高度。
 *
 * 更改最小元素的尺寸會影響條碼的整體寬度和高度。
 */
@property (nonatomic, strong) id<ICellRect> cellSize;

/**
 * \~English
 * @brief Error correction level for the PDF417 barcode.
 *
 * \~Chinese
 * @brief PDF417条形码的错误校正级别。
 *
 * \~Chinese-Traditional
 * @brief PDF417條碼的錯誤校正級別。
 */
@property (nonatomic, assign) Pdf417ErrCorrectionEnum errorCorrection;

/**
 * \~English
 * @brief The number of rows can be used to limit the height of the barcode.
 *
 * A zero value indicates that rows are determined automatically.
 *
 * Barcode must be allowed to grow in width or height in order to fit data. For this reason, limiting the growth of the barcode can only be done in one direction either by limiting width or height via rows or columns.
 *
 * \~Chinese
 * @brief 行数可以用来限制条形码的高度。
 *
 * 零值表示行数是自动确定的。
 *
 * 条形码必须允许在宽度或高度上增长以适应数据。因此，限制条形码的增长只能在一个方向上进行，要么通过行数限制高度，要么通过列数限制宽度。
 *
 * \~Chinese-Traditional
 * @brief 行數可以用來限制條碼的高度。
 *
 * 零值表示行數是自動確定的。
 *
 * 條碼必須允許在寬度或高度上增長以適應數據。因此，限制條碼的增長只能在一個方向上進行，要麼通過行數限制高度，要麼通過列數限制寬度。
 */
@property (nonatomic, assign) int rows;

/**
 * \~English
 * @brief The number of columns can be used to limit the width of the barcode.
 *
 * A zero value indicates that columns are determined automatically.
 *
 * Barcode must be allowed to grow in width or height in order to fit data. For this reason, limiting the growth of the barcode can only be done in one direction either by limiting width or height via rows or columns.
 *
 * Note that additional non-data columns result from Start/Stop pattern and the Left/Right row indicators. This creates additional 4 columns.
 *
 * \~Chinese
 * @brief 列数可以用来限制条形码的宽度。
 *
 * 零值表示列数是自动确定的。
 *
 * 条形码必须允许在宽度或高度上增长以适应数据。因此，限制条形码的增长只能在一个方向上进行，要么通过行数限制高度，要么通过列数限制宽度。
 *
 * 请注意，由于开始/停止模式和左右行指示器，生成了额外的非数据列。这会创建额外的4列。
 *
 * \~Chinese-Traditional
 * @brief 列數可以用來限制條碼的寬度。
 *
 * 零值表示列數是自動確定的。
 *
 * 條碼必須允許在寬度或高度上增長以適應數據。因此，限制條碼的增長只能在一個方向上進行，要麼通過行數限制高度，要麼通過列數限制寬度。
 *
 * 請注意，由於開始/停止模式和左右行指示器，生成了額外的非數據列。這會創建額外的4列。
 */
@property (nonatomic, assign) int columns;

/**
 * \~English
 * @brief Initializes a new instance of the APdf417 class with a specified starting point and data.
 *
 * \~Chinese
 * @brief 使用指定的起始点和数据初始化APdf417类的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的起始點和數據初始化APdf417類的新實例。
 */
- (instancetype)initWithStart:(id<IPoint>)start data:(NSString *)data;

/**
 * \~English
 * @brief Limits a given value to a specified range.
 *
 * @param value The value to be limited.
 * @param min The minimum allowable value.
 * @param max The maximum allowable value.
 * @return The value constrained within the specified range.
 *
 * \~Chinese
 * @brief 将给定的值限制在指定范围内。
 *
 * @param value 要限制的值。
 * @param min 允许的最小值。
 * @param max 允许的最大值。
 * @return 约束在指定范围内的值。
 *
 * \~Chinese-Traditional
 * @brief 將給定的值限制在指定範圍內。
 *
 * @param value 要限制的值。
 * @param min 允許的最小值。
 * @param max 允許的最大值。
 * @return 約束在指定範圍內的值。
 */
- (int)limitRange:(int)value minimum:(int)min maximum:(int)max;

@end

#endif /* APdf417_h */
