//
//  QRBarcode.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/16.
//

#ifndef QRBarcode_h
#define QRBarcode_h

#import <Foundation/Foundation.h>
#import <UniPRT/AQRBarcode.h>
#import <UniPRT/IPoint.h>
#import <UniPRT/QRCodeManualEncodingEnum.h>
#import <UniPRT/QRCodeErrorCorrectionEnum.h>
#import <UniPRT/QRCodeModelEnum.h>
#import <UniPRT/RotateEnum.h>
#import <UniPRT/PrintPlane.h>
#import <UniPRT/TsplCmd.h>
#import <UniPRT/TSPL.h>

/**
 * \~English
 * @class QRBarcode
 * @brief A class representing a QR code barcode for TSPL.
 *
 * This class inherits from AQRBarcode and provides functionality for initializing and describing QR code barcodes in TSPL.
 *
 * @ingroup TSPL
 *
 * \~Chinese
 * @class QRBarcode
 * @brief 表示 TSPL 中 QR 码条形码的类。
 *
 * 该类继承自 AQRBarcode，提供了在 TSPL 中初始化和描述 QR 码条形码的功能。
 *
 * @ingroup TSPL
 *
 * \~Chinese-Traditional
 * @class QRBarcode
 * @brief 表示 TSPL 中 QR 碼條形碼的類。
 *
 * 該類繼承自 AQRBarcode，提供了在 TSPL 中初始化和描述 QR 碼條形碼的功能。
 *
 * @ingroup TSPL
 */
@interface QRBarcode : AQRBarcode

/**
 * \~English
 * @brief Initializes a new instance of the QRBarcode class.
 *
 * @return A new instance of QRBarcode.
 *
 * \~Chinese
 * @brief 初始化 QRBarcode 类的新实例。
 *
 * @return QRBarcode 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 初始化 QRBarcode 類的新實例。
 *
 * @return QRBarcode 的新實例。
 */
- (instancetype)init;

/**
 * \~English
 * @brief Initializes a new instance of the QRBarcode class with the specified start point and data.
 *
 * @param start The starting point of the QR code.
 * @param data The data to be encoded in the QR code.
 * @return A new instance of QRBarcode.
 *
 * \~Chinese
 * @brief 使用指定的起点和数据初始化 QRBarcode 类的新实例。
 *
 * @param start QR 码的起始点。
 * @param data 要在 QR 码中编码的数据。
 * @return QRBarcode 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的起點和數據初始化 QRBarcode 類的新實例。
 *
 * @param start QR 碼的起始點。
 * @param data 要在 QR 碼中編碼的數據。
 * @return QRBarcode 的新實例。
 */
- (instancetype)initWithStart:(id<IPoint>)start data:(NSString *)data;

/**
 * \~English
 * @brief Initializes a new instance of the QRBarcode class with the specified start point and manually encoded data.
 *
 * @param start The starting point of the QR code.
 * @param dataManuallyEncoded An array of manually encoded data points for the QR code.
 * @return A new instance of QRBarcode.
 *
 * \~Chinese
 * @brief 使用指定的起点和手动编码数据初始化 QRBarcode 类的新实例。
 *
 * @param start QR 码的起始点。
 * @param dataManuallyEncoded QR 码的手动编码数据点数组。
 * @return QRBarcode 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的起點和手動編碼數據初始化 QRBarcode 類的新實例。
 *
 * @param start QR 碼的起始點。
 * @param dataManuallyEncoded QR 碼的手動編碼數據點數組。
 * @return QRBarcode 的新實例。
 */
- (instancetype)initWithStart:(id<IPoint>)start manuallyEncodedData:(NSArray<NSArray *> *)dataManuallyEncoded;

/**
 * \~English
 * @brief Returns a string that can be sent to the printer.
 *
 * This method generates a string representation of the QRBarcode instance that
 * is formatted and ready to be sent to a printer. The string includes all the
 * necessary commands and data required for printing in TSPL.
 *
 * @return A string description of the QRBarcode instance.
 *
 * \~Chinese
 * @brief 返回可以发送到打印机的字符串。
 *
 * 该方法生成 QRBarcode 实例的字符串表示形式，
 * 格式化并准备好发送到打印机。字符串包括在 TSPL 中打印所需的所有命令和数据。
 *
 * @return QRBarcode 实例的字符串描述。
 *
 * \~Chinese-Traditional
 * @brief 返回可以發送到打印機的字符串。
 *
 * 該方法生成 QRBarcode 實例的字符串表示形式，
 * 格式化並準備好發送到打印機。字符串包括在 TSPL 中打印所需的所有命令和數據。
 *
 * @return QRBarcode 實例的字符串描述。
 */
- (NSString *)description;

@end

#endif /* QRBarcode_h */
