//
//  JsonStringTokenizer.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/2.
//

#ifndef JsonStringTokenizer_h
#define JsonStringTokenizer_h


#endif /* JsonStringTokenizer_h */

#import <Foundation/Foundation.h>

NS_ASSUME_NONNULL_BEGIN

/** @cond */

/**
 * @class JsonStringTokenizer
 * @brief A utility class for tokenizing and extracting information from JSON strings.
 * @ingroup JSONMng
 */
@interface JsonStringTokenizer : NSObject

/**
 * @brief Checks if a specified key exists in the given JSON string.
 *
 * @param key The key to check for.
 * @param jsonStr The JSON string to search in.
 * @return YES if the key exists, NO otherwise.
 */
+ (BOOL)hasKey:(NSString *)key inJsonString:(NSString *)jsonStr;

/**
 * @brief Checks if a specified key path exists in the given JSON string.
 *
 * @param keyPath The key path to check for.
 * @param jsonStr The JSON string to search in.
 * @return YES if the key path exists, NO otherwise.
 */
+ (BOOL)hasKeyAtPath:(NSString *)keyPath inJsonString:(NSString *)jsonStr;

/**
 * @brief Gets the value associated with a specified key from the given JSON string.
 *
 * @param key The key whose value is to be retrieved.
 * @param jsonStr The JSON string to search in.
 * @return The value associated with the key, or nil if the key does not exist.
 */
+ (NSString *)getKeyValue:(NSString *)key fromJsonString:(NSString *)jsonStr;

/**
 * @brief Gets the value associated with a specified key path from the given JSON string.
 *
 * @param keyPath The key path whose value is to be retrieved.
 * @param jsonStr The JSON string to search in.
 * @return The value associated with the key path, or nil if the key path does not exist.
 */
+ (NSString *)getKeyValueAtPath:(NSString *)keyPath fromJsonString:(NSString *)jsonStr;

/**
 * @brief Gets key-value pairs from a specified element list in the given JSON data.
 *
 * @param elemName The name of the element list.
 * @param jsonData The JSON data to search in.
 * @return A dictionary containing key-value pairs from the element list.
 */
+ (NSDictionary<NSString *, NSString *> *)getKeyValuePairsFromElementList:(NSString *)elemName inJsonData:(NSString *)jsonData;

/**
 * @brief Checks if a specified message in the given JSON string is a solicited message.
 *
 * @param msgStr The JSON string to search in.
 * @return YES if the message is solicited, NO otherwise.
 */
+ (BOOL)isSolicitedMessageInJson:(NSString *)msgStr;

/**
 * @brief Checks if a specified command in the given JSON string was successful.
 *
 * @param msgStr The JSON string to search in.
 * @return YES if the command was successful, NO otherwise.
 */
+ (BOOL)isCommandSuccessInJson:(NSString *)msgStr;

/**
 * @brief Gets the message ID from the given JSON string.
 *
 * @param msgStr The JSON string to search in.
 * @return The message ID, or nil if the message ID does not exist.
 */
+ (NSString *)getMessageIdFromJson:(NSString *)msgStr;

/**
 * @brief Gets the expected message ID for a response from the given JSON string.
 *
 * @param msgStr The JSON string to search in.
 * @return The expected message ID for the response, or nil if it does not exist.
 */
+ (NSString *)getExpectedMessageIdOnResponseFromJson:(NSString *)msgStr;

@end

/** @endcond */

NS_ASSUME_NONNULL_END
