import SwiftUI
import UniPRT

struct ContentView: View {
    @State private var resultText: String = ""
    
    let buttonTitles = ["Pdf417", "SimpleLabel", "RfidEncode", "Maxicodes", "DataMatrix", "Aztec", "QRCode", "PglPdf417", "PglSimpleLabel", "PglRfidEncode", "PglMaxicodes", "PglDataMatrix", "PglAztec", "PglQRCode", "PglRulerLabel"]
    
    var body: some View {
        VStack {
            ScrollView {
                LazyVGrid(columns: [GridItem(.flexible()), GridItem(.flexible()), GridItem(.flexible())], spacing: 10) {
                    ForEach(buttonTitles.indices, id: \.self) { index in
                        Button(action: {
                            buttonTapped(index)
                        }) {
                            Text(buttonTitles[index])
                                .frame(width: UIScreen.main.bounds.width / 3 - 20, height: 50)
                                .background(Color.gray)
                                .foregroundColor(.white)
                                .cornerRadius(5)
                        }
                    }
                }
            }
            .padding(.top, 100)
            
            TextEditor(text: $resultText)
                .frame(height: 450)
                .background(Color.accentColor)
                .cornerRadius(5)
                .padding(.horizontal, 20)
        }
        .background(Color.white)
        .edgesIgnoringSafeArea(.all)
    }
    
    func buttonTapped(_ index: Int) {
        var result: UniPRT.Label?
        var pglResult: PglLabel?
        
        switch index {
        case 0:
            result = BcdPdf417()
        case 1:
            result = SimpleTextLabel(name: "Mr. Milky Cheese", address: "123 No Way Road")
        case 2:
            result = RfidEncode()
        case 3:
            result = BcdMaxicodes()
        case 4:
            result = BcdDataMatrix()
        case 5:
            result = BcdAztec()
        case 6:
            result = BcdQRCode()
        case 7:
            pglResult = PglBcdPdf417()
        case 8:
            pglResult = PglSimpleLabel(name: "Mr. Milky Cheese", address: "123 No Way Road")
        case 9:
            pglResult = PglRfidEncode()
        case 10:
            pglResult = PglBcdMaxicodes()
        case 11:
            pglResult = PglBcdDataMatrix()
        case 12:
            pglResult = PglBcdAztec()
        case 13:
            pglResult = PglBcdQRCode()
        case 14:
            pglResult = RuleredLabel(width: 4.0, length: 6.0, inchUnits: true, rulerMargin: 1.0 / 8.0)
        default:
            break
        }
        
        if let result = result {
            resultText = result.description
        } else if let pglResult = pglResult {
            resultText = pglResult.description
        } else {
            resultText = ""
        }
    }
    
    // Add your label generating functions here, translated from Objective-C to Swift
}

struct ContentView_Previews: PreviewProvider {
    static var previews: some View {
        ContentView()
    }
}

func BcdPdf417() -> UniPRT.Label? {
    
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))
    
    let lbl = Label(name: "Pdf417Bcodes")
    
    let someText = "The happiness in your life depends on the quality of your thoughts. --Marcus Aurelius"
    let someShortText = "PI = 3.1415"
    
    let bcdDefault = Pdf417Barcode(start: Points(x: 0.25, y: 0.50), data: someText)
    bcdDefault?.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    
    let bcdErrCorrectionLvl0 = Pdf417Barcode(start: Points(x: 0.25, y: 1.50), data: someShortText)
    bcdErrCorrectionLvl0?.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdErrCorrectionLvl0?.errorCorrection = .level0
    
    let bcdErrCorrectionLvl5 = Pdf417Barcode(start: Points(x: 0.25, y: 2.00), data: someShortText)
    bcdErrCorrectionLvl5?.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdErrCorrectionLvl5?.errorCorrection = .level5
    
    let bcdRowsLimited = Pdf417Barcode(start: Points(x: 0.25, y: 3.00), data: someShortText)
    bcdRowsLimited?.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdRowsLimited?.rows = 15
    
    let bcdColsLimited = Pdf417Barcode(start: Points(x: 0.25, y: 4.00), data: someShortText)
    bcdColsLimited?.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdColsLimited?.columns = 5
    
    lbl!.add(bcdDefault)
    lbl!.add(bcdErrCorrectionLvl0)
    lbl!.add(bcdErrCorrectionLvl5)
    lbl!.add(bcdRowsLimited)
    lbl!.add(bcdColsLimited)
    
    return lbl!
}

func SimpleTextLabel(name: String, address: String) -> UniPRT.Label? {
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 600, unit: .inch))
    
    let lbl = Label(name: "SimpleLabel")
    let inchRuler = Ruler(scale: .inch)
    
    let line1 = _Line(start: Points(x: 2.5, y: 1.0 / 16.0), end: Points(x: 2.5, y: 1.0), lineThickness: 1.0 / 32.0)
    line1!.ruler = inchRuler
    lbl!.add(line1)
    
    let line2 = _Line(start: Points(x: 0.12, y: 1.0), end: Points(x: 3.88, y: 1.0), lineThickness: 1.0 / 32.0)
    line2!.ruler = inchRuler
    lbl!.add(line2)
    
    let line3 = _Line(start: Points(x: 0.12, y: 3.5), end: Points(x: 3.88, y: 3.5), lineThickness: 1.0 / 32.0)
    line3!.ruler = inchRuler
    lbl!.add(line3)
    
    let line4 = _Box(start: Points(x: 0.5, y: 1.25), end: Points(x: 3.5, y: 2.25), lineThickness: 1.0 / 16.0)
    line4!.ruler = inchRuler
    lbl!.add(line4)
    
    let barcodeItem128 = BarcodeItem(start: Points(x: 0.5, y: 1.0 + 1.5 + 1.0 / 4.0 + 1.2), height: 1.2, data: "Code 128")
    let bcd128 = Barcode_1D(barcodeItem: barcodeItem128)
    bcd128!.barcodeType = .dCode128
    bcd128!.printHumanReadable = true
    bcd128!.rotation = .none
    bcd128!.ruler = inchRuler
    bcd128!.barWidths = BarWidths(narrowBar: 0.015, wideBar: 0.015)
    bcd128!.barWidths?.ruler = Ruler(scale: .inch)
    lbl!.add(bcd128)
    
    let barcodeItem93 = BarcodeItem(start: Points(x: 0.5, y: 3.5 - 1.0 / 8.0 - 0.6), height: 0.6, data: "CODE 93")
    let bcd93 = Barcode_1D(barcodeItem: barcodeItem93)
    bcd93!.barcodeType = .dCode93
    bcd93!.printHumanReadable = true
    bcd93!.rotation = .none
    bcd93!.ruler = inchRuler
    bcd93!.barWidths = BarWidths(narrowBar: 0.025, wideBar: 0.025 * 3)
    bcd93!.barWidths?.ruler = Ruler(scale: .inch)
    lbl!.add(bcd93)
    
    return lbl!
}

func RfidEncode() -> UniPRT.Label? {
    let lbl = Label(name: "Rfidlbl!")

    let a32BitField: UInt32 = 0x11223344
    let a16BitField: UInt16 = 0xBEEF
    let a6CharAsciiString = "MyData"
    var epcHexData = RfidConvert.toHex(from: a32BitField)
    epcHexData! += RfidConvert.toHex(fromBytes: a6CharAsciiString.data(using: .ascii)!)
    epcHexData! += RfidConvert.toHex(fromUShort: a16BitField)

    let epc = Rfid_Write(memBlock: .EPC, data: epcHexData)
    lbl!.add(epc)

    var userHexData = RfidConvert.toHex(fromASCIIString: "MyUserData")
    userHexData! += "0ABCDE0F"
    let userMem = Rfid_Write(memBlock: .user, data: userHexData)
    userMem!.offsetFromStart = 2
    lbl!.add(userMem)

    return lbl!
}

func BcdMaxicodes() -> UniPRT.Label? {
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 203, unit: .inch))
    let lbl = Label(name: "MaxiBcds")

    let maxiDataStructCarrier = MaxicodeMsgStructured(mode: .mode2, postalCode: "902557317", countryCode: "800", serviceClass: "200", remainingMsg: "Maxicode Carrier Standard")
    let maxicodeBarcodeSc = MaxicodeBarcode(start: Points(x: 0.5, y: 0.5), data: maxiDataStructCarrier)
    maxicodeBarcodeSc!.ruler = Ruler(scale: .inch)

    let maxiData = MaxicodeMsg(mode: .mode4, primaryMsg: "123456789", remainingMsg: "Maxicode unstructured")
    let maxicodeBarcode = MaxicodeBarcode(start: Points(x: 0.5, y: 3.5), data: maxiData)
    maxicodeBarcode!.ruler = Ruler(scale: .inch)

    lbl!.add(maxicodeBarcodeSc)
    lbl!.add(maxicodeBarcode)

    return lbl!
}

func BcdDataMatrix() -> UniPRT.Label? {

    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 600, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))
    
    let lbl = Label(name: "DMatrixBcds")


    let dfltMatrix = DataMatrixBarcode(start: Points(x: 0.25, y: 0.25), data: "Default DataMatrix")

    let rectMatrix = DataMatrixBarcode(start: Points(x: 1.25, y: 0.25), data: "Rectangular DataMatrix")
    rectMatrix!.rotation = .counterClockwise
    rectMatrix!.rectangle = true
    rectMatrix!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let dfltMatrixMultiLine = DataMatrixBarcode(start: Points(x: 2.25, y: 0.25), data: "Line 1 DataMatrix")
    let Eol = dfltMatrixMultiLine!.ctrlChar(0x0D) + dfltMatrixMultiLine!.ctrlChar(0x0A) // CR + LF
    dfltMatrixMultiLine!.data += Eol + "Line 2 content" + Eol + "Line 3 content"

    let rectMatrixUserDefinedDimensions = DataMatrixBarcode(start: Points(x: 1.25, y: 1.75), data: "DataMatrix with user defined dimensions")
    rectMatrixUserDefinedDimensions!.rectangle = true
    rectMatrixUserDefinedDimensions!.rowsCols = CGSize(width: 16, height: 36) as NSValue
    rectMatrixUserDefinedDimensions!.cellSize = CellSquare(xDim: 0.030, ruler: Ruler(scale: .inch))

    lbl!.add(dfltMatrix)
    lbl!.add(rectMatrix)
    lbl!.add(dfltMatrixMultiLine)
    lbl!.add(rectMatrixUserDefinedDimensions)

    return lbl!
}

func BcdAztec() -> UniPRT.Label? {

    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))
    
    let lbl = Label(name: "AztecBcodes")


    let someText = "Mr. AirTraveler, seat A, flight 200"

    let bcdDefault = AztecBarcode(start: Points(x: 0.25, y: 1.0), data: someText)
    bcdDefault!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let bcdFixedErrCorr = AztecBarcode(start: Points(x: 1.5, y: 1.0), data: someText)
    bcdFixedErrCorr!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdFixedErrCorr!.type = .fixedErrCorrection
    bcdFixedErrCorr!.fixedErrCorrection = 30

    let bcdCompact = AztecBarcode(start: Points(x: 0.25, y: 2.25), data: someText)
    bcdCompact!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdCompact!.type = .compact
    bcdCompact!.layers = 4

    let bcdFull = AztecBarcode(start: Points(x: 1.5, y: 2.25), data: someText)
    bcdFull!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdFull!.type = .full
    bcdFull!.layers = 5

    let bcdRuneA = AztecBarcode(start: Points(x: 0.25, y: 4.00), data: "0")
    bcdRuneA!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdRuneA!.type = .rune

    let bcdRuneB = AztecBarcode(start: Points(x: 0.75, y: 4.00), data: "255")
    bcdRuneB!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdRuneB!.type = .rune

    let bcdRuneC = AztecBarcode(start: Points(x: 1.25, y: 4.00), data: "123")
    bcdRuneC!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdRuneC!.type = .rune

    lbl!.add(bcdDefault)
    lbl!.add(bcdFixedErrCorr)
    lbl!.add(bcdCompact)
    lbl!.add(bcdFull)
    lbl!.add(bcdRuneA)
    lbl!.add(bcdRuneB)
    lbl!.add(bcdRuneC)

    return lbl!
}

func BcdQRCode() -> UniPRT.Label? {

    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))
    
    let lbl = Label(name: "QRCodes")


    let enText = "Tree in the forest"
    let jaText = "森の中の木"

    let english = QRBarcode(start: Points(x: 0.25, y: 1.0), data: enText)
    english!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let englishMasked = QRBarcode(start: Points(x: 1.5, y: 1.0), data: enText)
    englishMasked!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    englishMasked!.mask = .mask4

    let japanese = QRBarcode(start: Points(x: 0.25, y: 2.25), data: jaText)
    japanese!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    japanese!.mask = .mask1

    let japaneseMasked = QRBarcode(start: Points(x: 1.5, y: 2.25), data: jaText)
    japaneseMasked!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    japaneseMasked!.mask = .mask4

    let autoEncData = QRBarcode(start: Points(x: 0.25, y: 3.75), data: "12345678 TREE IN THE FOREST 森の中の木")
    autoEncData!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    autoEncData!.mask = .mask4

    let manualModeData: [[Any]] = [
        [NSNumber(value: QRCodeManualEncodingEnum.numeric.rawValue), "12345678"],
        [NSNumber(value: QRCodeManualEncodingEnum.alphaNumeric.rawValue), " TREE IN THE FOREST "],
        [NSNumber(value: QRCodeManualEncodingEnum.binary.rawValue), "森の中の木"]
    ]

    let manualEncData = QRBarcode(start: Points(x: 1.75, y: 3.75), manuallyEncodedData: manualModeData)
    manualEncData!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    manualEncData!.mask = .mask4

    lbl!.add(english)
    lbl!.add(englishMasked)
    lbl!.add(japanese)
    lbl!.add(japaneseMasked)
    lbl!.add(autoEncData)
    lbl!.add(manualEncData)

    return lbl!
}

//PGL Functions
func PglRfidEncode() -> PglLabel {
    let lbl = PglLabel(name: "Rfidlbl!")

    let a32BitField: UInt32 = 0x11223344
    let a16BitField: UInt16 = 0xBEEF
    let a6CharAsciiString = "MyData"
    
    var epcHexData = String(format: "%08X", a32BitField) + stringToHex(a6CharAsciiString)
    epcHexData += String(format: "%04X", a16BitField)

    let epc = PglRfid_Write(memBlock: .EPC, data: epcHexData)
    lbl!.add(epc)
    
    var userDataHex = stringToHex("MyUserData")
    userDataHex += "0ABCDE0F"
    
    let userMem = PglRfid_Write(memBlock: .user, data: userDataHex)
    userMem!.offsetFromStart = 2
    lbl!.add(userMem)

    return lbl!
}

func PglBcdPdf417() -> PglLabel {
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 600, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))
    let lbl = PglLabel(name: "Pdf417Bcodes")

    let someText = "The happiness in your life depends on the quality of your thoughts. --Marcus Aurelius"
    let someShortText = "PI = 3.1415"

    let bcdDefault = PglPdf417Barcode(start: Points(x: 0.25, y: 0.50), data: someText)
    bcdDefault!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))

    let bcdErrCorrectionLvl0 = PglPdf417Barcode(start: Points(x: 0.25, y: 1.50), data: someShortText)
    bcdErrCorrectionLvl0!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdErrCorrectionLvl0!.errorCorrection = .level0

    let bcdErrCorrectionLvl5 = PglPdf417Barcode(start: Points(x: 0.25, y: 2.00), data: someShortText)
    bcdErrCorrectionLvl5!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdErrCorrectionLvl5!.errorCorrection = .level5

    let bcdRowsLimited = PglPdf417Barcode(start: Points(x: 0.25, y: 3.00), data: someShortText)
    bcdRowsLimited!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdRowsLimited!.rows = 15

    let bcdColsLimited = PglPdf417Barcode(start: Points(x: 0.25, y: 4.00), data: someShortText)
    bcdColsLimited!.cellSize = CellRect(xDim: 0.015, yDim: 0.050, ruler: Ruler(scale: .inch))
    bcdColsLimited!.columns = 5

    lbl!.add(bcdDefault)
    lbl!.add(bcdErrCorrectionLvl0)
    lbl!.add(bcdErrCorrectionLvl5)
    lbl!.add(bcdRowsLimited)
    lbl!.add(bcdColsLimited)

    return lbl!
}

func PglBcdAztec() -> PglLabel {
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))
    let lbl = PglLabel(name: "AztecBcodes")

    let someText = "Mr. AirTraveler, seat A, flight 200"

    let bcdDefault = PglAztecBarcode(start: Points(x: 0.25, y: 1.0), data: someText)
    bcdDefault!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let bcdFixedErrCorr = PglAztecBarcode(start: Points(x: 1.5, y: 1.0), data: someText)
    bcdFixedErrCorr!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdFixedErrCorr!.type = .fixedErrCorrection
    bcdFixedErrCorr!.fixedErrCorrection = 30

    let bcdCompact = PglAztecBarcode(start: Points(x: 0.25, y: 2.25), data: someText)
    bcdCompact!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdCompact!.type = .compact
    bcdCompact!.layers = 4

    let bcdFull = PglAztecBarcode(start: Points(x: 1.5, y: 2.25), data: someText)
    bcdFull!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdFull!.type = .full
    bcdFull!.layers = 5

    let bcdRuneA = PglAztecBarcode(start: Points(x: 0.25, y: 4.00), data: "0")
    bcdRuneA!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdRuneA!.type = .rune

    let bcdRuneB = PglAztecBarcode(start: Points(x: 0.75, y: 4.00), data: "255")
    bcdRuneB!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdRuneB!.type = .rune

    let bcdRuneC = PglAztecBarcode(start: Points(x: 1.25, y: 4.00), data: "123")
    bcdRuneC!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    bcdRuneC!.type = .rune

    lbl!.add(bcdDefault)
    lbl!.add(bcdFixedErrCorr)
    lbl!.add(bcdCompact)
    lbl!.add(bcdFull)
    lbl!.add(bcdRuneA)
    lbl!.add(bcdRuneB)
    lbl!.add(bcdRuneC)

    return lbl!
}

func PglBcdQRCode() -> PglLabel {
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))
    let lbl = PglLabel(name: "QRCodes")

    let enText = "Tree in the forest"
    let jaText = "森の中の木"

    let english = PglQRBarcode(start: Points(x: 0.25, y: 1.0), data: enText)
    english!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let englishMasked = PglQRBarcode(start: Points(x: 1.5, y: 1.0), data: enText)
    englishMasked!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    englishMasked!.mask = .mask4

    let japanese = PglQRBarcode(start: Points(x: 0.25, y: 2.25), data: jaText)
    japanese!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let japaneseMasked = PglQRBarcode(start: Points(x: 1.5, y: 2.25), data: jaText)
    japaneseMasked!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    japaneseMasked!.mask = .mask4

    let autoEncData = PglQRBarcode(start: Points(x: 0.25, y: 3.75), data: "12345678 TREE IN THE FOREST 森の中の木")
    autoEncData!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    autoEncData!.mask = .mask4

    let manualModeData: [[Any]] = [
        [NSNumber(value: QRCodeManualEncodingEnum.numeric.rawValue), "12345678"],
        [NSNumber(value: QRCodeManualEncodingEnum.alphaNumeric.rawValue), " TREE IN THE FOREST "],
        [NSNumber(value: QRCodeManualEncodingEnum.binary.rawValue), "森の中の木"]
    ]

    let manualEncData = PglQRBarcode(start: Points(x: 1.75, y: 3.75), manuallyEncodedData: manualModeData)
    manualEncData!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))
    manualEncData!.mask = .mask4

    lbl!.add(english)
    lbl!.add(englishMasked)
    lbl!.add(japanese)
    lbl!.add(japaneseMasked)
    lbl!.add(autoEncData)
    lbl!.add(manualEncData)

    return lbl!
}

func PglBcdDataMatrix() -> PglLabel {
    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: .inch))
    Defaults.setRuler(Ruler(scale: .inch))
    let lbl = PglLabel(name: "DMatrixBcds")

    let dfltMatrix = PglDataMatrixBarcode(start: Points(x: 0.25, y: 0.25), data: "Default DataMatrix")

    let rectMatrix = PglDataMatrixBarcode(start: Points(x: 1.25, y: 0.25), data: "Rectangular DataMatrix")
    rectMatrix!.rotation = .counterClockwise
    rectMatrix!.rectangle = true
    rectMatrix!.cellSize = CellSquare(xDim: 0.025, ruler: Ruler(scale: .inch))

    let dfltMatrixMultiLine = PglDataMatrixBarcode(start: Points(x: 2.25, y: 0.25), data: "Line 1 DataMatrix")
    let Eol = dfltMatrixMultiLine!.ctrlChar(0x0D) + dfltMatrixMultiLine!.ctrlChar(0x0A) // CR + LF
    dfltMatrixMultiLine!.data += Eol + "Line 2 content" + Eol + "Line 3 content"

    let rectMatrixUserDefinedDimensions = PglDataMatrixBarcode(start: Points(x: 1.25, y: 1.75), data: "DataMatrix with user defined dimensions")
    rectMatrixUserDefinedDimensions!.rectangle = true
    rectMatrixUserDefinedDimensions!.rowsCols = CGSize(width: 16, height: 36) as NSValue
    rectMatrixUserDefinedDimensions!.cellSize = CellSquare(xDim: 0.030, ruler: Ruler(scale: .inch))

    lbl!.add(dfltMatrix)
    lbl!.add(rectMatrix)
    lbl!.add(dfltMatrixMultiLine)
    lbl!.add(rectMatrixUserDefinedDimensions)

    return lbl!
}

func PglBcdMaxicodes() -> PglLabel {
    let lbl = PglLabel(name: "MaxiBcds")

    let maxiDataStructCarrier = PglMaxicodeMsgStructured(mode: .mode2, postalCode: "90255", countryCode: "800", serviceClass: "200", remainingMsg: "Maxicode Carrier Standard")
    let maxicodeBarcodeSc = PglMaxicodeBarcode(start: Points(x: 0.5, y: 0.5), data: maxiDataStructCarrier)
    maxicodeBarcodeSc!.ruler = Ruler(scale: .inch)

    let maxiDataOss = PglMaxicodeMsgStructuredOpenSystemStandard(mode: .mode3, year: "24", postalCode: "OHA123", countryCode: "123", serviceClass: "400", remainingMsg: "Maxicode Open Standard Format")
    let maxicodeBarcodeOss = PglMaxicodeBarcode(start: Points(x: 0.5, y: 2.0), data: maxiDataOss)
    maxicodeBarcodeOss!.ruler = Ruler(scale: .inch)

    let maxiData = PglMaxicodeMsg(mode: .mode4, primaryMsg: "123456789", remainingMsg: "Maxicode unstructured")
    let maxicodeBarcode = PglMaxicodeBarcode(start: Points(x: 0.5, y: 3.5), data: maxiData)
    maxicodeBarcode!.ruler = Ruler(scale: .inch)

    lbl!.add(maxicodeBarcodeSc)
    lbl!.add(maxicodeBarcodeOss)
    lbl!.add(maxicodeBarcode)

    return lbl!
}

func PglSimpleLabel(name: String, address: String) -> PglLabel {
    let lbl = PglLabel(name: "SimpleLabel")!

    let inchRuler = Ruler(scale: ScaleEnum.inch)!
    let mmRuler = Ruler(scale: ScaleEnum.MM)!

    let line1 = _PglLine(start: Points(x: 2.5, y: 1.0 / 16.0)!, end: Points(x: 2.5, y: 1.0)!, lineThickness: 1.0 / 32.0)!
    line1.ruler = inchRuler
    lbl.add(line1)

    let line2 = _PglLine(start: Points(x: 0.12, y: 1.0)!, end: Points(x: 3.88, y: 1.0)!, lineThickness: 1.0 / 32.0)!
    line2.ruler = inchRuler
    lbl.add(line2)

    let line3 = _PglLine(start: Points(x: 0.12, y: 3.5)!, end: Points(x: 3.88, y: 3.5)!, lineThickness: 1.0 / 32.0)!
    line3.ruler = inchRuler
    lbl.add(line3)

    let box1 = _PglBox(start: Points(x: 0.5, y: 1.25)!, end: Points(x: 3.5, y: 2.25)!, lineThickness: 1.0 / 16.0)!
    box1.ruler = inchRuler
    lbl.add(box1)

    let productText = _PglText()!
    productText.fontSizeUnits = FontSizeUnitsEnum.ruler
    productText.ruler = inchRuler
    productText.alignment = AlignEnum.center
    productText.fontName = "93952.sf"

    let textItem1 = TextItem(start: Points(x: 2.0, y: 1.25 + 7.0 / 16.0)!, data: "MY MAGIC")!
    textItem1.fontSize = FontSize(x: 3.0 / 16.0, y: 7.0 / 16.0)
    productText.text.add(textItem1)

    let textItem2 = TextItem(start: Points(x: 2.0, y: 1.25 + 1.0 - 3.0 / 16.0)!, data: "PRODUCT")!
    textItem2.fontSize = FontSize(x: 3.0 / 16.0, y: 7.0 / 16.0)
    productText.text.add(textItem2)

    lbl.add(productText)

    let boldToFrom = _PglText()!
    boldToFrom.fontSizeUnits = FontSizeUnitsEnum.ruler
    boldToFrom.ruler = mmRuler
    boldToFrom.fontStyle = FontStyleEnum.bold
    boldToFrom.fontName = "92248.sf"

    let bold_textItem1 = TextItem(start: Points(x: 5.0, y: 5.0)!, data: "TO:")!
    bold_textItem1.fontSize = FontSize(x: 2.5, y: 5.0)
    boldToFrom.text.add(bold_textItem1)

    let bold_textItem2 = TextItem(start: Points(x: (2.5 + 1 / 16.0) * 25.4, y: 5.0)!, data: "FROM:")!
    boldToFrom.text.add(bold_textItem2)

    lbl.add(boldToFrom)

    let companyName = _PglText()!
    companyName.fontSizeUnits = FontSizeUnitsEnum.percent
    companyName.ruler = mmRuler
    companyName.fontStyle = FontStyleEnum.italic
    companyName.fontName = "92500.sf"

    let companyNameTextItem = TextItem(start: Points(x: (2.5 + 1 / 16.0 + 1 / 8.0) * 25.4, y: 17.0)!, data: "Happy Inc.")!
    companyNameTextItem.fontSize = FontSize(x: 2.0, y: 3.0)
    companyName.text.add(companyNameTextItem)

    lbl.add(companyName)

    let nameTxt = _PglText()!
    nameTxt.fontSizeUnits = FontSizeUnitsEnum.ruler
    nameTxt.ruler = mmRuler
    nameTxt.fontStyle = FontStyleEnum.italic

    let nameTextItem = TextItem(start: Points(x: 8.0, y: 10.0)!, data: name)!
    nameTextItem.fontSize = FontSize(x: 2.5, y: 5.0)
    nameTxt.text.add(nameTextItem)

    lbl.add(nameTxt)

    let addressTxt = _PglText()!
    addressTxt.ruler = mmRuler

    let addressTextItem = TextItem(start: Points(x: 8.0, y: 17.0)!, data: address)!
    addressTxt.text.add(addressTextItem)

    lbl.add(addressTxt)

    let bcd128 = PglBarcode_1D(barcodeItem: BarcodeItem(start: Points(x: 0.5, y: (1.5 + 1 / 4.0 + 1.2))!, height: 1.2, data: "Code 128")!)!
    bcd128.barcodeType = BarcodeTypeEnum1D.dCode128
    bcd128.printHumanReadable = true
    bcd128.rotation = RotateEnum.none
    bcd128.ruler = inchRuler
    bcd128.barWidths = PglBarWidths(narrowBar: 0.015, wideBar: 0.015 * 4.1)!
    bcd128.barWidths!.ruler = inchRuler
    lbl.add(bcd128)

    let bcd93 = PglBarcode_1D(barcodeItem: BarcodeItem(start: Points(x: 0.5, y: 3.5 - 1 / 8.0 - 0.6)!, height: 0.6, data: "CODE 93")!)!
    bcd93.barcodeType = BarcodeTypeEnum1D.dCode93
    bcd93.printHumanReadable = true
    bcd93.rotation = RotateEnum.none
    bcd93.ruler = inchRuler
    bcd93.barWidths = PglBarWidths(narrowBar: 0.025, wideBar: 0.025 * 4.1)!
    bcd93.barWidths!.ruler = inchRuler
    lbl.add(bcd93)

    let dmCustomer = PglDataMatrixBarcode(start: Points(x: 2.7, y: 4.0)!, data: name)!
    dmCustomer.cellSize = CellSquare(xDim: 0.040, ruler: inchRuler)
    dmCustomer.ruler = inchRuler
    let Eol = dmCustomer.ctrlChar(0x0D)! + dmCustomer.ctrlChar(0x0A)!
    dmCustomer.data = "\(dmCustomer.data!)\(Eol)\(address)"
    lbl.add(dmCustomer)

    return lbl
}

func RulerLines(length: Float, vertical: Bool, inchUnits: Bool, margin: Float) -> [_PglLine] {
    var rulerLines: [_PglLine] = []
    let tickRuler = inchUnits ? Ruler(scale: .inch)! : Ruler(scale: .MM)!

    var rulerLength = length
    var tickThickness: Float = 0.010
    var tickLength: Float = 1 / 16.0
    let ticksPerUnit: Float = inchUnits ? 16.0 : 1.0

    var margin = margin

    if !inchUnits {
        tickThickness *= MM_PER_INCH
        tickLength *= MM_PER_INCH
        margin *= MM_PER_INCH
    }

    rulerLength -= tickThickness

    for i in stride(from: 1, through: (rulerLength * ticksPerUnit), by: 1) {
        var tick = tickLength
        if inchUnits {
            if Int(i) % 16 == 0 {
                tick *= 3.5
            } else if Int(i) % 8 == 0 {
                tick *= 2.5
            } else if Int(i) % 4 == 0 {
                tick *= 2.0
            } else if Int(i) % 2 == 0 {
                tick *= 1.5
            }
        } else {
            if Int(i) % 10 == 0 {
                tick *= 3.0
            } else if Int(i) % 5 == 0 {
                tick *= 1.5
            }
        }

        let tickLine: _PglLine
        if vertical {
            tickLine = _PglLine(xStart: margin, yStart: i / ticksPerUnit, xEnd: margin + tick, yEnd: i / ticksPerUnit, lineThickness: tickThickness)!
        } else {
            tickLine = _PglLine(xStart: i / ticksPerUnit, yStart: margin, xEnd: i / ticksPerUnit, yEnd: margin + tick, lineThickness: tickThickness)!
        }


        tickLine.ruler = tickRuler
        rulerLines.append(tickLine)
    }

    return rulerLines
}

func RuleredLabel(width: Float, length: Float, inchUnits: Bool, rulerMargin: Float) -> PglLabel {
    let verRulerTicks = RulerLines(length: length, vertical: true, inchUnits: inchUnits, margin: rulerMargin)
    let horRulerTicks = RulerLines(length: width, vertical: false, inchUnits: inchUnits, margin: rulerMargin)

    Defaults.setPrinterResolution(PrintResolution(dotsPerUnit: 300, unit: ScaleEnum.inch)!)

    let rulerLbl = PglLabel(name: "Ruler")!
    for tickLine in verRulerTicks {
        rulerLbl.add(tickLine)
    }
    for tickLine in horRulerTicks {
        rulerLbl.add(tickLine)
    }

    return rulerLbl
}


func stringToHex(_ input: String) -> String {
    let utf8String = input.utf8
    var hexString = ""
    for byte in utf8String {
        hexString += String(format: "%02X", byte)
    }
    return hexString
}

