//
//  IComm.h
//  sdk_json_Objective-C_IOS
//
//  Created by Realbuber on 2024/4/23.
//

#ifndef IComm_h
#define IComm_h


#endif /* IComm_h */
#import <Foundation/Foundation.h>

/**
 * @protocol IComm
 * @ingroup Comm
 *
 * \~English
 * @brief A protocol defining the interface for communication classes.
 *
 * \~Chinese
 * @brief 定义通信类接口的协议。
 *
 * \~Chinese-Traditional
 * @brief 定義通信類接口的協議。
 */
@protocol IComm <NSObject>

/**
 * \~English
 * @brief The descriptor for the communication interface.
 *
 * \~Chinese
 * @brief 通信接口的描述符。
 *
 * \~Chinese-Traditional
 * @brief 通信接口的描述符。
 */
@property (nonatomic, readonly) NSString *descriptor;

/**
 * \~English
 * @brief The number of bytes available to read.
 *
 * \~Chinese
 * @brief 可读取的字节数。
 *
 * \~Chinese-Traditional
 * @brief 可讀取的字節數。
 */
@property (nonatomic, readonly) int bytesAvailable;

/**
 * \~English
 * @brief A flag indicating whether the connection is established.
 *
 * \~Chinese
 * @brief 指示连接是否已建立的标志。
 *
 * \~Chinese-Traditional
 * @brief 指示連接是否已建立的標誌。
 */
@property (nonatomic, readonly) BOOL isConnected;

/**
 * \~English
 * @brief Closes the communication interface.
 *
 * \~Chinese
 * @brief 关闭通信接口。
 *
 * \~Chinese-Traditional
 * @brief 關閉通信接口。
 */
- (void)close;

/**
 * \~English
 * @brief Opens the communication interface.
 *
 * \~Chinese
 * @brief 打开通信接口。
 *
 * \~Chinese-Traditional
 * @brief 打開通信接口。
 */
- (void)open;

/**
 * \~English
 * @brief Reads data from the communication interface.
 *
 * @return The data read from the interface.
 *
 * \~Chinese
 * @brief 从通信接口读取数据。
 *
 * @return 从接口读取的数据。
 *
 * \~Chinese-Traditional
 * @brief 從通信接口讀取數據。
 *
 * @return 從接口讀取的數據。
 */
- (NSData *)read;

/**
 * \~English
 * @brief Writes data from a binary reader (input stream) to the output stream.
 *
 * @param binDataIn The output stream to write the data into.
 *
 * \~Chinese
 * @brief 将数据从二进制读取器（输入流）写入输出流。
 *
 * @param binDataIn 要写入数据的输出流。
 *
 * \~Chinese-Traditional
 * @brief 將數據從二進制讀取器（輸入流）寫入輸出流。
 *
 * @param binDataIn 要寫入數據的輸出流。
 */
- (void)readToBinaryWriter:(NSOutputStream *)binDataIn;

/**
 * \~English
 * @brief Waits until data is available in the current thread. The current thread sleeps until data is received or the timeout is reached. This is a blocking call.
 *
 * @param msTimeOut The timeout interval in milliseconds before returning if no data is received.
 *
 * \~Chinese
 * @brief 等待数据在当前线程中可用。当前线程将休眠，直到接收到数据或达到超时。这是一个阻塞调用。
 *
 * @param msTimeOut 在未接收到数据之前返回的超时间隔（毫秒）。
 *
 * \~Chinese-Traditional
 * @brief 等待數據在當前線程中可用。當前線程將休眠，直到接收到數據或達到超時。這是一個阻塞調用。
 *
 * @param msTimeOut 在未接收到數據之前返回的超時間隔（毫秒）。
 */
- (void)waitForDataWithTimeout:(NSTimeInterval)msTimeOut;

/**
 * \~English
 * @brief Writes all bytes from the array passed in.
 *
 * @param binReader The input stream to read the data from.
 *
 * \~Chinese
 * @brief 将传入数组中的所有字节写入。
 *
 * @param binReader 要从中读取数据的输入流。
 *
 * \~Chinese-Traditional
 * @brief 將傳入數組中的所有字節寫入。
 *
 * @param binReader 要從中讀取數據的輸入流。
 */
- (void)writeFromBinaryReader:(NSInputStream *)binReader;

/**
 * \~English
 * @brief Writes data to the communication interface.
 *
 * @param dataOut The data to write to the interface.
 *
 * \~Chinese
 * @brief 将数据写入通信接口。
 *
 * @param dataOut 要写入接口的数据。
 *
 * \~Chinese-Traditional
 * @brief 將數據寫入通信接口。
 *
 * @param dataOut 要寫入接口的數據。
 */
- (void)write:(NSData *)dataOut;

/**
 * \~English
 * @brief Writes data from the input stream to the output stream and returns the data received in the output stream. The data returned is any data received or up to the completion token if received. Waits for a response timeout before returning.
 *
 * @param binDataIn The output stream to store the response received from the connection.
 * @param binDataOut The input stream data to write to the connection.
 * @param responseStartTimeOut The timeout interval to wait for the first data to be received (time to first byte).
 * @param responseEndTimeOut The timeout interval after the last byte if no new data is received (time after last byte).
 * @param completionToken The string token denoting the end of the expected response. This can be an empty string if simply waiting for a response after writing data to the connection.
 * @return The data received from the connection.
 *
 * \~Chinese
 * @brief 将输入流中的数据写入输出流，并返回输出流中接收到的数据。返回的数据是接收到的任何数据或接收到的完成令牌。等待响应超时后返回。
 *
 * @param binDataIn 用于存储从连接接收到的响应的输出流。
 * @param binDataOut 要写入连接的输入流数据。
 * @param responseStartTimeOut 等待接收到第一个数据的超时间隔（首次字节时间）。
 * @param responseEndTimeOut 在接收到最后一个字节后如果没有新数据则等待的超时间隔（最后字节时间）。
 * @param completionToken 表示预期响应结束的字符串令牌。如果在写入数据到连接后只是等待响应，这可以是一个空字符串。
 * @return 从连接接收到的数据。
 *
 * \~Chinese-Traditional
 * @brief 將輸入流中的數據寫入輸出流，並返回輸出流中接收到的數據。返回的數據是接收到的任何數據或接收到的完成令牌。等待響應超時後返回。
 *
 * @param binDataIn 用於存儲從連接接收到的響應的輸出流。
 * @param binDataOut 要寫入連接的輸入流數據。
 * @param responseStartTimeOut 等待接收到第一個數據的超時間隔（首次字節時間）。
 * @param responseEndTimeOut 在接收到最後一個字節後如果沒有新數據則等待的超時間隔（最後字節時間）。
 * @param completionToken 表示預期響應結束的字符串令牌。如果在寫入數據到連接後只是等待響應，這可以是一個空字符串。
 * @return 從連接接收到的數據。
 */
- (NSData *)writeAndWaitForResponseToBinaryWriter:(NSOutputStream *)binDataIn
                                   fromBinaryReader:(NSInputStream *)binDataOut
                               responseStartTimeout:(int)responseStartTimeOut
                                 responseEndTimeout:(int)responseEndTimeOut
                                    completionToken:(NSString *)completionToken;

/**
 * \~English
 * @brief Writes byte data to the output stream and returns the data received. The data returned is any data received or up to the completion token if received. Waits for a response timeout before returning.
 *
 * @param text The source byte array to write to the connection.
 * @param responseStartTimeOut The timeout interval to wait for the first data to be received (time to first byte).
 * @param responseEndTimeOut The timeout interval after the last byte if no new data is received (time after last byte).
 * @param completionToken The string token denoting the end of the expected response. This can be an empty string if simply waiting for a response after writing data to the connection.
 * @return The array of bytes received from the connection.
 *
 * \~Chinese
 * @brief 将字节数据写入输出流并返回接收到的数据。返回的数据是接收到的任何数据或接收到的完成令牌。等待响应超时后返回。
 *
 * @param text 要写入连接的源字节数组。
 * @param responseStartTimeOut 等待接收到第一个数据的超时间隔（首次字节时间）。
 * @param responseEndTimeOut 在接收到最后一个字节后如果没有新数据则等待的超时间隔（最后字节时间）。
 * @param completionToken 表示预期响应结束的字符串令牌。如果在写入数据到连接后只是等待响应，这可以是一个空字符串。
 * @return 从连接接收到的字节数组。
 *
 * \~Chinese-Traditional
 * @brief 將字節數據寫入輸出流並返回接收到的數據。返回的數據是接收到的任何數據或接收到的完成令牌。等待響應超時後返回。
 *
 * @param text 要寫入連接的源字節數組。
 * @param responseStartTimeOut 等待接收到第一個數據的超時間隔（首次字節時間）。
 * @param responseEndTimeOut 在接收到最後一個字節後如果沒有新數據則等待的超時間隔（最後字節時間）。
 * @param completionToken 表示預期響應結束的字符串令牌。如果在寫入數據到連接後只是等待響應，這可以是一個空字符串。
 * @return 從連接接收到的字節數組。
 */
- (NSData *)writeAndWaitForResponse:(NSData *)text
               responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                 responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                    completionToken:(NSString *)completionToken;

@end
