//
//  RfidReport.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/13.
//

#ifndef RfidReport_h
#define RfidReport_h


#endif /* RfidReport_h */

// RfidReport.h

#import <Foundation/Foundation.h>

/**
 * \~English
 * @enum RfidDataType
 * @brief Enumeration for different types of RFID data.
 *
 * \~Chinese
 * @enum RfidDataType
 * @brief RFID 数据类型的枚举。
 *
 * \~Chinese-Traditional
 * @enum RfidDataType
 * @brief RFID 數據類型的枚舉。
 */
typedef NS_ENUM(NSInteger, RfidDataType) {
    /**
     * \~English
     * @brief Electronic Product Code.
     *
     * \~Chinese
     * @brief 电子产品代码。
     *
     * \~Chinese-Traditional
     * @brief 電子產品代碼。
     */
    EPC,

    /**
     * \~English
     * @brief User memory.
     *
     * \~Chinese
     * @brief 用户内存。
     *
     * \~Chinese-Traditional
     * @brief 用戶內存。
     */
    USR,

    /**
     * \~English
     * @brief Tag ID.
     *
     * \~Chinese
     * @brief 标签 ID。
     *
     * \~Chinese-Traditional
     * @brief 標籤 ID。
     */
    TID,

    /**
     * \~English
     * @brief Access Control.
     *
     * \~Chinese
     * @brief 访问控制。
     *
     * \~Chinese-Traditional
     * @brief 訪問控制。
     */
    ACS,

    /**
     * \~English
     * @brief Unknown or not recognized data type.
     *
     * \~Chinese
     * @brief 未知或未识别的数据类型。
     *
     * \~Chinese-Traditional
     * @brief 未知或未識別的數據類型。
     */
    Unknown
};

/**
 * \~English
 * @class RfidReport
 * @brief Rfid report based on RFID tag messages received from printer.
 * @ingroup Reports
 *
 * \~Chinese
 * @class RfidReport
 * @brief 基于从打印机接收到的 RFID 标签消息的 RFID 报告。
 * @ingroup Reports
 *
 * \~Chinese-Traditional
 * @class RfidReport
 * @brief 基於從打印機接收到的 RFID 標籤消息的 RFID 報告。
 * @ingroup Reports
 */
@interface RfidReport : NSObject

/**
 * \~English
 * @brief The raw report dictionary containing RFID details.
 *
 * \~Chinese
 * @brief 包含 RFID 详细信息的原始报告字典。
 *
 * \~Chinese-Traditional
 * @brief 包含 RFID 詳細信息的原始報告字典。
 */
@property (nonatomic, strong, readonly) NSDictionary<NSString *, NSString *> *rawReport;

/**
 * \~English
 * @brief Checks if the RFID operation failed.
 *
 * @return YES if the operation failed, NO otherwise.
 *
 * \~Chinese
 * @brief 检查 RFID 操作是否失败。
 *
 * @return 如果操作失败，返回 YES，否则返回 NO。
 *
 * \~Chinese-Traditional
 * @brief 檢查 RFID 操作是否失敗。
 *
 * @return 如果操作失敗，返回 YES，否則返回 NO。
 */
- (BOOL)failed;

/**
 * \~English
 * @brief String representation of RFID data.
 *
 * @return A string representing the RFID data.
 *
 * \~Chinese
 * @brief RFID 数据的字符串表示形式。
 *
 * @return 表示 RFID 数据的字符串。
 *
 * \~Chinese-Traditional
 * @brief RFID 數據的字符串表示形式。
 *
 * @return 表示 RFID 數據的字符串。
 */
- (NSString *)data;

/**
 * \~English
 * @brief Type of RFID data contained in Data.
 *
 * @return The data type as an RfidDataType value.
 *
 * \~Chinese
 * @brief 数据中包含的 RFID 数据类型。
 *
 * @return 数据类型，作为 RfidDataType 值。
 *
 * \~Chinese-Traditional
 * @brief 數據中包含的 RFID 數據類型。
 *
 * @return 數據類型，作為 RfidDataType 值。
 */
- (RfidDataType)dataType;

/**
 * \~English
 * @brief True if report result of write operation. False if read operation.
 *
 * @return YES if it was a write operation, NO otherwise.
 *
 * \~Chinese
 * @brief 如果报告结果是写操作，则为真。读操作则为假。
 *
 * @return 如果是写操作，返回 YES，否则返回 NO。
 *
 * \~Chinese-Traditional
 * @brief 如果報告結果是寫操作，則為真。讀操作則為假。
 *
 * @return 如果是寫操作，返回 YES，否則返回 NO。
 */
- (BOOL)isWriteOperation;

/** @cond */

/**
 * \~English
 * @brief Sets the raw report dictionary.
 *
 * @param rawReport The dictionary containing the raw RFID report information.
 *
 * \~Chinese
 * @brief 设置原始报告字典。
 *
 * @param rawReport 包含原始 RFID 报告信息的字典。
 *
 * \~Chinese-Traditional
 * @brief 設置原始報告字典。
 *
 * @param rawReport 包含原始 RFID 報告信息的字典。
 */
- (void)setRawReport:(NSDictionary<NSString *, NSString *> *)rawReport;

/** @endcond */

@end
