﻿#include <cstdio>
#include <iostream>
#include <string>
#include <sstream>
#include <string>
#include <cstring>
#include <iomanip>
#include <fstream>
#include <vector>
#include <thread>
#include <mutex>
#include <atomic>
#include <chrono>
#include <condition_variable>
#include <dlfcn.h> // 包含動態加載庫的頭文件

#include "OdvReport.h"


enum COMM_TYP
{
	TCP_COMM = 1
};



// TCP
enum DescriptorPortType {

	// Data
	DATA,
	// Management
	MGMT,
	// Status
	STATUS
};


struct PrinterInfo {
	std::string Model;
	std::string SerialNumber;
	std::string FirmwarePartNumber;
	std::string FirmwareVersion;
	std::string PrintheadResolution;
	bool HasRfidOption;
	bool HasOdvOption;
};


enum INFO_TYP
{
	ODV_TYP = 0,
	RFID_TYP,

};


using byte = unsigned char;

// CommSDK. dll export
typedef int (*BytesAvailableFunc)();
typedef bool (*ConnectedFunc)();
typedef void (*ReadFunc)(byte* buffer, int size);
typedef void (*WriteFunc)(byte* dataOut, int iDataSize);
typedef byte* (*WriteAndWaitForResponseFunc)(byte* dataOut, int iDataSize, int responseStartTimeOut, int responseEndTimeOut, char* completetionToken);
typedef void (*CloseFunc)();
typedef void (*OpenFunc)();
typedef void (*TcpConnectFunc)(char* ipAddress, int port);
typedef void (*GetCommFunc)(void*& ptrComm); 


// Tool APIs
typedef void (*SendPrintFileFunc)(char* ipAddress, char* fileName);


// JsonSDK. dll export
typedef void (*MessengerGetFunc)(void* ptrComm, COMM_TYP commTyp, int maxInputMsgCapacity, bool usingDataPort);
typedef void (*MessengerReleaseFunc)();
typedef void (*MessengerSendMsgFunc)(char* strCommand, char* strContent);
typedef void (*MessengerReadNextMsgFunc)(const char** strNextMsg);
typedef void (*MessengerSendMsgAndWaitForResponseFunc)(char* strCommand, char* strContent, int maxWaitTimeSecs, const char** strResponse);
typedef int (*MessengerUnreadMsgCountFunc)();
typedef void (*OdvMonitorConnectionFunc)(char* lpAddress);
typedef void (*PrinterMonitorConnectionFunc)(char* lpAddress, INFO_TYP InfoTyp);
typedef void (*GetPrinterInfoFunc)(PrinterInfo* pPrtInfo, INFO_TYP InfoTyp);
typedef bool (*GetOdvReportListeningFunc)();
typedef void (*SetOdvReportListeningFunc)(bool value);
typedef void (*OdvMonitorDisposeFunc)();
typedef void (*PrinterMonitorDisposeFunc)(INFO_TYP InfoTyp);
typedef void (*SetOdvReportCallbackFunc)(void callback(const OdvReport& report));


BytesAvailableFunc pBytesAvailable = NULL;
ConnectedFunc pConnected = NULL;
ReadFunc pRead = NULL;
WriteFunc pWrite = NULL;
WriteAndWaitForResponseFunc pWriteAndWaitForResponse = NULL;
CloseFunc pClose = NULL;
OpenFunc pOpen = NULL;
TcpConnectFunc pTcpConnect = NULL;
GetCommFunc pGetComm = NULL;
SendPrintFileFunc pSendPrintFile = NULL;


MessengerGetFunc pMessengerGet = NULL;
MessengerReleaseFunc pMessengerRelease = NULL;
MessengerSendMsgFunc pMessengerSendMsg = NULL;
MessengerReadNextMsgFunc pMessengerReadNextMsg = NULL;
MessengerSendMsgAndWaitForResponseFunc pMessengerSendMsgAndWaitForResponse = NULL;
MessengerUnreadMsgCountFunc pMessengerUnreadMsgCount = NULL;


OdvMonitorConnectionFunc pOdvMonitorConnection = NULL;
PrinterMonitorConnectionFunc pPrinterMonitorConnection = NULL;
GetPrinterInfoFunc pGetPrinterInfo = NULL;
GetOdvReportListeningFunc pGetOdvReportListening = NULL;
SetOdvReportListeningFunc pSetOdvReportListening = NULL;
OdvMonitorDisposeFunc pOdvMonitorDispose = NULL;
PrinterMonitorDisposeFunc pPrinterMonitorDispose = NULL;
SetOdvReportCallbackFunc pSetOdvReportCallback = NULL;


void* handleCommSDKLib;
void* handleJsonSDKLib;

void ShowPrinterInfo(PrinterInfo* pPrtInfo, INFO_TYP InfoTyp)
{
	pGetPrinterInfo(pPrtInfo, InfoTyp);

	std::cout << std::endl;
	std::cout << "Printer Model: " << pPrtInfo->Model << std::endl;
	std::cout << "Printer SN: " << pPrtInfo->SerialNumber << std::endl;
	std::cout << "Printer FW PN: " << pPrtInfo->FirmwarePartNumber << std::endl;
	std::cout << "Printer FW Ver: " << pPrtInfo->FirmwareVersion << std::endl;
	std::cout << "Printhead Resolution (Dots/Inch): " << pPrtInfo->PrintheadResolution << std::endl;
	std::cout << std::endl;
	std::cout << "Has ODV: " << (pPrtInfo->HasOdvOption ? "yes" : "no") << std::endl;


}


void OdvReportCallback(const OdvReport& report)
{
	if (report.Failed())
	{
		std::cout << "\nBarcode Failed." << std::endl;
	}
	else
	{
		std::cout << "\nBarcode Passed." << std::endl;
		std::cout << "Grade: " << report.OverallGrade() << std::endl;

		if (report.OverallGradeAsFloat() > 3.5)
		{
			std::cout << "Print Quality passed.  \n  Overall Grade= " << std::to_string(report.OverallGradeAsFloat()) << std::endl;
		}
		else
		{
			std::cout << "Print Quality Failed.  \n  Overall Grade= " << std::to_string(report.OverallGradeAsFloat()) << std::endl;
		}

		std::cout << "Barcode Symbology: " << report.Symbology() << std::endl;
		std::cout << "Barcode Data: " << report.Data() << std::endl;
	}
}

int main()
{
	// 加載 libCommSDK.so 庫
	handleCommSDKLib = dlopen("libCommSDK.so", RTLD_LAZY);
	if (!handleCommSDKLib) {
		std::cerr << "Failed to load libCommSDK.so: " << dlerror() << std::endl;
		return 1;
	}

	// 加載 libJsonSDK.so 庫
	handleJsonSDKLib = dlopen("libJsonSDK.so", RTLD_LAZY);
	if (!handleJsonSDKLib) {
		std::cerr << "Failed to load libCommSDK.so: " << dlerror() << std::endl;
		return 1;
	}


	pBytesAvailable = reinterpret_cast<BytesAvailableFunc>(dlsym(handleCommSDKLib, "BytesAvailable"));
	pConnected = reinterpret_cast<ConnectedFunc>(dlsym(handleCommSDKLib, "Connected"));
	pRead = reinterpret_cast<ReadFunc>(dlsym(handleCommSDKLib, "Read"));
	pWrite = reinterpret_cast<WriteFunc>(dlsym(handleCommSDKLib, "Write"));
	pWriteAndWaitForResponse = reinterpret_cast<WriteAndWaitForResponseFunc>(dlsym(handleCommSDKLib, "WriteAndWaitForResponse"));
	pClose = reinterpret_cast<CloseFunc>(dlsym(handleCommSDKLib, "Close"));
	pOpen = reinterpret_cast<OpenFunc>(dlsym(handleCommSDKLib, "Open"));
	pTcpConnect = reinterpret_cast<TcpConnectFunc>(dlsym(handleCommSDKLib, "TcpConnect"));
	pGetComm = reinterpret_cast<GetCommFunc>(dlsym(handleCommSDKLib, "GetComm"));
	pSendPrintFile = reinterpret_cast<SendPrintFileFunc>(dlsym(handleCommSDKLib, "SendPrintFile"));

	if (!pBytesAvailable || !pConnected || !pRead || !pWrite || 
		!pWriteAndWaitForResponse || !pOpen || !pTcpConnect ||
		!pGetComm || !pSendPrintFile)
	{
		std::cerr << "Failed to get address of libCommSDK.so API functions: " << dlerror() << std::endl;
		dlclose(handleCommSDKLib);
		return 1;
	}


	pMessengerGet = reinterpret_cast<MessengerGetFunc>(dlsym(handleJsonSDKLib, "MessengerGet"));
	pMessengerRelease = reinterpret_cast<MessengerReleaseFunc>(dlsym(handleJsonSDKLib, "MessengerRelease"));
	pMessengerSendMsg = reinterpret_cast<MessengerSendMsgFunc>(dlsym(handleJsonSDKLib, "MessengerSendMsg"));
	pMessengerReadNextMsg = reinterpret_cast<MessengerReadNextMsgFunc>(dlsym(handleJsonSDKLib, "MessengerReadNextMsg"));
	pMessengerSendMsgAndWaitForResponse = reinterpret_cast<MessengerSendMsgAndWaitForResponseFunc>(dlsym(handleJsonSDKLib, "MessengerSendMsgAndWaitForResponse"));
	pMessengerUnreadMsgCount = reinterpret_cast<MessengerUnreadMsgCountFunc>(dlsym(handleJsonSDKLib, "MessengerUnreadMsgCount"));

	pOdvMonitorConnection = reinterpret_cast<OdvMonitorConnectionFunc>(dlsym(handleJsonSDKLib, "OdvMonitorConnection"));
	pPrinterMonitorConnection = reinterpret_cast<PrinterMonitorConnectionFunc>(dlsym(handleJsonSDKLib, "PrinterMonitorConnection"));
	pGetPrinterInfo = reinterpret_cast<GetPrinterInfoFunc>(dlsym(handleJsonSDKLib, "GetPrinterInfo"));
	pGetOdvReportListening = reinterpret_cast<GetOdvReportListeningFunc>(dlsym(handleJsonSDKLib, "GetOdvReportListening"));
	pSetOdvReportListening = reinterpret_cast<SetOdvReportListeningFunc>(dlsym(handleJsonSDKLib, "SetOdvReportListening"));
	pOdvMonitorDispose = reinterpret_cast<OdvMonitorDisposeFunc>(dlsym(handleJsonSDKLib, "OdvMonitorDispose"));
	pPrinterMonitorDispose = reinterpret_cast<PrinterMonitorDisposeFunc>(dlsym(handleJsonSDKLib, "PrinterMonitorDispose"));
	pSetOdvReportCallback = reinterpret_cast<SetOdvReportCallbackFunc>(dlsym(handleJsonSDKLib, "SetOdvReportCallback"));



	if (!pMessengerGet || !pMessengerRelease || !pMessengerSendMsg || !pMessengerReadNextMsg || !pMessengerSendMsgAndWaitForResponse || !pMessengerUnreadMsgCount ||
		!pOdvMonitorConnection || !pPrinterMonitorConnection || !pGetPrinterInfo || !pGetOdvReportListening || !pSetOdvReportListening || !pOdvMonitorDispose ||
		!pPrinterMonitorDispose || !pSetOdvReportCallback)
	{
		std::cerr << "Failed to get address of libJsonSDK.so API functions: " << dlerror() << std::endl;
		dlclose(handleJsonSDKLib);
		return 1;
	}


	PrinterInfo PrtInfo;
	std::string ptrIpOdv = "10.0.10.177";

	pOdvMonitorConnection((char*)ptrIpOdv.c_str());
	pPrinterMonitorConnection((char*)ptrIpOdv.c_str(), INFO_TYP::ODV_TYP);

	std::cout << "\n=======Odv Printer Info: ";
	ShowPrinterInfo(&PrtInfo, INFO_TYP::ODV_TYP);


	if (PrtInfo.HasOdvOption == false)
	{
		std::cout << "WARNING: Missing ODV option on printer at: " << ptrIpOdv << std::endl;
		return 1;
	}


	// ============================
	// -------------  ODV Printer: setup for RFID report monitoring
	// ============================
	pSetOdvReportListening(true);
	pSetOdvReportCallback(OdvReportCallback);


	pSendPrintFile((char*)(ptrIpOdv.c_str()), (char*)"DM_PRINTRONIX_1.pgl");
	std::cout << "Sending datamatrix barcode print job..." << std::endl;


	int userInput;

	// 循環等待用戶輸入，直到用戶輸入0為止
	do {

		std::cout << std::endl;
		std::cout << std::endl;
		std::cout << "Press 0 to exit";
		std::cout << std::endl;
		std::cin >> userInput;
	} while (userInput != 0);

	// 使用者輸入0後程式結束
	std::cout << std::endl;
	std::cout << "Program exit" << std::endl;
	std::cout << std::endl;

	pClose();
	pOdvMonitorDispose();
	pPrinterMonitorDispose(INFO_TYP::ODV_TYP);


	// 釋放 libCommSDK.so / libJsonSDK.so 庫
	dlclose(handleCommSDKLib);
	dlclose(handleJsonSDKLib);
	return 0;

}



