import SwiftUI
import UniPRT

struct ContentView: View {
    @State private var ipText = "10.0.10.178"
    @State private var macAddressText = "34:81:F4:43:3D:A8"
    @State private var resultText = ""
    @State private var communicationType = 0 // 0 for TCP
    @State private var inputText = ""
    @StateObject private var viewModel = CommunicationViewModel()

    var body: some View {
        NavigationView {
            Form {
                Section(header: Text("Connection Type")) {
                    Picker("Select Type", selection: $communicationType) {
                        Text("TCP").tag(0)
                    }
                    .pickerStyle(SegmentedPickerStyle())
                }
                
                Section(header: Text("IP Address")) {
                    TextField("Enter IP Address", text: $ipText)
                        .keyboardType(.decimalPad)
                }
                
                Section(header: Text("Input")) {
                    TextEditor(text: $inputText)
                        .keyboardType(.asciiCapable)
                        .frame(height: 100) // 設置高度
                        .border(Color.secondary, width: 1) // 設置邊框
                }
                
                Section(header: Text("Output")) {
                    ScrollView {
                        Text(resultText)
                            .frame(maxWidth: .infinity, alignment: .leading)
                    }
                    .frame(height: 100)
                    .border(Color.secondary, width: 1)
                }
                
                Button("Connect/Close") {
                    resultText = viewModel.connectCloseAction(ip: ipText)
                }
                
                Button("Open") {
                    resultText = viewModel.openAction()
                }
                
                Button("Write") {
                    resultText = viewModel.writeAction(stringData: inputText)
                }

                Button("Write and Wait for Response") {
                    resultText = viewModel.writeAndWaitForResponseAction(stringData: inputText)
                }
                
                Button("Read") {
                    resultText = viewModel.readAction()
                }
                
            }
            .navigationBarTitle("Communication")
        }
    }
}

class CommunicationViewModel: ObservableObject {
    @Published var resultText = ""
    var tcpComm: TcpComm?

    func openAction() -> String {
        self.resultText = "open error"
        tcpComm?.open()
        self.resultText = "TCP Connection opened"
        return self.resultText
    }

    func writeAction(stringData: String) -> String {
        self.resultText = "write Error"
        tcpComm?.write(stringData.data(using: .utf8))
        self.resultText = "TCP Data written"
        return self.resultText
    }

    func readAction() -> String {
        if let data = tcpComm?.read() {
            self.resultText = String(data: data, encoding: .utf8) ?? "Failed to decode data"
        } else {
            self.resultText = "Error Occured."
        }
        return self.resultText
    }

    func connectCloseAction(ip: String) -> String {
        self.resultText = "connect error"
        print(ip)
        if tcpComm != nil {
            tcpComm?.close()
            tcpComm = nil
            self.resultText = "TCP Connection closed"
        } else {
            // Reconnect logic
            tcpComm = TcpComm(ipAddress: ip, port: 9100)
            self.resultText = "TCP Reconnected"
        }
        return self.resultText
    }

    func writeAndWaitForResponseAction(stringData: String) -> String {
        if let response = tcpComm?.writeAndWait(forResponse: stringData.data(using: .utf8), responseStartTimeOut: 3000, responseEndTimeOut: 3000, completionToken: "\r\n") {
            self.resultText = String(data: response, encoding: .utf8) ?? "Failed to decode response"
        } else {
            self.resultText = "Error Occured."
        }
        return self.resultText
    }
}

#Preview {
    ContentView()
}
