//
//  Config.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/4/19.
//

#ifndef Config_h
#define Config_h


#endif /* Config_h */

#import <Foundation/Foundation.h>
#import <UniPRT/JsonStringTokenizer.h>

/**
 * \~English
 * @class Config
 * @brief A class representing a configuration with various properties and methods to manipulate them.
 * @ingroup Settings
 *
 * \~Chinese
 * @class Config
 * @brief 表示配置的类，具有各种属性和方法来操作它们。
 * @ingroup Settings
 *
 * \~Chinese-Traditional
 * @class Config
 * @brief 表示配置的類，具有各種屬性和方法來操作它們。
 * @ingroup Settings
 */
@interface Config : NSObject

// Initializers

/**
 * \~English
 * @brief cfgContent is data received from printer or data previously saved to file and reloaded.
 *
 * @param cfgContent The content of the configuration.
 * @return An initialized instance of Config, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief cfgContent 是从打印机接收的数据或之前保存到文件并重新加载的数据。
 *
 * @param cfgContent 配置的内容。
 * @return 一个初始化的 Config 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief cfgContent 是從打印機接收的數據或之前保存到文件並重新加載的數據。
 *
 * @param cfgContent 配置的內容。
 * @return 一個初始化的 Config 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithCfgContent:(NSString *)cfgContent;

/**
 * \~English
 * @brief Initializes a new instance with default values.
 *
 * @return An initialized instance of Config, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用默认值初始化一个新实例。
 *
 * @return 一个初始化的 Config 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用默認值初始化一個新實例。
 *
 * @return 一個初始化的 Config 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)init;

// Methods

/**
 * \~English
 * @brief The configuration content sent/received from printer.
 *
 * @return A string representation of the configuration.
 *
 * \~Chinese
 * @brief 从打印机发送/接收的配置内容。
 *
 * @return 配置的字符串表示形式。
 *
 * \~Chinese-Traditional
 * @brief 從打印機發送/接收的配置內容。
 *
 * @return 配置的字符串表示形式。
 */
- (NSString *)description;

/**
 * \~English
 * @brief Gets the number associated with the configuration.
 *
 * Configuration number. 0-8 valid. 0 is factory configuration. 1-8 are user stored configurations.
 *
 * This number refers to the configuration number stored in the printer memory.
 *
 * @return The number associated with the configuration.
 *
 * \~Chinese
 * @brief 获取与配置关联的编号。
 *
 * 配置编号。有效范围为 0-8。0 为出厂配置，1-8 为用户存储的配置。
 *
 * 此编号指的是存储在打印机内存中的配置编号。
 *
 * @return 与配置关联的编号。
 *
 * \~Chinese-Traditional
 * @brief 獲取與配置關聯的編號。
 *
 * 配置編號。有效範圍為 0-8。0 為出廠配置，1-8 為用戶存儲的配置。
 *
 * 此編號指的是存儲在打印機內存中的配置編號。
 *
 * @return 與配置關聯的編號。
 */
- (NSInteger)getNumber;

/**
 * \~English
 * @brief Sets the number associated with the configuration.
 *
 * Configuration number. 0-8 valid. 0 is factory configuration. 1-8 are user stored configurations.
 *
 * This number refers to the configuration number stored in the printer memory.
 *
 * @param iNumber The number to set.
 *
 * \~Chinese
 * @brief 设置与配置关联的编号。
 *
 * 配置编号。有效范围为 0-8。0 为出厂配置，1-8 为用户存储的配置。
 *
 * 此编号指的是存储在打印机内存中的配置编号。
 *
 * @param iNumber 要设置的编号。
 *
 * \~Chinese-Traditional
 * @brief 設置與配置關聯的編號。
 *
 * 配置編號。有效範圍為 0-8。0 為出廠配置，1-8 為用戶存儲的配置。
 *
 * 此編號指的是存儲在打印機內存中的配置編號。
 *
 * @param iNumber 要設置的編號。
 */
- (void)setNumber:(int)iNumber;

/**
 * \~English
 * @brief Gets the model associated with the configuration.
 *
 * The Model of the printer configuration is valid for.
 *
 * The configuration can only be downloaded to the printers that match the model.
 *
 * @return The model associated with the configuration.
 *
 * \~Chinese
 * @brief 获取与配置关联的型号。
 *
 * 打印机配置有效的型号。
 *
 * 配置只能下载到与型号匹配的打印机。
 *
 * @return 与配置关联的型号。
 *
 * \~Chinese-Traditional
 * @brief 獲取與配置關聯的型號。
 *
 * 打印機配置有效的型號。
 *
 * 配置只能下載到與型號匹配的打印機。
 *
 * @return 與配置關聯的型號。
 */
- (NSString *)getModel;

/**
 * \~English
 * @brief Sets the model associated with the configuration.
 *
 * The Model of the printer configuration is valid for.
 *
 * The configuration can only be downloaded to the printers that match the model.
 *
 * @param strModel The model to set.
 *
 * \~Chinese
 * @brief 设置与配置关联的型号。
 *
 * 打印机配置有效的型号。
 *
 * 配置只能下载到与型号匹配的打印机。
 *
 * @param strModel 要设置的型号。
 *
 * \~Chinese-Traditional
 * @brief 設置與配置關聯的型號。
 *
 * 打印機配置有效的型號。
 *
 * 配置只能下載到與型號匹配的打印機。
 *
 * @param strModel 要設置的型號。
 */
- (void)setModel:(NSString *)strModel;

/**
 * \~English
 * @brief Gets the name associated with the configuration.
 *
 * The configuration name. This property can be left empty if naming is not desired.
 *
 * The property can be used to give the configuration a more descriptive name. e.g "ShipLabel"
 *
 * Note that name should be kept short to prevent truncation by printer.
 *
 * @return The name associated with the configuration.
 *
 * \~Chinese
 * @brief 获取与配置关联的名称。
 *
 * 配置名称。如果不需要命名，此属性可以留空。
 *
 * 此属性可用于为配置提供更具描述性的名称。例如 "ShipLabel"
 *
 * 请注意，名称应保持简短以防止被打印机截断。
 *
 * @return 与配置关联的名称。
 *
 * \~Chinese-Traditional
 * @brief 獲取與配置關聯的名稱。
 *
 * 配置名稱。如果不需要命名，此屬性可以留空。
 *
 * 此屬性可用於為配置提供更具描述性的名稱。例如 "ShipLabel"
 *
 * 請注意，名稱應保持簡短以防止被打印機截斷。
 *
 * @return 與配置關聯的名稱。
 */
- (NSString *)getName;

/**
 * \~English
 * @brief Sets the name associated with the configuration.
 *
 * The configuration name. This property can be left empty if naming is not desired.
 *
 * The property can be used to give the configuration a more descriptive name. e.g "ShipLabel"
 *
 * Note that name should be kept short to prevent truncation by printer.
 *
 * @param strName The name to set.
 *
 * \~Chinese
 * @brief 设置与配置关联的名称。
 *
 * 配置名称。如果不需要命名，此属性可以留空。
 *
 * 此属性可用于为配置提供更具描述性的名称。例如 "ShipLabel"
 *
 * 请注意，名称应保持简短以防止被打印机截断。
 *
 * @param strName 要设置的名称。
 *
 * \~Chinese-Traditional
 * @brief 設置與配置關聯的名稱。
 *
 * 配置名稱。如果不需要命名，此屬性可以留空。
 *
 * 此屬性可用於為配置提供更具描述性的名稱。例如 "ShipLabel"
 *
 * 請注意，名稱應保持簡短以防止被打印機截斷。
 *
 * @param strName 要設置的名稱。
 */
- (void)setName:(NSString *)strName;

/**
 * \~English
 * @brief Gets the program file associated with the configuration.
 *
 * The printer program file that configuration can be downloaded to.
 *
 * If left empty, printer will not prevent downloading configuration to different Program File.
 *
 * Leave this with value read from printer if you want to make sure configuration is only applied to printers with matching Program File.
 *
 * @return The program file associated with the configuration.
 *
 * \~Chinese
 * @brief 获取与配置关联的程序文件。
 *
 * 可以下载配置的打印机程序文件。
 *
 * 如果留空，打印机将不会阻止将配置下载到不同的程序文件。
 *
 * 如果要确保配置仅应用于具有匹配程序文件的打印机，请保留从打印机读取的值。
 *
 * @return 与配置关联的程序文件。
 *
 * \~Chinese-Traditional
 * @brief 獲取與配置關聯的程序文件。
 *
 * 可以下載配置的打印機程序文件。
 *
 * 如果留空，打印機將不會阻止將配置下載到不同的程序文件。
 *
 * 如果要確保配置僅應用於具有匹配程序文件的打印機，請保留從打印機讀取的值。
 *
 * @return 與配置關聯的程序文件。
 */
- (NSString *)getProgramFile;

/**
 * \~English
 * @brief Sets the program file associated with the configuration.
 *
 * The printer program file that configuration can be downloaded to.
 *
 * If left empty, printer will not prevent downloading configuration to different Program File.
 *
 * Leave this with value read from printer if you want to make sure configuration is only applied to printers with matching Program File.
 *
 * @param strProgramFile The program file to set.
 *
 * \~Chinese
 * @brief 设置与配置关联的程序文件。
 *
 * 可以下载配置的打印机程序文件。
 *
 * 如果留空，打印机将不会阻止将配置下载到不同的程序文件。
 *
 * 如果要确保配置仅应用于具有匹配程序文件的打印机，请保留从打印机读取的值。
 *
 * @param strProgramFile 要设置的程序文件。
 *
 * \~Chinese-Traditional
 * @brief 設置與配置關聯的程序文件。
 *
 * 可以下載配置的打印機程序文件。
 *
 * 如果留空，打印機將不會阻止將配置下載到不同的程序文件。
 *
 * 如果要確保配置僅應用於具有匹配程序文件的打印機，請保留從打印機讀取的值。
 *
 * @param strProgramFile 要設置的程序文件。
 */
- (void)setProgramFile:(NSString *)strProgramFile;

/**
 * \~English
 * @brief Gets the data associated with the configuration.
 *
 * The printer configuration values. These should not be modified.
 *
 * These values should not be modified and should only be read, stored for backup purposes, and redownloaded to printer.
 *
 * @return The data associated with the configuration.
 *
 * \~Chinese
 * @brief 获取与配置关联的数据。
 *
 * 打印机配置值。请勿修改这些值。
 *
 * 这些值不应被修改，且应仅用于读取、备份存储以及重新下载到打印机。
 *
 * @return 与配置关联的数据。
 *
 * \~Chinese-Traditional
 * @brief 獲取與配置關聯的數據。
 *
 * 打印機配置值。請勿修改這些值。
 *
 * 這些值不應被修改，且應僅用於讀取、備份存儲以及重新下載到打印機。
 *
 * @return 與配置關聯的數據。
 */
- (NSString *)getData;

/**
 * \~English
 * @brief Sets the data associated with the configuration.
 *
 * The printer configuration values. These should not be modified.
 *
 * These values should not be modified and should only be read, stored for backup purposes, and redownloaded to printer.
 *
 * @param strData The data to set.
 *
 * \~Chinese
 * @brief 设置与配置关联的数据。
 *
 * 打印机配置值。请勿修改这些值。
 *
 * 这些值不应被修改，且应仅用于读取、备份存储以及重新下载到打印机。
 *
 * @param strData 要设置的数据。
 *
 * \~Chinese-Traditional
 * @brief 設置與配置關聯的數據。
 *
 * 打印機配置值。請勿修改這些值。
 *
 * 這些值不應被修改，且應僅用於讀取、備份存儲以及重新下載到打印機。
 *
 * @param strData 要設置的數據。
 */
- (void)setData:(NSString *)strData;

@end
